<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\AdminModule\Model\FeatureValues;

class FeatureValueForm extends BaseControl
{
	/** @var featureValue */
	public $featureValue;

	/** @var featureValueTexts */
	public $featureValueTexts;

	/** @var FeatureValues */
	protected $featureValueServices;

	public function __construct(FeatureValues $featureValues)
	{
		$this->featureValueServices = $featureValues;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$features = [null => ''];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join(FeatureTexts::class, 'ft', 'WITH', 'f.id = ft.id')
			         ->where('ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getResult() as $g) {
			$features[$g['id']] = $g['name'];
		}

		if (count($features) == 1) {
			return $form;
		}

		$form->addSelect('feature', 'eshopCatalog.featureValueForm.feature', $features)->setRequired();
		$form->addText('value', 'eshopCatalog.featureValueForm.value')->setIsMultilanguage()->setMaxLength(255);
		$form->addSubmit('saveAndNew', NULL ?: 'eshopCatalog.defaultForm.saveAndNew')->getControlPrototype()->addHtml('<i class="fa fa-save"></i>');
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->featureValue) {
				$featureValue = $this->featureValue;
			} else {
				$featureValue = new FeatureValue();
				$featureValue->setFeature($this->em->getReference(Feature::class, $values->feature));
			}

			$texts      = $featureValue->getFeatureValueTexts()->toArray();
			$langValues = $form->convertMultilangValuesToArray();

			foreach ($langValues as $lang => $vals) {
				if ($vals['value'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text = $texts[$lang] ?? new FeatureValueTexts($featureValue, $lang);
				$text->setName($vals['value']);
				$this->em->persist($text);

				$texts[$lang] = $text;
			}

			$featureValue->setTexts($texts);

			$this->em->persist($featureValue);
			$this->em->flush();
			$form->addCustomData('featureValueId', $featureValue->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.featureValueForm.featureValueSaved', Flashes::FLASH_SUCCESS);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setFeatureValue($id): void
	{
		$this->featureValue = $this->em->getRepository(FeatureValue::class)->find($id);

		if (!$this->featureValue)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form = $this['form'];
		$form->setDefaults([
			'feature' => $this->featureValue->feature->getId(),
		]);

		$mTexts = $form->prepareMultilangTextsForForm(['value' => 'name'], $this->featureValue->getFeatureValueTexts()->toArray());
		$form->setDefaults($mTexts['texts']);
	}
}

