<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\FilterService;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Paginator;

/**
 * Class ProductsList
 * @package EshopCatalog\FrontModule\Components
 */
class ProductsList extends BaseControl
{
	/** @var Config */
	protected $config;

	/** @var Products */
	protected $productsService;

	/** @var int */
	protected $categoryId;

	/** @var Product[] */
	protected $products;

	/** @var int */
	protected $allProductsCount;

	/** @var string */
	protected $searchQuery;

	/** @var int */
	public $page = 1;

	/** @var int */
	public $limit = 12;

	/** @var int */
	public $start = 0;

	/** @var bool */
	public $paginatorEnabled = true;

	/** @var BaseControl|ProductsFilter */
	public $filterComponent;

	/** @var Tags */
	protected $tagsService;

	/** @var Paginator */
	protected $paginator;

	/** @var IProductPreviewFactory */
	protected $productPreviewFactory;

	public function __construct(Config $config, Products $products, Tags $tags, IProductPreviewFactory $productPreviewFactory)
	{
		$this->config                = $config;
		$this->productsService       = $products;
		$this->tagsService           = $tags;
		$this->productPreviewFactory = $productPreviewFactory;
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		$this->page = $this->getParameter('page', 1);
		$this->handlePaginator($this->page);

		if ($this->getPresenter()->isAjax())
			$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function() {
				$this->redrawControl('list');
			});
	}

	public function render()
	{
		$this->template->paginator = $this->getPaginator();
		$this->template->limit     = $this->limit;
		$this->template->start     = $this->start;
		$this->template->products  = $this->getProducts();
		$this->template->count     = $this->getCount();

		$this->template->render($this->getTemplateFile(null, 'ProductsList'));
	}

	public function handlePaginator($page)
	{
		$this->page  = $page;
		$this->start = $page > 1 ? $this->limit * ($page - 1) : 0;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentProductPreview(): Multiplier
	{
		return new Multiplier(function(int $id): ProductPreview {
			$control = $this->productPreviewFactory->create();

			$control->setProductById($id);

			return $control;
		});
	}


	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	/**
	 * @return Paginator
	 */
	public function getPaginator(): Paginator
	{
		$paginator = new Paginator();
		$paginator->setItemCount($this->getCount());
		$paginator->setItemsPerPage($this->config->get('productsListPerPage'));
		$paginator->setPage($this->page);

		return $paginator;
	}

	/**
	 * @return Product[]|null
	 */
	public function getProducts()
	{
		if (!$this->products) {
			if ($this->categoryId) {
				$filter                               = $this->filterComponent->getFilterValues();
				$filter['onlyStock']                  = true;
				FilterService::$cacheTags['features'] = 'listing/' . $this->categoryId;
				FilterService::$cacheTags['variants'] = 'listing/' . $this->categoryId;
				FilterService::$cacheTags['minMax']   = 'listing/' . $this->categoryId;
				$this->products                       = $this->productsService->getProductsInCategory($this->categoryId, $this->start, $this->limit, $this->filterComponent->getFilterValues()['sort'] ?? null, $filter);

				if (empty($this->products)) {
					if (empty($filter))
						$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundCategory');
					else
						$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundFilter');
				}
			} else if ($this->searchQuery) {
				FilterService::$cacheTags['features'] = 'listing/search';
				FilterService::$cacheTags['variants'] = 'listing/search';
				FilterService::$cacheTags['minMax']   = 'listing/search';
				$this->products                       = $this->productsService->getProductsBySearch($this->searchQuery, $this->start, $this->limit, $this->filterComponent->getFilterValues()['sort'] ?? null, $this->filterComponent->getFilterValues());

				if (empty($this->products))
					$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundSearch');
			}

			foreach ($this->products as &$p) {
				$this->tagsService->loadTagsToProduct($p);
			}

			if ($this->products)
				$this->productsService->loadQuantity($this->products);
		}

		return $this->products;
	}

	protected function getCount()
	{
		if (!$this->allProductsCount) {
			if ($this->categoryId) {
				$filter                 = $this->filterComponent->getFilterValues();
				$filter['onlyStock']    = true;
				$this->allProductsCount = count($this->productsService->getProductsIdInCategory($this->categoryId, null, null, null, $filter));
			} else if ($this->searchQuery)
				$this->allProductsCount = count($this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $this->filterComponent->getFilterValues()));
		}

		return $this->allProductsCount;
	}

	/**
	 * @param $category
	 *
	 * @return $this
	 */
	public function setCategory($category)
	{
		$this->categoryId = $category;

		return $this;
	}

	/**
	 * @param int $limit
	 *
	 * @return $this
	 */
	public function setLimit(int $limit)
	{
		$this->limit = $limit;

		return $this;
	}

	/**
	 * @param BaseControl $component
	 *
	 * @return $this
	 */
	public function setFilterComponent($component)
	{
		$this->filterComponent = $component;
		$filter                = $this->filterComponent->getFilterValues();
		if ($this->categoryId) {
			FilterService::$cacheTags['features'] = 'listing/' . $this->categoryId;
			FilterService::$cacheTags['variants'] = 'listing/' . $this->categoryId;
			FilterService::$cacheTags['minMax']   = 'listing/' . $this->categoryId;
			FilterService::$cacheTags['manu']     = 'listing/' . $this->categoryId;
			$this->filterComponent->productIds    = $this->productsService->getProductsIdInCategory($this->categoryId);
			$this->filterComponent->categoryId    = $this->categoryId;
			foreach ($this->filterComponent->getFeatures() as &$feature) {
				foreach ($feature->items as $item) {
					$tmp = $filter;
					unset($tmp['sort']);
					$tmp['features'][$feature->id] = [$item->id];
					$tmp2                          = ['features' => [$feature->id => [$item->id]],];
					$this->filterComponent->setFeatureProductsCount($feature->id, $item->id, ($this->productsService->getProductsInCategoryCount($this->categoryId, $tmp)));
					$this->filterComponent->setFeatureProductsCount($feature->id, $item->id, ($this->productsService->getProductsInCategoryCount($this->categoryId, $tmp2)), 'init');
				}
			}
			foreach ($this->filterComponent->getManufacturers()->items as $item) {
				$tmp = $filter;
				unset($tmp['sort']);
				$tmp['manu'] = $item->id;
				$tmp2        = ['manu' => $item->id];
				$this->filterComponent->setManufacturerProductsCount((int) $item->id, (int) ($this->productsService->getProductsInCategoryCount($this->categoryId, $tmp)));
				$this->filterComponent->setManufacturerProductsCount((int) $item->id, (int) ($this->productsService->getProductsInCategoryCount($this->categoryId, $tmp2)), 'init');
			}
			unset($filter['priceRange']);
			$this->filterComponent->productIdsNoPriceFilter = $this->productsService->getProductsIdInCategory($this->categoryId, null, null, null, $filter);
		} else if ($this->searchQuery) {
			FilterService::$cacheTags['features'] = 'listing/search';
			FilterService::$cacheTags['variants'] = 'listing/search';
			FilterService::$cacheTags['minMax']   = 'listing/search';
			FilterService::$cacheTags['manu']     = 'listing/search';
			$this->filterComponent->productIds    = $this->productsService->getProductsIdBySearch($this->searchQuery);
			foreach ($this->filterComponent->getFeatures() as &$feature) {
				foreach ($feature->items as $item) {
					$tmp = $filter;
					unset($tmp['sort']);
					$tmp['features'][$feature->id] = [$item->id];
					$tmp2                          = ['features' => [$feature->id => [$item->id]]];
					$this->filterComponent->setFeatureProductsCount($feature->id, $item->id, count($this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $tmp)));
					$this->filterComponent->setFeatureProductsCount($feature->id, $item->id, count($this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $tmp2)), 'init');
				}
			}
			foreach ($this->filterComponent->getManufacturers()->items as $item) {
				$tmp = $filter;
				unset($tmp['sort']);
				$tmp['manu'] = $item->id;
				$tmp2        = ['manu' => $item->id];
				$this->filterComponent->setManufacturerProductsCount((int) $item->id, (int) count($this->productsService->getProductsIdInCategory($this->categoryId, null, null, null, $tmp)));
				$this->filterComponent->setManufacturerProductsCount((int) $item->id, (int) count($this->productsService->getProductsIdInCategory($this->categoryId, null, null, null, $tmp2)), 'init');
			}
			unset($filter['priceRange']);
			$this->filterComponent->productIdsNoPriceFilter = $this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $filter);
		}

		return $this;
	}

	/**
	 * @param int[] $ids
	 *
	 * @return $this
	 */
	public function setProductsBySearch($q)
	{
		$this->searchQuery = $q;

		return $this;
	}
}
