<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\BaseService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use Kdyby\Doctrine\EntityManager;
use Nette\Caching\Cache;
use Nette\SmartObject;

/**
 * Class Features
 * @package EshopCatalog\FrontModule\Model
 */
class Features extends BaseFrontEntityService
{
	protected $entityClass = Feature::class;

	const CACHE_NAMESPACE = 'eshopCatalogVariants';

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['features'],
		Cache::EXPIRE  => '1 month',
		Cache::SLIDING => true,
	];

	/** @var array */
	protected $cFeatures, $cFeatureValues;

	/**
	 * Features constructor.
	 *
	 * @param CacheService $cacheService
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	public function getPublishedFeatures(): array
	{
		$features = [];
		foreach ($this->getEr()->createQueryBuilder('f')
			         ->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft')
			         ->where('f.isPublished = 1')->andWhere('ft.lang = :lang')
			         ->setParameter('lang', $this->defaultLang->locale)
			         ->orderBy('f.position')
			         ->getQuery()->getScalarResult() as $row)
			$features[$row['id']] = $row['name'];

		return $features;
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	public function getFeatureById($id)
	{
		if (!$this->cFeatures[$id]) {
			$locale = $this->defaultLang->locale;
			$key    = 'features_' . $locale;

			$this->cFeatures = $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
				$dep = $this->cacheDep;

				$data = [];
				foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
					         ->select('f.id as id, ft.name as name')
					         ->andWhere('f.isPublished = 1')
					         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
					         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row)
					$data[$row['id']] = $row;

				return $data;
			});
		}

		return $this->cFeatures[$id];
	}

	/**
	 * @param int $id
	 *
	 * @return mixed
	 */
	public function getValuesByFeatureId(int $id)
	{
		if (!isset($this->cFeatures[$id]['values'])) {
			$key = 'featureIdValues';

			$data = $this->cacheService->defaultCache->load($key, function(&$dep) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'featureValues';

				$data = [];

				foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
					         ->select('fv.id as id, IDENTITY(fv.feature) as featureId')
					         ->andWhere('fv.isPublished = 1')->orderBy('fv.position')
					         ->getQuery()->getArrayResult() as $row)
					$data[$row['featureId']][] = $row['id'];

				return $data;
			});

			foreach ($data as $k => $v) {
				foreach ($v as $vId)
					$this->cFeatures[$k]['values'][$vId] = $this->getFeatureValueById($vId);
			}
		}

		return $this->cFeatures[$id]['values'];
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	public function getFeatureValueById($id)
	{
		if (!$this->cFeatureValues[$id]) {
			$locale = $this->defaultLang->locale;
			$key    = 'featureValues_' . $locale;

			$this->cFeatureValues = $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'featureValues';

				$data = [];
				foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
					         ->select('fv.id as id, fvt.name as name')
					         ->andWhere('fv.isPublished = 1')
					         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang ')
					         ->orderBy('fv.position')
					         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row)
					$data[$row['id']] = $row;

				return $data;
			});
		}

		return $this->cFeatureValues[$id];
	}
}
