<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use Nette\Caching\Cache;

class ProductVariants extends BaseFrontEntityService
{
	protected $entityClass = ProductVariant::class;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var Variants */
	protected $variantsService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['productVariants'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var array */
	protected $cVariants, $cVariantValues;

	/**
	 * ProductVariants constructor.
	 *
	 * @param CacheService $cacheService
	 * @param Variants     $variants
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService, Variants $variants, DefaultLang $defaultLang)
	{
		$this->cacheService    = $cacheService;
		$this->variantsService = $variants;
		$this->defaultLang     = $defaultLang;
	}

	/**
	 * @param $id
	 *
	 * @return Dao\ProductVariant|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		//		return $this->cacheService->productCache->load('productVariant_' . $id, function(&$dep) use ($id) {
		//			$dep = $this->cacheDep;

		$result = $this->getEr()->createQueryBuilder('pv')->select('pv')
			->andWhere('pv.id = :id')->setParameter('id', $id)->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

		if ($result) {
			$dep[Cache::TAGS][] = 'product/' . $result['product'];

			return $this->fillDao($result);
		}

		return null;
		//		});
	}

	public function getProductVariantsForProduct($idProduct)
	{
		$pvs = $this->cacheService->productCache->load('variantsForProduct_' . $idProduct, function(&$dep) use ($idProduct) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'product/' . $idProduct;

			$data = [];
			foreach ($this->getEr()->createQueryBuilder('pv')->addSelect('IDENTITY(pv.product) as product')
				         ->andWhere('pv.product = :id')->setParameter('id', $idProduct)
				         ->getQuery()->getArrayResult() as $row) {
				$data[$row[0]['id']] = $this->fillDao($row);
			}

			foreach ($this->em->getRepository(ProductVariantCombination::class)->createQueryBuilder('pvc')
				         ->select('IDENTITY(pvc.variant) as variant, IDENTITY(pvc.value) as value, IDENTITY(pvc.productVariant) as productVariant')
				         ->where('pvc.productVariant IN (:pv)')->setParameter('pv', array_keys($data))
				         ->leftJoin('pvc.variant', 'v')->orderBy('v.position')
				         ->getQuery()->getArrayResult() as $row) {
				$data[$row['productVariant']]->variants[] = [$row['variant'], $row['value']];
			}

			return $data;
		});

		$pvs = $data;
		foreach ($pvs as $pv) {
			$tmp = [];
			foreach ($pv->variants as $v) {
				$variant = $this->variantsService->getVariantById($v[0]);
				$value   = $this->variantsService->getVariantValueById($v[1]);
				$tmp[]   = [
					'variantId'        => $variant['id'],
					'variantName'      => $variant['name'],
					'variantValueId'   => $value['id'],
					'variantValueName' => $value['name'],
				];
			}
			$pv->variants = $tmp;
		}

		return $pvs;
	}

	/**
	 * @param array $productVariant
	 *
	 * @return Dao\ProductVariant
	 */
	protected function fillDao($data): Dao\ProductVariant
	{
		$locale = $this->defaultLang->locale;

		if (isset($data[0])) {
			$data = array_merge_recursive($data, $data[0]);
			unset($data[0]);
		}

		$pv = new Dao\ProductVariant($data['id']);
		$pv->setEan($data['ean']);
		$pv->setIsDefault($data['isDefault']);
		$pv->setQuantity($data['quantity']);
		$pv->setUnitPriceImpact($data['unitPriceImpact']);
		$pv->setProduct($data['product']);

		return $pv;
	}
}
