<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Navigation\Home;
use Kdyby\Events\Subscriber;
use Nette\Caching\Cache;
use Nette\SmartObject;

class ProductListener implements Subscriber
{
	use SmartObject;

	/** @var CacheService */
	protected $cacheService;

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	public function getSubscribedEvents()
	{
		return [];
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $args
	 */
	public function onPreFlush($product, $args)
	{
		$this->cacheService->clean('product', [
			Cache::TAGS => [
				'product/' . $product->getId(),
			],
		]);

		$this->cacheService->clean('navigation', [
			Cache::TAGS => [
				Home::CACHE_PRODUCT . '/' . $product->getId(),
			],
		]);

		foreach ($args->getEntityManager()->getUnitOfWork()->getScheduledEntityInsertions() as $entity) {
			$tags = [];

			if ($entity instanceof Product) {
				if ($entity->getDefaultCategory())
					$tags[] = 'listing/' . $entity->getDefaultCategory()->getId();

				foreach ($entity->getCategoryProducts()->toArray() as $row)
					$tags[] = 'listing/' . $row->category->getId();
				$tags[] = 'listing/search';

				if ($tags)
					$this->cacheService->clean('default', [
						Cache::TAGS => $tags,
					]);
			}
		}
	}

	/**
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postHandler(Product $product, LifecycleEventArgs $event)
	{
		$em        = $event->getEntityManager();
		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($product);

		$tags = ['listing/search'];

		if ($product->getDefaultCategory())
			$tags[] = 'listing/' . $product->getDefaultCategory()->getId();

		if ($product->getCategoryProducts())
			foreach ($product->getCategoryProducts()->toArray() as $row)
				$tags[] = 'listing/' . $row->category->getId();

		$this->cacheService->clean('default', [
			Cache::TAGS => $tags,
		]);
	}
}
