<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\Model\UI\BaseControl;
use Core\Components\Flashes\Flashes;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Entities\Product;
use Nette\Utils\Html;

/**
 * TODO cache při odstranění
 *
 * Class CategoriesGrid
 * @package EshopCatalog\AdminModule\Components\Categories
 */
class CategoriesGrid extends BaseControl
{
	/** @var Categories */
	protected $categoriesService;

	/** @var ICategoryFormFactory */
	protected $categoryFormFactory;

	/**
	 * CategoriesGrid constructor.
	 *
	 * @param Categories           $categories
	 * @param ICategoryFormFactory $categoryFormFactory
	 */
	public function __construct(Categories $categories, ICategoryFormFactory $categoryFormFactory)
	{
		$this->categoriesService   = $categories;
		$this->categoryFormFactory = $categoryFormFactory;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->categoriesService->removeCategory($id)) {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
			$presenter->payload->treeViewItemDeleted = true;
		} else
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleEdit(int $id)
	{
		$category = $this->categoriesService->get($id);

		if ($category) {
			$this['categoryForm']->categoryId = $id;
			$this->template->modalTitle       = $this->t('eshopCatalog.title.editCategory') . ' ' . $category->getCategoryText()->getName();
			$this->template->modal            = 'categoryForm';
			$this->redrawControl('modal');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->getGridData());
		$grid->setTreeView([$this, 'getGridData'], 'childs');
		$grid->setSortable();
		$grid->setSortableHandler('categoriesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('name', 'eshopCatalog.defaultGrid.name', 'Categories:edit')->setRenderer(function($item) {
			$name = " {$item['name']}";
			for ($i = 1; $i < $item['lvl']; $i++)
				$name = '---' . $name;

			return Html::el('a', [
				'class' => 'ajax',
				'href'  => $this->link('edit!', $item['id']),
			])->setText($name);
		});
		$grid->addColumnText('productsCount', 'eshopCatalog.defaultGrid.productsCount')->setAlign('center');
		$grid->addColumnStatus('isPublished', 'eshopCatalog.defaultGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('edit', '', 'edit')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	public function getGridData(int $id = null)
	{
		$qb = $this->categoriesService->getEr()->createQueryBuilder('c', 'c.id')
			->select('c.id, c.isPublished, ct.name')
			->leftJoin('c.categoryTexts', 'ct')
			->where('ct.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			->groupBy('c.id')
			->addOrderBy('c.root')->addOrderBy('c.lft');

		if ($id) {
			$qb->andWhere('c.parent = :cId')->setParameter('cId', $id);
		} else {
			$qb->andWhere('c.lvl = 1');
		}

		$data = $qb->getQuery()->getArrayResult();

		foreach ($data as $k => $row) {
			$data[$k]['productsCount'] = 0;
		}

		$ids = array_keys($data);
		foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('IDENTITY(c.parent) as parent, count(c.id) as count')
			         ->where('c.parent IN (:parents)')->setParameter('parents', $ids)->groupBy('c.parent')
			         ->getQuery()->getArrayResult() as $row) {
			$data[$row['parent']]['childs'] = 1;
		}

		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')->select('count(p.id) as count, IDENTITY(p.idCategoryDefault) as cat')
			         ->where('p.idCategoryDefault IN (:cats)')->setParameter('cats', $ids)
			         ->groupBy('p.idCategoryDefault')
			         ->getQuery()->getArrayResult() as $row) {
			$data[$row['cat']]['productsCount'] += $row['count'];
		}

		foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('c.lft, c.gt, c.id')
			         ->where('c.id IN (:ids)')->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $cat) {
			$t                                 = $this->categoriesService->getEr()->createQueryBuilder('c')
				->select('COUNT(cp.product) as count')
				->leftJoin('c.categoryProducts', 'cp')
				->where('c.lft >= :lft')->andWhere('c.gt <= :gt')
				->setParameters([
					'lft' => $cat['lft'],
					'gt'  => $cat['gt'],
				])->getQuery()->getSingleScalarResult();
			$data[$cat['id']]['productsCount'] += $t;
		}

		return $data;
	}

	/**
	 * @return CategoryForm
	 */
	protected function createComponentCategoryForm()
	{
		$control = $this->categoryFormFactory->create();
		if ($this->getParameter('id'))
			$control->setCategory($this->getParameter('id'));

		$redrawItem = function(int $id) {
			$cat      = $this->categoriesService->get($id);
			$parentId = null;
			if ($cat->getParent() && $cat->getParent()->getLvl() > 0)
				$parentId = $cat->getParent()->getId();
			$this['grid']->setDataSource($this->getGridData($parentId));
			$this['grid']->redrawItem($id);
		};

		$control['form']->onSuccessSave[]         = function(BaseForm $form) use ($redrawItem) {
			$redrawItem($form->getCustomData('categoryId'));
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) use ($redrawItem) {
			$redrawItem($form->getCustomData('categoryId'));
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->categoriesService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$cat = $this->categoriesService->get($id);
			$this['grid']->setDataSource($this->getGridData($cat->getParent()->getId()));
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();

		$id   = $request->getPost('id');
		$move = $request->getPost('move');

		if ($id && $move) {
			$er  = $this->categoriesService->getEr();
			$cat = $this->categoriesService->get($id);

			if ($cat) {
				if ($move < 0)
					$er->moveDown($cat, abs($move));
				else if ($move > 0)
					$er->moveUp($cat, $move);
				$presenter->flashMessageSuccess('default.positionChanged');
			}
		} else
			$presenter->flashMessageDanger('default.positionChangeFailed');

		$this->em->flush();
		if ($id)
			$this->categoriesService->cleanCacheDeep($id);

		$presenter->redrawControl('flashes');
	}
}
