<?php declare(strict_types=1);

namespace EshopCatalog\AdminModule\Presenters;

use EshopCatalog\AdminModule\Components\Categories\ICategoryFormFactory;
use EshopCatalog\AdminModule\Components\Categories\ICategoriesGridFactory;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Entities\Category;
use Core\Components\Flashes\Flashes;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\Html;

/**
 * Class CategoriesPresenter
 * @package EshopCatalog\AdminModule\Presenters
 */
class CategoriesPresenter extends BasePresenter
{
	/** @var Categories @inject */
	public $categoriesService;

	protected function startup()
	{
		parent::startup();
		$this->setHeader('eshopCatalog.title.categories', 'fa fa-list');
	}

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */
	public function actionDefault()
	{
		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addCategory',
				'link'  => $this->link('add!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
		]]);
	}

	public function actionEdit($id)
	{
		$category = $this->em->getRepository(Category::class)->find($id);

		if (!$category)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader($this->t('eshopCatalog.title.editCategory') . ': ' . $category->getCategoryText()->name, 'fa fa-list-alt');
	}

	public function actionClose($id)
	{
		$this->redirect('Default:');
	}

	/*******************************************************************************************************************
	 * ==================================   Handle
	 */

	public function handleAdd()
	{
		$this->template->modalTitle       = $this->t('eshopCatalog.title.newCategory');
		$this->template->modal            = 'categoryForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	protected function createComponentCategoriesGrid(ICategoriesGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentCategoryForm(ICategoryFormFactory $category)
	{
		$control = $category->create();
		if ($this->getParameter('id'))
			$control->setCategory($this->getParameter('id'));

		$redrawItem = function(int $id) {
			$cat      = $this->categoriesService->get($id);
			$parentId = null;
			if ($cat->getParent() && $cat->getParent()->getLvl() > 0)
				$parentId = $cat->getParent()->getId();
			$this['categoriesGrid']['grid']->setDataSource($this['categoriesGrid']->getGridData($parentId));
			$this->getPresenter()->payload->datagridTreeItemAppend = [
				'grid'   => $this['categoriesGrid']->getName() . '-' . $this['categoriesGrid']['grid']->getName(),
				'dataId' => $parentId,
				'id'     => $cat->getId(),
			];
			$this['categoriesGrid']['grid']->redrawItem($id);
		};

		$control['form']->onAnchor[] = function() use ($control, $redrawItem) {
			$control['form']->onSuccessSave[]         = function(BaseForm $form) use ($redrawItem) {
				if ($form->getCustomData('categoryId'))
					$redrawItem($form->getCustomData('categoryId'));
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
			};
			$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) use ($redrawItem) {
				if ($form->getCustomData('categoryId'))
					$redrawItem($form->getCustomData('categoryId'));
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
			};
			$control['form']['saveControl']->closeModalOnCancel();
		};

		return $control;
	}
}
