<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Presenters;

use EshopCatalog\AdminModule\Components\Products\IProductsFeatureFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsVariantFormFactory;

;

use EshopCatalog\AdminModule\Components\Products\IProductsCategoryFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsGridFactory;
use EshopCatalog\AdminModule\Components\Tags\ITagFormFactory;
use EshopCatalog\AdminModule\Components\Tags\ITagsGridFactory;
use EshopCatalog\AdminModule\Components\VatRates\IVatRateFormFactory;
use EshopCatalog\AdminModule\Components\VatRates\IVatRatesGridFactory;
use EshopCatalog\Model\Entities\Product;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\VatRate;
use Nette\Http\IResponse;

/**
 * Class ProductsPresenter
 * @package EshopCatalog\AdminModule\Presenters
 */
class ProductsPresenter extends BasePresenter
{

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */

	public function actionDefault()
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.products'));
		$this->setHeader('eshopCatalog.title.products', 'fa fa-list');

		$this->getSession('productForm')->remove();

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addProduct',
				'link'  => 'Products:add',
				'ico'   => 'plus',
			],
			[
				'title' => 'eshopCatalog.menu.addVatRate',
				'link'  => 'Products:addVatRate',
				'ico'   => 'plus',
			],
			[
				'title' => 'eshopCatalog.menu.addTag',
				'link'  => 'Products:addTag',
				'ico'   => 'plus',
			],
			[
				'title' => 'eshopCatalog.menu.newExport',
				'link'  => 'Products:export',
				'ico'   => 'edit',
			],
		]]);
	}

	public function actionExport()
	{
		$this->redirect(':EshopCatalog:Cron:Export:porovnavace', ['type' => ['zbozi', 'google', 'heureka']]);
	}

	public function actionAdd()
	{
		$this->setHeader('eshopCatalog.title.newProduct', 'fa fa-list-alt');
	}

	public function actionAddVatRate()
	{
		$this->setHeader('eshopCatalog.title.newVatRate', 'fa fa-list-alt');
	}

	public function actionAddTag()
	{
		$this->setHeader('eshopCatalog.title.newTag', 'fa fa-list-alt');
	}

	public function actionEdit($id)
	{
		/** @var Product $product */
		$product = $this->em->getRepository(Product::class)->find($id);

		if (!$product)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader($this->t('eshopCatalog.title.editProduct') . ': ' . $product->getProductText($this->translator->getLocale())->name, 'fa fa-list-alt');
	}

	public function actionEditVatRate($id)
	{
		/** @var VatRate $vatRate */
		$vatRate = $this->em->getRepository(VatRate::class)->find($id);

		if (!vatRate)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader($this->t('eshopCatalog.title.editVatRate') . ': ' . $vatRate->name, 'fa fa-list-alt');
	}

	public function actionEditTag($id)
	{
		/** @var Tag $tag */
		$tag = $this->em->getRepository(Tag::class)->find($id);

		if (!$tag)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader($this->t('eshopCatalog.title.editTag') . ': ' . $tag->getText($this->translator->getLocale())->name, 'fa fa-list-alt');
	}

	public function actionEditCategories($ids)
	{
		$this->setHeader('eshopCatalog.title.editProductsCategories', 'fa fa-list-alt');
	}

	public function actionEditVariants($ids)
	{
		$this->setHeader('eshopCatalog.title.editProductsVariants', 'fa fa-list-alt');
	}

	public function actionEditFeatures($ids)
	{
		$this->setHeader('eshopCatalog.title.editProductsFeatures', 'fa fa-list-alt');
	}

	public function actionClose($id)
	{
		$this->redirect('Default:');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	protected function createComponentProductForm(IProductFormFactory $product)
	{
		$control = $product->create();
		if ($this->getParameter('id'))
			$control->setProduct($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) use ($control) {
			$control->cleanSessionProduct();
			$this->redirect('Products:edit', $form->getCustomData('productId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) use ($control) {
			$control->cleanSessionProduct();
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) use ($control) {
			$control->cleanSessionProduct();
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsGrid(IProductsGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentVatRateForm(IVatRateFormFactory $vatRate)
	{
		$control = $vatRate->create();
		if ($this->getParameter('id'))
			$control->setVatRate($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editVatRate', $form->getCustomData('vatRateId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentVatRatesGrid(IVatRatesGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentProductsCategoryForm(IProductsCategoryFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editCategories', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsVariantForm(IProductsVariantFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editVariants', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsFeatureForm(IProductsFeatureFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editFeatures', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentTagForm(ITagFormFactory $tag)
	{
		$control = $tag->create();
		if ($this->getParameter('id'))
			$control->setTag($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editTag', $form->getCustomData('tagType'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentTagsGrid(ITagsGridFactory $factory)
	{
		return $factory->create();
	}

}
