<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\FrontModule\Model\Dao;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Nette\Caching\Cache;

/**
 * class Categories
 * @package EshopCatalog\Model
 *
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'eshopCatalogCategories';

	public static $allowGenerateLink = true;

	protected $entityClass = Category::class;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	public $cacheDep = [
		Cache::TAGS    => ['categories'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var array */
	protected $cCats, $cStructured, $cNavs, $cPath, $cQueryPath;

	/** @var bool */
	protected $cacheExist;

	/** @var LinkGenerator */
	public $linkGenerator;

	/**
	 * Categories constructor.
	 *
	 * @param CacheService $cacheService
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	protected function cacheExist()
	{
		if ($this->cacheExist === null)
			$this->cacheExist = count(glob(TMP_DIR . '/cache/_' . self::CACHE_NAMESPACE . '/*')) > 1;

		return $this->cacheExist;
	}

	public function getStructured(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cStructured[$lang])) {
			$key = 'structured/' . $lang;

			$this->cStructured[$lang] = $this->cacheService->categoryCache->load($key, function(&$dep) use ($lang) {
				$dep = $this->cacheDep;
				/** @var Dao\Category[] $flat */
				$flat = [];
				/** @var Dao\Category[][] $grouped */
				$grouped = [];
				/** @var Dao\Category[] $tree */
				$tree = [];

				$published = $this->getEr()->createQueryBuilder('c')
					->addSelect('ct, IDENTITY(c.parent) as parent')
					->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
					->where('c.isPublished = 1')
					->setParameter('lang', $lang)
					->orderBy('c.root')->addOrderBy('c.lft')
					->getQuery()->getResult(Query::HYDRATE_ARRAY);

				foreach ($published as $v) {
					$cat           = $v[0];
					$cat['texts']  = $cat['categoryTexts'][$lang];
					$cat['parent'] = $v['parent'];

					$grouped[$cat['parent']][$cat['id']] = $this->fillDao($cat);
					$flat[$cat['id']]                    = &$grouped[$cat['parent']][$cat['id']];
				}

				$this->em->clear([Category::class, CategoryTexts::class]);

				$createTree = function(&$list, $parent) use (&$createTree) {
					$tree = [];
					foreach ($parent as $k => $l) {
						if (isset($list[$l->id])) {
							if ($l->lvl >= 1)
								foreach ($list[$l->id] as $kk => $v) {
									$list[$l->id][$kk]->setParent($parent[$k]);
								}
							$l->setChild($createTree($list, $list[$l->id]));
						}

						$tree[] = $l;
					}

					return $tree;
				};

				$baseCat = $this->getEr()->createQueryBuilder('c')->select('c.id')
					->where('c.lvl = 0')->setMaxResults(1)->getQuery()->getSingleScalarResult();

				if (!$baseCat)
					return [];

				$dao = (new Dao\Category())
					->setId((int) $baseCat)
					->setParentId(0);

				$grouped[0][] = $dao;

				$tree = $createTree($grouped, $grouped[0])[0]->getChild();

				foreach ($flat as $k => &$v) {
					$v->link = $this->generateLink($v->id);
				}

				return $tree;
			});
		}

		return $this->cStructured[$lang];
	}

	/**
	 * @param string|null $lang
	 *
	 * @return Dao\Category[]
	 */
	public function getCategories(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if ($this->cCats[$lang] === null) {
			$key = 'cats/' . $lang;

			$this->cCats[$lang] = $this->cacheService->categoryCache->load($key, function(&$dep) use ($lang) {
				$dep    = $this->cacheDep;
				$result = [];

				$loop = function(array $arr) use (&$loop, &$result) {
					/** @var Dao\Category[] $arr */
					foreach ($arr as $v) {
						$result[$v->id] = $v;

						if ($v->getChild())
							$loop($v->getChild());
					}
				};

				$loop($this->getStructured($lang));

				return $result;
			});
		}

		return $this->cCats[$lang];
	}

	protected function generateLink(int $id): ?string
	{
		return $this->linkGenerator ? $this->linkGenerator->link('EshopCatalog:Front:Default:category', ['id' => $id]) : '';
	}

	/**
	 * @param Dao\Category $category
	 *
	 * @return Dao\Category[]
	 */
	public function getPath(Dao\Category $category): array
	{
		if (!$this->cPath[$category->id]) {
			$path = [$category];
			$p    = $category->getParent();

			while ($p) {
				$path[] = $p;
				$p      = $p->getParent();
			}
			$this->cPath[$category->id] = array_reverse($path);
		}

		return $this->cPath[$category->id];
	}

	public function getQueryPath(int $id): array
	{
		$lang = $this->translator->getLocale();
		$key  = 'path/' . $id . '/' . $lang;

		if (!isset($this->cQueryPath[$lang])) {
			foreach ($this->getEr()->createQueryBuilder('c')
				         ->select('c.id, IDENTITY(c.parent) as parent, ct.alias')
				         ->andWhere('c.isPublished = 1 OR c.lvl = 0')
				         ->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				         ->setParameter('lang', $lang)
				         ->groupBy('c.id')
				         ->getQuery()->getArrayResult() as $row)
				$this->cQueryPath[$lang][$row['id']] = $row;
		}

		if (!$this->cQueryPath[$key]) {
			$flat = $this->cQueryPath[$lang];
			$path = [];
			$i    = 0;
			$cat  = $flat[$id];

			while ($cat && $i < 20) {
				$path[$cat['id']] = $cat;
				$cat              = $flat[$cat['parent']] ?? null;
			}

			$this->cQueryPath[$key] = array_reverse($path, true);
		}

		return $this->cQueryPath[$key];
	}

	public function get(int $id): ?Dao\Category
	{
		return $this->getCategories()[$id] ?? null;
	}

	/**
	 * @param string $alias
	 *
	 * @return Dao\Category[]
	 */
	public function findByAlias(string $alias): array
	{
		$result = [];

		foreach ($this->getCategories() as $c) {
			if ($c->alias == $alias)
				$result[] = $c;
		}

		return $result;
	}

	public function findNavigationId(int $categoryId): ?int
	{
		if (!$this->cNavs) {
			$this->cNavs = $this->em->getRepository(Navigation::class)->createQueryBuilder('n')
				->select('n.id, n.componentParams')
				->where('n.componentType = :type')->setParameter('type', 'eshopCatalog.navigation.category')
				->getQuery()->getArrayResult();
		}

		foreach ($this->cNavs as $row) {
			if ($row['componentParams']['category'] == $categoryId)
				return (int) $row['id'];
		}

		return null;
	}

	/**
	 * @param Dao\Category      $category
	 * @param DaoNavigationItem $navigation
	 *
	 * @return Dao\Category[]
	 */
	public function getBreadcrumb(Dao\Category $category, DaoNavigationItem $navigation): array
	{
		$path = $this->getPath($category);

		$path[0]->name = $navigation->title;

		return $path;
	}

	protected function fillDao($category)
	{
		/** @var Dao\Category $c */
		$c = (new Dao\Category())
			->setId((int) $category['id'])
			->setName($category['texts']['name'])
			->setNameH1($category['texts']['nameH1'])
			->setAlias($category['texts']['alias'])
			->setShortDescription($category['texts']['shortDescription'])
			->setDescription($category['texts']['description'])
			->setImage($category['image'])
			->setLvl((int) $category['lvl'])
			->setSeo($category['texts']['seo'])
			->setCreated($category['created'])
			->setModified($category['modified'] ?: $category['created'])
			->setFiltersFromParent((int) $category['filtersFromParent'])
			->setAttrs($category['attrs'] ?: [])
			->setRod($category['rod']);

		$c->canProductsAddToCart = $category['canProductsAddToCart'] ? true : false;
		if ($category['canProductsAddToCart'] === null)
			$c->canProductsAddToCart = true;

		if ($category['parent'])
			$c->setParentId($category['parent'] ? (int) $category['parent'] : null);

		return $c;
	}
}
