<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Export;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Provider\IXmlExportService;
use EshopCatalog\Model\Config;
use EshopOrders\Model\PaymentSpeditions;
use Nette\Utils\FileSystem;

abstract class XmlService implements IXmlExportService
{
	/** @var string */
	protected $name;

	/** @var string */
	public $baseUrl;

	/** @var PaymentSpeditions @inject */
	public $paymentSpeditions;

	/** @var array|null */
	protected $cSpeditions = null;

	protected $fileHandle;

	public function getFilePathTemp(): string
	{
		return TMP_DIR . "/exportproducts/{$this->name}.xml";
	}

	public function getFilePath(): string
	{
		return WWW_DIR . "/exportproducts/{$this->name}.xml";
	}

	public function startExport(): void
	{
		FileSystem::createDir(dirname($this->getFilePathTemp()));;
		$this->fileHandle = fopen($this->getFilePathTemp(), 'w');
	}

	public function writeToFile(string $data): void
	{
		if ($this->fileHandle)
			fwrite($this->fileHandle, $data);
	}

	public function setBaseUrl(string $baseUrl) { $this->baseUrl = $baseUrl; }

	public function getName(): string { return $this->name; }

	public function prepareExportData(Product $product): array
	{
		$ef       = $product->getExtraFields();
		$export   = $ef['export'][$this->name];
		$category = [];

		if (isset($ef['categoryExports'][$this->name])) {
			$vals = $ef['categoryExports'][$this->name];

			$category['status']       = $vals['status'];
			$category['categoryText'] = $vals['categoryText'];
		}

		if (!$export)
			$export = ['status' => 2];

		if (empty($export) || $export['status'] == 2)
			$export['status'] = $category['status'];

		if (!$export['categoryText'])
			$export['categoryText'] = $category['categoryText'];

		if (!$export['productName'])
			$export['productName'] = trim(
				(Config::load('exportGrader.addManufacturerToName') &&$product->getManufacturer() ? $product->getManufacturer()->name . ' ' : '')
				. $product->getName());

		if (!$export['product'])
			$export['product'] = $export['productName'];

		foreach (['product', 'productName'] as $v)
			$export[$v] = preg_replace('/\s\s+/', ' ', $export[$v]);

		return $export;
	}

	public function endExport()
	{
		fclose($this->fileHandle);
		FileSystem::rename($this->getFilePathTemp(), $this->getFilePath(), true);
	}

	public function canExportFeature(int $featureId)
	{
		return 1;
	}
}
