<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\UI\BaseControl;
use EshopCatalog\AdminModule\Model\Features;
use Nette\Utils\Html;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\AdminModule\Model\FeatureValues;

class FeatureValuesGrid extends BaseControl
{
	/** @var Feature */
	public $feature;

	/** @var Features */
	protected $featuresService;

	/** @var FeatureValues */
	protected $featureValuesServices;

	public function __construct(Features $features, FeatureValues $featureValues)
	{
		$this->featuresService       = $features;
		$this->featureValuesServices = $featureValues;
	}

	public function render()
	{
		if (!$this->feature)
			return '';

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->featureValuesServices->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleRepairPositions()
	{
		$this->featureValuesServices->repairPositions($this->feature->getId());
		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		if (!$this->feature)
			return $grid;

		$qb = $this->featureValuesServices->getEr()->createQueryBuilder('fv')->addSelect('fvt')
			->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->andWhere('fv.feature = :id')->setParameter('id', $this->feature->getId())
			->orderBy('fv.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('featureValuesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.value')->setRenderer(function($row) {
			return Html::el('a', ['href' => $this->getPresenter()->link('VariantsFeatures:editFeatureValue', $row->getId())])
				->setText($row->getFeatureValueText($this->translator->getLocale())->name);
		});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filters
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editFeatureValue')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->featureValuesServices->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->featureValuesServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else
			$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setFeature($id)
	{
		$this->feature = $this->featuresService->get($id);

		if (!$this->feature)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}

