<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Core\Model\Event\DaoEvent;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;
use Users\Model\Http\UserStorage;
use EshopCatalog\FrontModule\Model\Dao;

class ProductsFacade
{
	/** @var ITranslator */
	protected $translator;

	/** @var CacheService */
	protected $cacheService;

	/** @var Products */
	public $productsService;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var UserStorage */
	protected $userStorage;

	/** @var Albums */
	protected $albumsService;

	/** @var Manufacturers */
	protected $manufacturersService;

	/** @var Tags */
	protected $tagsService;

	/** @var Categories */
	protected $categoriesService;

	public function __construct(ITranslator $translator, CacheService $cacheService, Products $products, EventDispatcher $eventDispatcher,
	                            UserStorage $userStorage, Albums $albums, Manufacturers $manufacturers, Tags $tags, Categories $categories)
	{
		$this->translator           = $translator;
		$this->cacheService         = $cacheService;
		$this->productsService      = $products;
		$this->eventDispatcher      = $eventDispatcher;
		$this->userStorage          = $userStorage;
		$this->albumsService        = $albums;
		$this->manufacturersService = $manufacturers;
		$this->tagsService          = $tags;
		$this->categoriesService    = $categories;
	}

	/**
	 * Vrátí DAO produktů z cache nebo získá data z db a uloží do cache
	 *
	 * @param array $ids
	 *
	 * @return Dao\Product[]
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProducts(array $ids): array
	{
		$whereIds = [];
		/** @var Dao\Product[] $result */
		$result = [];
		$locale = $this->translator->getLocale();

		$keys = [];
		foreach ($ids as $id)
			$keys[] = 'product/' . $id . '/' . $locale;

		// Načtení z cache
		foreach ($this->cacheService->productCache->bulkLoad($keys) as $key => $product) {
			$tmp = explode('/', $key);
			$id  = $tmp[1];

			if ($product)
				$result[$id] = $product;
			else
				$whereIds[] = $id;
		}

		// Vytvoření získání produktů co nejsou v cache
		if (!empty($whereIds)) {
			$query = (new ProductQuery($locale))
				->withCategories()
				->withTexts()
				->withVatRate()
				->byId($whereIds);

			$qb = $query->getQueryBuilder($this->productsService->getEr());
			$qb->groupBy('p.id');

			foreach ($this->productsService->customQB as $cqb)
				$cqb($qb);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$tmp                = $this->productsService->normalizeHydrateArray($row);
				$tmp['extraFields'] = $this->productsService->getAllExtraFields()[$tmp['id']] ?? [];
				$dao                = $this->productsService->fillDao($tmp);

				if ($this->linkGenerator)
					$dao->setLink($this->linkGenerator->link('EshopCatalog:Front:Default:product', [
						'id'     => $dao->getId(),
						'locale' => $locale,
					]));

				$cacheDep                = $this->productsService->cacheDep;
				$cacheDep[Cache::TAGS][] = 'product/' . $dao->getId();
				$cacheDep[Cache::EXPIRE] = '1 week';
				$this->cacheService->productCache->save('product/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				$result[$tmp['id']] = $dao;
			}
		}

		if (!empty($result)) {
			$this->productsService->loadPrices($result,
				$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);
			$this->tagsService->loadTagsToProduct($result);
			$this->productsService->loadFeatures($result);
			$this->productsService->loadQuantity($result);
			foreach ($result as $id => $product) {
				if ($product->galleryId)
					$product->setGallery($this->albumsService->get($product->galleryId));

				if ($product->manufacturerId)
					$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));

				if ($product->defaultCategoryId) {
					$product->defaultCategory = $this->categoriesService->get($product->defaultCategoryId);
					$product->canAddToCart    = $product->defaultCategory->canProductsAddToCart;
				}

				$this->eventDispatcher->dispatch(new DaoEvent($product), Products::class . '::afterFillDao');
			}
		}

		return $result;
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Product|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProduct(int $id): ?Dao\Product
	{
		return $this->getProducts([$id])[$id] ?? null;
	}
}
