<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Presenters;

use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Components\IProductsFilterFactory;
use EshopCatalog\FrontModule\Components\IProductsListFactory;
use EshopCatalog\FrontModule\Components\ProductPreview;
use EshopCatalog\FrontModule\Components\ProductsFilter;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Event\CategoryHeadEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use Gallery\FrontModule\Model\Albums;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Paths;

class DefaultPresenter extends BasePresenter
{
	/** @var Category */
	public $category;

	/** @var Product */
	public $product;

	/** @var TemplatePage */
	protected $templatePage;

	/** @var ProductsFacade @inject */
	public $productsFacade;

	/** @var Categories */
	protected $categoriesService;

	/** @var Products */
	protected $productsService;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var Paths */
	protected $pathsService;

	/** @var Albums */
	protected $albumsService;

	/** @var Tags */
	protected $tagsService;

	/** @var Config */
	protected $config;

	public function __construct(Config $config, Categories $categories, Products $products, TemplatePages $templatePages, Paths $paths, Albums $albums, Tags $tags)
	{
		$this->config               = $config;
		$this->categoriesService    = $categories;
		$this->productsService      = $products;
		$this->templatePagesService = $templatePages;
		$this->pathsService         = $paths;
		$this->albumsService        = $albums;
		$this->tagsService          = $tags;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateHome']);

		$this->template->pageClass          .= ' page-eshopcatalog-default';
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
	}

	public function actionProduct(int $id): void
	{
		$this->product = $product = $this->productsFacade->getProduct($id);

		if (!$product)
			$this->error('eshopCatalog.product.notFound');

		$this->templatePage = $templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateProduct']);

		$this->productsService->loadAlternative($product, $this->config->get('productDetailAlternativeLimit', 4));
		$this->productsService->loadVariants($product);
		$this->productsService->loadFeatures($product);
		$this->productsService->loadQuantity($product);
		$this->tagsService->loadTagsToProduct($product);
		if ($product->alternative)
			$this->tagsService->loadTagsToProduct($product->alternative);

		$this->setTitle($product->getSeo('title') ?: $product->name);
		$this->seoDataFromEntity = $product->getSeo();

		$this->setProductRemarketing();

		if ($product->getQuantity() <= 0)
			$this->seoDataFromEntity['robots'] = 'noindex';
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
	}

	public function actionCategory(int $id): void
	{
		$this->category = $category = $this->categoriesService->get($id);

		if (!$category)
			$this->error('eshopCatalog.category.notFound');

		$this->templatePage = $templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templatePage']);

		$this->category          = $category;
		$this->seoDataFromEntity = $category->getSeo();
		$this->templatePage      = $templatePage;

		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
	}

	public function actionSearch($q)
	{
		$templatePage       = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateResult']);
		$this->templatePage = $templatePage;

		$this['list']->setProductsBySearch((string) $q)
			->setFilterComponent($this['productsFilter']);

		$this->template->q                  = $q;
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-search';

		$this->setTitle($this->translator->translate('eshopCatalogFront.search.searchQuery', null, ['q' => $q]));
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->template));

		if (!$this->isAjax() && empty($this['list']->getProducts())) {
			$this->activeNavigation->setSeo('robots', 'noindex, nofollow');
		}

		$this->setCategoryRemarketing('view_search_results');

		if (Config::load('allowSearchInContent', false)) {
			$this->template->alowSearchInContent = true;
			$this->template->contentSeachLink    = $this->link(':Core:Front:Default:search', ['q' => $q]);
		}
	}

	/*******************************************************************************************************************
	 * ============================== Render
	 */

	public function renderProduct()
	{
		$this->categoriesService->getStructured();
		$product      = $this->product;
		$templatePage = $this->templatePage;

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-product product-' . $id;
		$this->template->product            = $product;

		if ($product->defaultCategoryId) {
			$category = $this->categoriesService->get((int) $product->defaultCategoryId);

			if (!$category)
				$category = $this->categoriesService->get((int) array_values($product->categories)[0]);

			if ($category) {
				$path    = $this->categoriesService->getPath($category);
				$baseNav = $this->categoriesService->findNavigationId($path[0]->id);
				$baseNav = $this->navigationsService->getNavigation($baseNav);

				if ($baseNav) {
					$this->template->breadcrumb = $this->categoriesService->getBreadcrumb($category, $baseNav);
				}
			}
		}
	}

	public function renderCategory()
	{
		$category     = $this->category;
		$templatePage = $this->templatePage;

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-category';
		$this->template->category           = $category;

		$path    = $this->categoriesService->getPath($category);
		$baseNav = $this->categoriesService->findNavigationId($path[0]->id);
		$baseNav = $this->navigationsService->getNavigation($baseNav);

		if ($baseNav) {
			$this->template->breadcrumb = $this->categoriesService->getBreadcrumb($category, $baseNav);
		}

		$this->processCategory();

		$this->setCategoryRemarketing('view_item_list');

		if (!$this->isAjax() && empty($this['list']->getProducts())) {
			$this->seoDataFromEntity['robots'] = 'noindex, nofollow';
		}

		if (!$category->getSeo('description'))
			$this['meta']->setMeta('description', Strings::truncate(str_replace('&nbsp;', ' ', strip_tags($category->description)), 130));
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentCartAddForm(ICartAddFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->product)
			$control->setProduct($this->product);

		return $control;
	}

	protected function createComponentList(IProductsListFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentProductsFilter(IProductsFilterFactory $factory)
	{
		$control = $factory->create();

		$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function() use ($control) {
			$do = $this->getParameter('do');
			if ($do === null || strpos((string) $do, 'cartAddForm-form-submit') === false) {
				$this->redrawControl('categoryHead');
				$this['list']->redrawControl('list');
			}
		});

		return $control;
	}

	protected function createComponentProductPreview(IProductPreviewFactory $factory): Multiplier
	{
		return new Multiplier(function(int $id) use ($factory): ProductPreview {
			$control = $factory->create();

			$control->setProductById($id);

			return $control;
		});
	}

	/*******************************************************************************************************************
	 * ==============================
	 */

	protected function processCategory()
	{
		$this['list']->setCategory($this->category->getId())
			->setFilterComponent($this['productsFilter']);
		$paginator    = $this['list']->getPaginator();
		$title        = $this->category->getSeo('title') ?: $this->category->name;
		$description  = $this->category->description;
		$categoryName = $this->category->nameH1;

		$this->eventDispatcher->dispatch('eshopCatalog.default.actionCategory.head', new CategoryHeadEvent($title, $categoryName, $description,
			$this['productsFilter']->getFilterValues(), $this->category));

		$titleSuffix = null;
		if ($paginator->getPage() > 1) {
			$titleSuffix = $this->translator->translate('eshopCatalogFront.pageXOfY', null, ['x' => $paginator->getPage(),
			                                                                                 'y' => $paginator->getPageCount()]);
			$title       .= ' ' . $titleSuffix;
			$description = '';
		}

		$this->setTitle($title);
		$this->template->categoryName = $categoryName;
		$this->template->description  = $description;
		$this->template->titleSuffix  = $titleSuffix;
		$this->template->pageTitle    = trim($categoryName . ' ' . $titleSuffix);
		$this->template->page         = $paginator->getPage();
	}

	protected function setProductRemarketing()
	{
		$gtmRemarketingData = [
			'event' => 'view_item',
			'value' => $this->product->getPrice(),
			'items' => [
				[
					'id'                       => $this->product->getId(),
					'google_business_vertical' => 'retail',
				],
			],
		];
		$this['gtmRemarketing']->setData($gtmRemarketingData);
	}

	protected function setCategoryRemarketing($type)
	{
		$priceSum = 0;
		$items    = [];
		foreach ($this['list']->getProducts() as $product) {
			$priceSum += $product->getPrice();
			$items[]  = [
				'id'                       => $product->getId(),
				'google_business_vertical' => 'retail',
			];
		}

		$gtmRemarketingData = [
			'event' => $type,
			'value' => $priceSum,
			'items' => $items,
		];
		$this['gtmRemarketing']->setData($gtmRemarketingData);
	}
}
