<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Caching\Cache;

/**
 * Class FeatureProducts
 * @package EshopCatalog\FrontModule\Model
 */
class FeatureProducts extends BaseFrontEntityService
{

	protected $entityClass = FeatureProduct::class;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cProductFeatures = [];

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => ['featureProduct'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	/**
	 * @param int[] $ids
	 *
	 * @return Dao\FeatureProduct[]
	 * @throws \Throwable
	 */
	public function getFeaturesForProduct(array $ids): array
	{
		$whereIds = [];
		$result   = [];
		$locale   = $this->translator->getLocale();

		foreach ($ids as $id) {
			if (isset($this->cProductFeatures[$id]))
				$result[$id] = $this->cProductFeatures[$id];
			else
				$whereIds[] = $id;
		}

		if (!empty($whereIds)) {
			$qb = $this->getEr()->createQueryBuilder('fp')
				->select('IDENTITY(fp.product) as product, IDENTITY(ft) as featureId, ft.name as featureName, fv.id as featureValueId, fvt.name as featureValueName')
				->join('fp.feature', 'f')
				->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				->join('fp.featureValue', 'fv')
				->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				->setParameter('lang', $locale)
				->andWhere('fp.product IN (:id)')->setParameter('id', $ids);

			foreach ($whereIds as $id) {
				$this->cProductFeatures[$id] = [];
			}

			/** @var Dao\FeatureProduct[][] $data */
			$data = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				if (!$row['featureName'])
					continue;

				$data[$row['product']][$row['featureId']] = $this->fillDao($row);
			}

			foreach ($data as $productId => $features) {
				$result[$productId]                 = $features;
				$this->cProductFeatures[$productId] = $features;
			}
		}

		return $result;
	}

	/**
	 * @param array $arr
	 *
	 * @return Dao\FeatureProduct
	 */
	protected function fillDao($arr): Dao\FeatureProduct
	{
		$fp                 = new Dao\FeatureProduct();
		$fp->idProduct      = $arr['productId'];
		$fp->idFeature      = $arr['featureId'];
		$fp->name           = $arr['featureName'];
		$fp->idFeatureValue = $arr['featureValueId'];
		$fp->value          = $arr['featureValueName'];

		return $fp;
	}
}
