<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Components\Flashes\Flashes;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\AdminModule\Model\Features;

class FeatureForm extends BaseControl
{
	/** @var Feature */
	public $feature;

	/** @var FeatureTexts */
	public $featureTexts;

	/** @var \EshopCatalog\AdminModule\Model\Features */
	protected $featureServices;

	public function __construct(Features $features)
	{
		$this->featureServices = $features;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.featureForm.name')->setIsMultilanguage()
			->setMaxLength(255);
		$form->addBool('isPublished', 'eshopCatalog.featureForm.isPublished')->setDefaultValue(1);
		$form->addBool('useAsFilter', 'eshopCatalog.featureForm.useAsFilter')->setDefaultValue(1);

		$this->eventDispatcher->dispatch(FeatureForm::class . '::createForm',
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null));

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$feature = $this->feature ?: new Feature();

			$feature->isPublished = $values->isPublished;
			$feature->useAsFilter = $values->useAsFilter;

			$langValues = $form->convertMultilangValuesToArray();
			$texts      = $feature->getFeatureTexts()->toArray();

			foreach ($langValues as $lang => $vals) {
				if ($vals['name'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text       = $texts[$lang] ?? new FeatureTexts($feature, $lang);
				$text->name = $vals['name'];

				$this->em->persist($text);
				$texts[$lang] = $text;
			}

			$feature->setFeatureTexts($texts);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $feature;
			$this->eventDispatcher->dispatch(FeatureForm::class . '::formSuccess', $event);

			$this->em->persist($feature);
			$this->em->flush();
			$form->addCustomData('featureId', $feature->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.featureForm.featureSaved', Flashes::FLASH_SUCCESS);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setFeature($id)
	{
		$this->feature = $this->featureServices->get($id);

		if ($this->feature) {
			$form = $this['form'];
			$form->setDefaults([
				'isPublished' => $this->feature->isPublished,
				'useAsFilter' => $this->feature->useAsFilter,
			]);

			$mTexts = $form->prepareMultilangTextsForForm(['name'], $this->feature->getFeatureTexts()->toArray());
			$form->setDefaults($mTexts['texts']);

			$this->eventDispatcher->dispatch(FeatureForm::class . '::setFeature', new SetFormDataEvent($form, $this->feature));
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}

