<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Contributte\EventDispatcher\EventDispatcher;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use Nette\Utils\ArrayHash;
use Nette\Application\UI\Multiplier;

class CartAddForm extends BaseControl
{
	/** @var EventDispatcher */
	public $eventDispatcher;

	/** @var Products */
	protected $productsService;

	/** @var int @persistent */
	protected $productId;

	/** @var Product */
	protected $product;

	public function __construct(EventDispatcher $eventDispatcher, Products $products)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->productsService = $products;
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->productId && !$this->product) {
			$this->setProduct($this->productId);
		}
	}

	public function render(): void
	{
		$this->template->product       = $this->product;
		$this->template->addToCartText = $this->template->addToCartText ?? $this->t('eshopCatalogFront.cartAddForm.addToCart');
		$this->template->nowAdded      = $this->template->nowAdded ?? false;

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addHidden('productId');
		$form->addHidden('variantId')->setHtmlAttribute('class', ['variantIdInput']);
		$form->addText('quantity')->setDefaultValue(1)->addRule(BaseForm::MIN, '_eshopCatalogFront.cart.minimumQuantity', 1);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->getPresenter();

		$addedItem            = new AddedCartItem();
		$addedItem->quantity  = (int) $values->quantity ?: 1;
		$addedItem->productId = (int) $values->productId;
		$addedItem->optionId  = $values->variantId;

		$this->eventDispatcher->dispatch('eshopOrders.cartAddItem', new AddedCartItemEvent($addedItem));
		$this->template->addToCartText = $this->t('eshopCatalogFront.cartAddForm.addedToCart');
		$this->template->nowAdded      = true;
		$this->redrawControl('form');
	}

	/**
	 * @param int|Product $product
	 */
	public function setProduct($product): self
	{
		if ($product instanceof Product) {
			$this->product   = $product;
			$this->productId = $product->getId();
		} else {
			$this->productId = $product;
		}

		if ($this->productId) {
			$this['form']->setDefaults([
				'productId' => $this->productId,
				//TODO default variant ID
			]);
		}

		return $this;
	}
}
