<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Event\FilterEvent;
use EshopCatalog\FrontModule\Model\FeatureProducts;
use EshopCatalog\FrontModule\Model\FilterService;
use EshopCatalog\FrontModule\Model\ProductVariants;
use Slides\FrontModule\Model\Dao\Group;

/**
 * TODO filtr správně nedělat přes handle ale pomocí jednotného parametru
 *
 * Class ProductsFilter
 * @package EshopCatalog\FrontModule\Components
 */
class ProductsFilter extends BaseControl
{
	/** @var int[] */
	public $productIds, $productIdsNoPriceFilter;

	/** @var int */
	public $categoryId;

	/** @var FeatureProducts */
	protected $featureProducts;

	/** @var ProductVariants */
	protected $productVariants;

	/** @var FilterService */
	protected $filterService;

	/** @var Categories */
	protected $categoriesService;

	/** @var callable */
	public $onFilter = [];

	/** @var array */
	protected $cFeatures, $cMinMax, $cVariants, $cManu;

	/** @persistent */
	public $filter = [];

	/** @persistent */
	public $min;

	/** @persistent */
	public $max;

	/** @persistent */
	public $sort;

	public function __construct(FeatureProducts $featureProducts, ProductVariants $productVariants, FilterService $filterService, Categories $categories)
	{
		$this->featureProducts   = $featureProducts;
		$this->productVariants   = $productVariants;
		$this->filterService     = $filterService;
		$this->categoriesService = $categories;
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		$signal = $this->getPresenter()->getSignal();

		if ($this->getPresenter()->isAjax() && !(is_array($signal) && in_array('formSubmit', $signal))) {
			$this->eventDispatcher->dispatch(ProductsFilter::class . '::onFilter', new FilterEvent($this->getFilterValues()));
			$this->getPresenter()->payload->url = $this->link('this');
			$this->redrawControl('filters');
		}
	}

	public function render()
	{
		$ac           = $this->getFilterValues();
		$features     = $this->getFeatures();
		$variants     = $this->getVariants();
		$minMax       = $this->getMinMax();
		$manu         = $this->getManufacturers();
		$filterActive = false;

		$inActiveGroups = [];
		foreach ($features as $group) {
			$hasActive = false;
			foreach ($group->items as $itemK => $item) {
				if ($item->productsCountInit === 0) {
					unset($group->items[$itemK]);
					continue;
				}
				if (isset($ac['features'][$group->id]) && in_array($item->id, $ac['features'][$group->id])) {
					$item->isActive = true;
					$hasActive      = true;
					$filterActive   = true;
				}
			}

			if (!$hasActive)
				$inActiveGroups['feature'][] = $group->id;
		}

		foreach ($variants as $group) {
			$hasActive = false;
			foreach ($group->items as $item) {
				if (isset($ac['variants'][$group->id]) && in_array($item->id, $ac['variants'][$group->id])) {
					$item->isActive = true;
					$hasActive      = true;
					$filterActive   = true;
				}
			}

			if (!$hasActive)
				$inActiveGroups['variant'][] = $group->id;
		}

		$hasActive = false;
		foreach ($manu->items as $itemK => $item) {
			if ($item->productsCountInit === 0) {
				unset($manu->items[$itemK]);
				continue;
			}
			if (isset($ac['manu']) && in_array($item->id, $ac['manu'])) {
				$item->isActive = true;
				$hasActive      = true;
				$filterActive   = true;
			}
		}

		if (!$hasActive)
			$inActiveGroups['manu'] = $manu->id;

		$this->template->minMax         = $minMax;
		$this->template->currentMinMax  = ['min' => $this->min ?: $minMax['min'],
		                                   'max' => $this->max ?: $minMax['max']];
		$this->template->inActiveGroups = $inActiveGroups;
		$this->template->variants       = $variants;
		$this->template->features       = $features;
		$this->template->manu           = $manu;
		$this->template->sort           = [
			'recommended',
			'p.created DESC', 'p.created ASC',
			'p.price DESC', 'p.price ASC',
			'pt.name ASC', 'pt.name DESC',
		];

		$this->template->currentSort  = $this->sort;
		$this->template->filterActive = $filterActive || $this->sort || $this->min || $this->max;
		$this->template->resetLink    = $this->getPresenter()->getHttpRequest()->getUrl()->getPath();

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	public function handleSet($t, $k, $v)
	{
		$this->filter['f' . $t][$k][] = $v;
		if ($this->getPresenter()->isAjax())
			$this->onFilter($this->getFilterValues());
		else
			$this->redirect('this');
	}

	public function handlePrice($min, $max)
	{
		$this->min = $this->getMinMax()['min'] == $min ? null : $min;
		$this->max = $this->getMinMax()['max'] == $max ? null : $max;

		if ($this->getPresenter()->isAjax())
			$this->onFilter($this->getFilterValues());
		else
			$this->redirect('this');
	}

	public function handleSort($sort)
	{
		if ($sort == 'recommended')
			$sort = null;
		$this->sort = $sort;

		if ($this->getPresenter()->isAjax())
			$this->onFilter($this->getFilterValues());
		else
			$this->redirect('this');
	}

	public function handleFormSubmit()
	{
		$presenter   = $this->getPresenter();
		$httpRequest = $presenter->getHttpRequest();
		$this->resetFilter();

		$variants = $httpRequest->getPost('variant');
		if ($variants) {
			foreach ($variants as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['fv'][$g] = implode('|', $vals);
			}
		}
		$features = $httpRequest->getPost('feature');
		if ($features) {
			foreach ($features as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['ff'][$g] = implode('|', $vals);
			}
		}
		$manufacturers = $httpRequest->getPost('manu');
		if ($manufacturers) {
			foreach ($manufacturers as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['fm'][$g] = implode('|', $vals);
			}
		}

		$priceRange = $httpRequest->getPost('priceRange');
		if ($priceRange) {
			if (isset($priceRange['min']) && $this->getMinMax()['min'] != $priceRange['min'])
				$this->min = $priceRange['min'];
			if (isset($priceRange['max']) && $this->getMinMax()['max'] != $priceRange['max'])
				$this->max = $priceRange['max'];
		}

		$sort = $httpRequest->getPost('sort');
		if ($sort)
			$this->sort = $sort;

		if ($presenter->isAjax()) {
			$this->eventDispatcher->dispatch(ProductsFilter::class . '::onFilter', new FilterEvent($this->getFilterValues()));
			$this->getPresenter()->payload->url = $this->link('this');
			$this->redrawControl('filters');
		} else {
			$this->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */


	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function getFeatures()
	{
		if ($this->cFeatures === null) {
			if ($this->categoryId) {
				$cat             = $this->categoriesService->get($this->categoryId);
				$this->cFeatures = $this->filterService->getCategoryFilters((int) $this->categoryId, $cat->filtersFromParent ? true : false);
			} else {
				$this->cFeatures = $this->filterService->getFeatures($this->productIds);
			}
		}

		return $this->cFeatures;
	}

	public function setManufacturerProductsCount(int $item, int $count, string $type = 'default')
	{
		if (isset($this->cManu->items[$item])) {
			if ($type == 'default')
				$this->cManu->items[$item]->productsCount = $count;
			else if ($type == 'init')
				$this->cManu->items[$item]->productsCountInit = $count;
		}
	}

	/**
	 * @param int    $group
	 * @param int    $item
	 * @param int    $count
	 * @param string $type
	 */
	public function setFeatureProductsCount(int $group, int $item, int $count, string $type = 'default')
	{
		if (isset($this->cFeatures[$group]->items[$item])) {
			if ($type == 'default')
				$this->cFeatures[$group]->items[$item]->productsCount = $count;
			else if ($type == 'init')
				$this->cFeatures[$group]->items[$item]->productsCountInit = $count;
		}
	}

	public function getVariants()
	{
		if (!$this->cVariants)
			$this->cVariants = $this->filterService->getVariants($this->productIds);

		return $this->cVariants;
	}

	public function getMinMax()
	{
		if (!$this->cMinMax && $this->productIdsNoPriceFilter)
			$this->cMinMax = $this->filterService->getMinMaxPrice($this->productIdsNoPriceFilter);

		return $this->cMinMax;
	}

	public function getManufacturers()
	{
		if (!$this->cManu) {
			$this->cManu = (new FilterGroup())
				->setType('manu')
				->setId(0)
				->setName($this->t('eshopCatalogFront.filter.manufacturer'))
				->setItems($this->filterService->getManufacturers($this->productIds));
		}

		return $this->cManu;
	}

	public function getFilterValues()
	{
		$return = [];

		if (isset($this->filter['fv']))
			foreach ($this->filter['fv'] as $group => $vals)
				$return['variants'][$group] = explode('|', $this->filter['fv'][$group]);
		if (isset($this->filter['ff']))
			foreach ($this->filter['ff'] as $group => $vals)
				$return['features'][$group] = explode('|', $this->filter['ff'][$group]);
		if (isset($this->filter['fm']))
			foreach ($this->filter['fm'] as $group => $vals)
				$return['manu'] = explode('|', $this->filter['fm'][$group]);
		if ($this->min)
			$return['priceRange']['min'] = $this->min;
		if ($this->max)
			$return['priceRange']['max'] = $this->max;

		if ($this->sort)
			$return['sort'] = $this->sort;

		return $return;
	}

	protected function resetFilter()
	{
		$this->filter = null;
		$this->min    = null;
		$this->max    = null;
		$this->sort   = null;
	}
}
