<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Caching\Cache;

/**
 * Class Manufacturers
 * @package EshopCatalog\FrontModule\Model
 */
class Manufacturers extends BaseFrontEntityService
{

	const CACHE_NAMESPACE = 'manufacturers';

	protected $entityClass = Manufacturer::class;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['manufacturers'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/**
	 * Manufacturers constructor.
	 *
	 * @param CacheService $cacheService
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @param int $id
	 *
	 * @return Manufacturer[]|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$locale = $this->defaultLang->locale;
		$key    = 'manufacturer_' . $locale . '_' . $id;

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'manufacturer/' . $id;

			$manufacturer = $this->getEr()->createQueryBuilder('m', 'm.id')
				->addSelect('mt')
				->andWhere('m.id = :id')->setParameter('id', $id)
				->leftJoin('m.texts', 'mt', 'WITH', 'mt.lang = :lang')
				->setParameter('lang', $locale)
				->andWhere('m.isPublished = 1')
				->groupBy('m.id')
				->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			return $manufacturer ? $this->fillDao($manufacturer) : null;
		});
	}

	/**
	 * @param array $a
	 *
	 * @return Dao\Manufacturer[]
	 */
	protected function fillDao($a)
	{
		$locale = $this->defaultLang->locale;

		$manufacturer = (new Dao\Manufacturer())
			->setId($a['id'])
			->setName($a['name'])
			->setLogo($a['logo'])
			->setDescription($a['texts'][$locale]['description']);

		return $manufacturer;
	}
}
