<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TSeo;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\Strings;

/**
 * @ORM\Table(name="eshop_catalog__product_texts", indexes={
 *     @ORM\Index(name="lang", columns={"lang"})
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"ProductTextListener"})
 */
class ProductTexts
{
	use MagicAccessors;
	use TSeo;

	/**
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Product", inversedBy="ProductTexts", cascade={"persist", "remove"})
	 * @ORM\JoinColumn(name="id", referencedColumnName="id", onDelete="CASCADE")
	 */
	private $id;

	/**
	 * @ORM\Id
	 * @ORM\Column(name="lang", type="string", length=2)
	 */
	private $lang;

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string", length=255)
	 */
	protected $name;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255)
	 */
	public $alias;

	/**
	 * @var string
	 * @ORM\Column(name="short_description", type="string", nullable=true, length=255)
	 */
	protected $shortDescription;

	/**
	 * @var string
	 * @ORM\Column(name="description", type="text", nullable=true)
	 */
	public $description;

	public function __construct($id, $lang)
	{
		$this->id   = $id;
		$this->lang = $lang;
		$this->seo  = [];
	}

	/**
	 * @param string $name
	 *
	 * @return $this
	 */
	public function setName($name)
	{
		$this->name = $name;
		if (!$this->alias)
			$this->alias = Strings::webalize($name);

		return $this;
	}

	/**
	 * @param string $alias
	 *
	 * @return $this
	 */
	public function setAlias($alias)
	{
		$this->alias = $alias && $alias != '' ? $alias : Strings::webalize($this->name);


		return $this;
	}

	public function getLang()
	{
		return $this->lang;
	}

	public function getProduct() { return $this->id; }

	public function setDescription(string $text): self
	{
		$this->description = $text;

		return $this;
	}

	public function setShortDescription(string $text): self
	{
		$this->shortDescription = Strings::truncate($text, 255);

		return $this;
	}
}

