<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;
use Users\Model\Http\UserStorage;
use EshopCatalog\FrontModule\Model\Dao;

class ProductsFacade
{
	/** @var ITranslator */
	protected $translator;

	/** @var CacheService */
	protected $cacheService;

	/** @var Products */
	protected $productsService;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var UserStorage */
	protected $userStorage;

	/** @var Albums */
	protected $albumsService;

	/** @var Manufacturers */
	protected $manufacturersService;

	/** @var Tags */
	protected $tagsService;

	public function __construct(ITranslator $translator, CacheService $cacheService, Products $products,
	                            UserStorage $userStorage, Albums $albums, Manufacturers $manufacturers, Tags $tags)
	{
		$this->translator           = $translator;
		$this->cacheService         = $cacheService;
		$this->productsService      = $products;
		$this->userStorage          = $userStorage;
		$this->albumsService        = $albums;
		$this->manufacturersService = $manufacturers;
		$this->tagsService          = $tags;
	}

	/**
	 * Vrátí DAO produktů z cache nebo získá data z db a uloží do cache
	 *
	 * @param array $ids
	 *
	 * @return Dao\Product[]
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProducts(array $ids): array
	{
		$whereIds = [];
		/** @var Dao\Product[] $result */
		$result = [];
		$locale = $this->translator->getLocale();

		// Prohledání vytvořených produktů v cache
		foreach ($ids as $id) {
			$product = $this->cacheService->productCache->load('product/' . $id . '/' . $locale);
			if ($product)
				$result[$id] = $product;
			else
				$whereIds[] = $id;
		}

		// Vytvoření získání produktů co nejsou v cache
		if (!empty($whereIds)) {
			$query = (new ProductQuery($locale))
				->withCategories()
				->withTexts()
				->withVatRate()
				->byId($whereIds);

			$qb = $query->getQueryBuilder($this->productsService->getEr());
			$qb->groupBy('p.id');

			foreach ($this->productsService->customQB as $cqb)
				$cqb($qb);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$tmp                = $this->productsService->normalizeHydrateArray($row);
				$tmp['extraFields'] = $this->productsService->getAllExtraFields()[$tmp['id']] ?? [];
				$dao                = $this->productsService->fillDao($tmp);

				if ($this->linkGenerator)
					$dao->setLink($this->linkGenerator->link('EshopCatalog:Front:Default:product', [
						'id'     => $dao->getId(),
						'locale' => $locale,
					]));

				$cacheDep                = $this->productsService->cacheDep;
				$cacheDep[Cache::TAGS][] = 'product/' . $dao->getId();
				$cacheDep[Cache::EXPIRE] = '1 week';
				$this->cacheService->productCache->save('product/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				$result[$tmp['id']] = $dao;
			}
		}

		if (!empty($result)) {
			if ($this->userStorage->getIdentity())
				$this->productsService->loadPrices($result, $this->userStorage->getIdentity()->getId());
			$this->tagsService->loadTagsToProduct($result);
			foreach ($result as $id => $product) {
				if ($product->galleryId)
					$product->setGallery($this->albumsService->get($product->galleryId));

				if ($product->manufacturerId)
					$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));
			}
		}

		return $result;
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Product|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProduct(int $id): ?Dao\Product
	{
		return $this->getProducts([$id])[$id] ?? null;
	}
}
