<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Presenters;

use EshopCatalog\FrontModule\Model\GoogleXmlService;
use EshopCatalog\FrontModule\Model\HeurekaXmlService;
use EshopCatalog\FrontModule\Model\ProductExports;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ZboziXmlService;
use Nette\Application\Responses\JsonResponse;
use Nette\Application\Responses\TextResponse;

class ExportPresenter extends BasePresenter
{
	/** @var Products */
	protected $productsService;

	/** @var ZboziXmlService */
	protected $zboziXmlService;

	/** @var GoogleXmlService */
	protected $googleXmlService;

	/** @var HeurekaXmlService */
	protected $heurekaXmlService;

	/** @var ProductExports */
	protected $productExports;

	protected $fileHandles = [];

	public function __construct(Products $products, ProductExports $productExports, ZboziXmlService $zboziXml, GoogleXmlService $googleXml, HeurekaXmlService $heurekaXml)
	{
		$this->productsService   = $products;
		$this->productExports    = $productExports;
		$this->zboziXmlService   = $zboziXml;
		$this->googleXmlService  = $googleXml;
		$this->heurekaXmlService = $heurekaXml;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionPorovnavace()
	{
		$types = $this->getHttpRequest()->getQuery('type');
		set_time_limit(600);

		if (in_array('zbozi', $types)) {
			$this->zboziXmlService->setBaseUrl(substr($this->baseUrl, 0, -1)); //bez koncoveho lomitka
			$this->startExportZbozi();
		}
		if (in_array('heureka', $types)) {
			$this->heurekaXmlService->setBaseUrl(substr($this->baseUrl, 0, -1)); //bez koncoveho lomitka
			$this->startExportHeureka();
		}
		if (in_array('google', $types)) {
			$this->googleXmlService->setBaseUrl(substr($this->baseUrl, 0, -1)); //bez koncoveho lomitka
			$this->startExportGoogle();
		}

		//casem mozna optimalizace - nacitat a vypisovat do souboru po castech (napr 100 produktu)
		$filter   = ['onlyStock' => true];
		$limit    = 300;
		$i        = 1;
		$exports  = $this->productExports->getAll();
		$products = $this->productsService->getProductsAll(0, $limit, 'p.id', $filter);
		$temp     = false;
		while ($products) {
			foreach ($products as $product) {
				$export = $exports[$product->getId()] ?? null;

				if (!$export)
					continue;
				$temp = true;
				if (in_array('zbozi', $types))
					if ($export['isZbozi'])
						$this->generateItemZbozi($product);
				if (in_array('heureka', $types))
					if ($export['isHeureka'])
						$this->generateItemHeureka($product);
				if (in_array('google', $types))
					if ($export['isGoogle'])
						$this->generateItemGoogle($product);
			}

			$start    = $limit * $i++;
			$products = [];
			$products = $this->productsService->getProductsAll($start, $limit, 'p.id', $filter);
		}

		if (in_array('zbozi', $types))
			$this->endExportZbozi();
		if (in_array('heureka', $types))
			$this->endExportHeureka();
		if (in_array('google', $types))
			$this->endExportGoogle();

		$response = new TextResponse('OK');
		$this->sendResponse($response);
	}

	/*******************************************************************************************************************
	 * ============================== Helper Export functions
	 * *************** START export
	 */

	protected function startExportZbozi()
	{
		$type                     = 'zbozi';
		$file                     = WWW_DIR . "/exportproducts/{$type}.xml";
		$this->fileHandles[$type] = fopen($file, 'w'); //pro zapis, vycistit soubor
		fwrite($this->fileHandles[$type], '<' . '?xml version="1.0" encoding="utf-8"?' . '>' . "\r\n");
		fwrite($this->fileHandles[$type], '<SHOP xmlns="http://www.zbozi.cz/ns/offer/1.0">' . "\r\n");
	}

	protected function startExportHeureka()
	{
		$type                     = 'heureka';
		$file                     = WWW_DIR . "/exportproducts/{$type}.xml";
		$this->fileHandles[$type] = fopen($file, 'w'); //pro zapis, vycistit soubor
		fwrite($this->fileHandles[$type], '<' . '?xml version="1.0" encoding="utf-8"?' . '>' . "\r\n");
		fwrite($this->fileHandles[$type], '<SHOP>' . "\r\n");
	}

	protected function startExportGoogle()
	{
		$type                     = 'google';
		$file                     = WWW_DIR . "/exportproducts/{$type}.xml";
		$this->fileHandles[$type] = fopen($file, 'w'); //pro zapis, vycistit soubor
		fwrite($this->fileHandles[$type], '<' . '?xml version="1.0" encoding="utf-8"?' . '>' . "\r\n");
		fwrite($this->fileHandles[$type], '<rss version="2.0" xmlns:g="http://base.google.com/ns/1.0">' . "\r\n" .
			'<channel>' . "\r\n" .
			'<title>' . $this->googleXmlService->getSiteName() . '</title>' . "\r\n" .
			'<link>' . $this->baseUrl . '</link>' . "\r\n" .
			'<description>Produkty obchodu ' . $this->googleXmlService->getSiteName() . '</description>' . "\r\n");
	}

	/***************** END export
	 */

	protected function endExportZbozi()
	{
		fwrite($this->fileHandles['zbozi'], '</SHOP>');
	}

	protected function endExportHeureka()
	{
		fwrite($this->fileHandles['heureka'], '</SHOP>');
	}

	protected function endExportGoogle()
	{
		fwrite($this->fileHandles['google'], "</channel> \r\n </rss>");
	}

	/***************** GENERATE ITEM
	 */

	protected function generateItemZbozi($product)
	{
		$shopItemDom = $this->zboziXmlService->createXmlShopitem($product);
		fwrite($this->fileHandles['zbozi'], $shopItemDom->saveXML($shopItemDom->documentElement) . "\r\n");
	}

	protected function generateItemHeureka($product)
	{
		$shopItemDom = $this->heurekaXmlService->createXmlShopitem($product);
		fwrite($this->fileHandles['heureka'], $shopItemDom->saveXML($shopItemDom->documentElement) . "\r\n");
	}

	protected function generateItemGoogle($product)
	{
		$shopItemDom = $this->googleXmlService->createXmlShopitem($product);
		fwrite($this->fileHandles['google'], $shopItemDom->saveXML($shopItemDom->documentElement) . "\r\n");
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

}
