<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Variants;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Variants;
use Nette\Utils\Html;

/**
 * Class VariantsGrid
 * @package EshopCatalog\AdminModule\Components\Variants
 */
class VariantsGrid extends BaseControl
{
	/** @var Variants */
	protected $variantsService;

	public function __construct(Variants $variants)
	{
		$this->variantsService = $variants;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->variantsService->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);

		$qb = $this->variantsService->getEr()->createQueryBuilder('v')->addSelect('vt')
			->join('v.texts', 'vt', 'WITH', 'vt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('v.position');

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('variantsGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(function($row) {
			return Html::el('a', ['href' => $this->getPresenter()->link('VariantsFeatures:editVariant', $row->getId())])
				->setText($row->getText($this->translator->getLocale())->name);
		});
		$grid->addColumnStatus('useAsFilter', 'eshopCatalog.defaultGrid.useAsFilter')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.enabled')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unabled')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridEnabledFilterChange'];
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editVariant')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		// Filter
		$grid->addFilterSelect('useAsFilter', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Prototype
		$grid->getColumn('useAsFilter')->getElementPrototype('td')->class[] = 'w1nw';
		$grid->getColumn('isPublished')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridEnabledFilterChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->variantsService->setUseAsFilter($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.filterChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.filterChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$this->redirect('this');
	}

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->variantsService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->variantsService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else
			$this->redirect('this');
	}
}
