<?php declare(strict_types = 1);


namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductExport;
use Doctrine\ORM\Query;
use Nette\Caching\Cache;

/**
 * Class ProductExports
 * @package EshopCatalog\FrontModule\Model
 */
class ProductExports extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'productExports';

	protected $entityClass = ProductExport::class;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['productExports'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/**
	 * ProductExports constructor.
	 *
	 * @param CacheService $cacheService
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @return array
	 */
	public function getAll()
	{
		$result = [];
		foreach ($this->getEr()->createQueryBuilder('e')->select('e, p.id as productId')
			         ->join(Product::class, 'p', 'WITH', 'p.productExport = e.id')
			         ->getQuery()->getArrayResult() as $row)
			$result[$row['productId']] = $row[0];

		return $result;
	}

	/**
	 * @param int $id
	 *
	 * @return Manufacturer[]|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$key = 'productExport_' . $locale . '_' . $id;

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'productExport/' . $id;

			$productExport = $this->getEr()->createQueryBuilder('pe', 'pe.id')
				->andWhere('pe.id = :id')->setParameter('id', $id)
				->groupBy('pe.id')
				->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			return $productExport ? $this->fillDao($productExport) : null;
		});
	}

	/**
	 * @param array $a
	 *
	 * @return Dao\ProductExport[]
	 */
	protected function fillDao($a)
	{
		//$locale = $this->defaultLang->locale;

		$manufacturer = (new Dao\ProductExport())
			->setId($a['id'])
			->setIsZbozi($a['isZbozi'])
			->setIsHeureka($a['isHeureka'])
			->setIsGoogle($a['isGoogle']);

		return $manufacturer;
	}
}
