<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Core\Model\Helpers\BaseService;
use Core\Model\Lang\DefaultLang;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\CategoryExport;
use EshopCatalog\Model\Entities\ManufacturerTexts;
use EshopCatalog\Model\Entities\Variant;
use EshopCatalog\Model\Entities\VariantValue;
use EshopCatalog\Model\Entities\VariantValueText;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use EshopCatalog\Model\Entities\ProductVariantSupplier;
use EshopCatalog\Model\Entities\Supplier;
use EshopCatalog\Model\Entities\VatRate;
use EshopCatalog\Model\Helpers\ExportEnums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Import\AdminModule\Model\DictionaryResolver;
use Kdyby\Doctrine\EntityManager;
use Nette\Utils\FileSystem;
use Nette\Utils\Image as NImage;
use Nette\Utils\Strings;
use Tracy\Debugger;

/**
 * Class ImportHelper
 * @package EshopCatalog\AdminModule\Model\Import
 */
class ImportHelper extends BaseService
{
	/** @var DefaultLang */
	protected $defaultLang;

	/** @var EntityManager */
	protected $em;

	/** @var DictionaryResolver */
	protected $dictionaryResolver;

	/** @var Manufacturer[] */
	protected $cManufacturers;

	/** @var Supplier[] */
	protected $cSuppliers;

	/** @var Category[] */
	protected $cCats = [];

	/** @var VatRate[] */
	protected $cVatRate = [];

	/** @var array */
	protected $cFeatures = [];

	/** @var array */
	protected $cVariants = [];

	/** @var callable[] */
	public $onError = [];

	/** @var ProductVariant[] */
	public $variantsForQuantity = [];

	/** @var array */
	public $featureProducts = [];

	/** @var string */
	public $lang;

	/**
	 * ImportHelper constructor.
	 *
	 * @param EntityManager      $entityManager
	 * @param DictionaryResolver $dictionaryResolver
	 * @param DefaultLang        $defaultLang
	 */
	public function __construct(EntityManager $entityManager, DictionaryResolver $dictionaryResolver, DefaultLang $defaultLang)
	{
		$this->em                 = $entityManager;
		$this->dictionaryResolver = $dictionaryResolver;
		$this->defaultLang        = $defaultLang;
	}

	/**
	 * @param string $ean
	 *
	 * @return Product|object|null
	 */
	public function findByEan($ean)
	{
		return $this->em->getRepository(Product::class)->findOneBy(['ean' => $ean]);
	}

	/**
	 * @param string $value
	 *
	 * @return Manufacturer
	 * @throws \Exception
	 */
	public function getManufacturer($value)
	{
		if ($value == '')
			return null;

		if (!$this->cManufacturers) {
			$this->cManufacturers = $this->em->getRepository(Manufacturer::class)->createQueryBuilder('m')
				->select('m.id, m.name')->getQuery()->getArrayResult();
		}

		// Vrátí výrobce pokud je nalezen
		foreach ($this->cManufacturers as $k => $manufacturer) {
			if (strtolower($manufacturer['name']) == strtolower($value)) {
				if (isset($manufacturer['id']))
					return $this->em->getReference(Manufacturer::class, $manufacturer['id']);
				if (isset($manufacturer['entity']))
					return $manufacturer['entity'];
			}
		}

		$manufacturer       = new Manufacturer();
		$manufacturer->name = $value;
		$this->em->persist($manufacturer)->flush($manufacturer);
		$this->cManufacturers[] = [
			'id'     => $manufacturer->getId(),
			'name'   => $value,
			'entity' => $manufacturer,
		];

		return $manufacturer;
	}

	/**
	 * @param string $value
	 *
	 * @return Supplier
	 * @throws \Exception
	 */
	public function getSupplier($value)
	{
		if (!$this->cSuppliers) {
			$this->cSuppliers = $this->em->getRepository(Supplier::class)->createQueryBuilder('s')
				->select('s.id, s.name')->getQuery()->getArrayResult();
		}

		// Vrátí dodavatele pokud je nalezen
		foreach ($this->cSuppliers as $k => $supplier) {
			if (strtolower($supplier['name']) == strtolower($value)) {
				if ($supplier['id'])
					return $this->em->getReference(Supplier::class, $supplier['id']);
				if (isset($supplier['entity']))
					return $supplier['entity'];
			}
		}

		$supplier = new Supplier($value);
		$this->em->persist($supplier)->flush($supplier);
		$this->cSuppliers[] = [
			'id'     => $supplier->getId(),
			'name'   => $value,
			'entity' => $supplier,
		];

		return $supplier;
	}

	/**
	 * @param $id
	 *
	 * @return Category|object|null
	 */
	public function findCategory($id)
	{
		return $this->em->getRepository(Category::class)->find($id);
	}

	/**
	 * @param $baseCat
	 * @param $deep
	 * @param $lang
	 *
	 * @return |null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function getDeepCategory($baseCat, $deep, $lang)
	{
		if (!is_array($deep))
			$deep = [$deep];

		$parent = $baseCat;
		$cat    = null;

		for ($i = 0; $i < count($deep); $i++) {
			$cat      = null;
			$name     = trim($deep[$i]);
			$cCatsKey = $i . '_' . $name;

			if (isset($this->cCats[$cCatsKey])) {
				return $this->cCats[$cCatsKey];
			} else {
				$found = $this->em->getRepository(CategoryTexts::class)->createQueryBuilder('ct')
					->select('IDENTITY(ct.id) as id')
					->where('ct.name = :name')->andWhere('ct.lang = :lang')
					->setParameters(['name' => $name, 'lang' => $lang])
					->setMaxResults(1)->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);
			}

			if ($found) {
				$cat = $this->em->getReference(Category::class, $found['id']);

				$this->cCats[$cCatsKey] = $cat;
			} else {
				$cat = new Category();
				$cat->setParent($parent);
				$this->em->persist($cat)->flush($cat);

				$ct = new CategoryTexts($cat, $lang);
				$ct->setName($name);
				$this->em->persist($ct)->flush($ct);
				$this->cCats[$cCatsKey] = $cat;

				foreach (ExportEnums::$services as $k => $v) {
					$cp = new CategoryExport($cat, $lang, $k);
					$cp->status = 0;
					$this->em->persist($cp);
					$this->em->flush($cp);
				}
			}
			$parent = $cat;
		}

		return $cat;
	}

	/**
	 * @param Product  $product
	 * @param Category $category
	 *
	 * @return CategoryProduct|object|null
	 */
	public function getCategoryProduct($product, $category)
	{
		$cat = $this->em->getRepository(CategoryProduct::class)->findOneBy([
			'product'  => $product,
			'category' => $category,
		]);
		if (!$cat)
			$cat = new CategoryProduct($product, $category);
		$this->em->persist($cat);

		return $cat;
	}

	/**
	 * @param Product $product
	 *
	 * @return Album
	 */
	public function getGallery($product)
	{
		if (!$product->gallery) {
			$gallery = new Album(UPLOADS_PATH . '/products');
			$this->em->persist($gallery);

			return $gallery;
		}

		return $product->gallery;
	}

	/**
	 * @param Album  $gallery
	 * @param string $url
	 *
	 * @return bool
	 */
	public function addImage($gallery, $url)
	{
		try {
			$images    = $gallery->getImages();
			$basename  = Strings::webalize(pathinfo($url, PATHINFO_FILENAME));
			$extension = pathinfo($basename, PATHINFO_EXTENSION);
			$exist     = false;

			if (!in_array(strtolower($extension), ['jpg', 'jpeg', 'png']))
				$basename = pathinfo($basename, PATHINFO_FILENAME) . '.jpg';

			foreach ($images->toArray() as $img) {
				/** @var Image $img */
				$fn = explode('_', $img->getFilename(), 2);
				if (isset($fn[1])) {
					$fn[1] = str_replace('-jpg', '', $fn[1]);
					if (basename($fn[1]) == $basename) {
						$exist = true;
						break;
					}
				}
			}

			if (!$exist) {
				// TODO dát do služby a nastavit i v imagesZone komponentě
				$image = new Image($gallery, $basename);
				$this->em->persist($image);

				$filename        = $image->getId() . '_' . $basename;
				$image->filename = $filename;
				$this->em->persist($image);
				$gallery->addImage($image);

				$dir = WWW_DIR . $gallery->generatePath();
				FileSystem::createDir($dir);
				copy($url, $dir . '/' . $filename);

				$this->onError[] = function() use ($dir, $filename) {
					FileSystem::delete($dir . '/' . $filename);

					if (count(scandir($dir)) == 2)
						FileSystem::delete($dir);
				};
			}
		} catch (\Exception $e) {
		}
	}

	/**
	 * @param Album[] $galleries
	 */
	public function moveGalleryImage($galleries)
	{
		foreach ($galleries as $gallery) {
			$gallery = $this->em->getRepository(Album::class)->find($gallery->getId());
			$path    = $gallery->generatePath();
			$dir     = WWW_DIR . $path;
			FileSystem::createDir($dir);

			foreach ($gallery->getImages() as $img) {
				$tmp = explode('_', $img->getFilename());

				if ($tmp[0] != '')
					continue;

				$oldFile       = $img->getFile();
				$img->path     = $gallery->generatePath();
				$img->filename = $img->getId() . $img->getFilename();

				rename($oldFile, $img->getFile());
				$this->em->persist($img);
			}
		}

		$this->em->flush();
	}

	/**
	 * @return array
	 */
	public function categoriesList()
	{
		$list = [];

		$qb = $this->em->getRepository(Category::class)->createQueryBuilder('c')
			->select('c.id', 'c.lvl', 'ct.name')
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->where('c.lvl > 0')
			->orderBy('c.root')->addOrderBy('c.lft');

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$deep = '';

			for ($i = 1; $i < $row['lvl']; $i++)
				$deep = '---' . $deep;

			$list[$row['id']] = $deep . ' ' . $row['name'];
		}

		return $list;
	}

	/**
	 * @return array
	 */
	public function vatRateList()
	{
		if (!$this->cVatRate) {
			$qb = $this->em->getRepository(VatRate::class)->createQueryBuilder('vr');

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$keys = [
					$row['rate'],
					$row['rate'] . '.00',
					($row['rate'] / 100),
					($row['rate'] + 100) / 100];

				foreach ($keys as $k)
					$this->cVatRate[str_replace(',', '.', (string) $k)] = $row['id'];
			}
		}

		return $this->cVatRate;
	}

	/**
	 * @param $val
	 *
	 * @return VatRate|null
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function getVatRate($val)
	{
		$list = $this->vatRateList();

		return isset($list[$val]) ? $this->em->getReference(VatRate::class, $list[$val]) : null;
	}

	/**
	 * @return array
	 */
	public function getFeatures()
	{
		if (!$this->cFeatures) {
			$featuresIdName = [];
			foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
				         ->select('f.id, ft.name')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $this->lang)->getQuery()->getArrayResult() as $v) {
				$name                     = strtolower($v['name']);
				$this->cFeatures[$name]   = ['id' => $v['id'], 'values' => []];
				$featuresIdName[$v['id']] = $name;
			}

			foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $this->lang)->getQuery()->getArrayResult() as $v) {
				$name                                                               = strtolower($v['name']);
				$this->cFeatures[$featuresIdName[$v['featureId']]]['values'][$name] = $v['id'];
			}
			unset($featuresIdName);
		}

		return $this->cFeatures;
	}

	/**
	 * @return array
	 */
	public function getVariants()
	{
		if (!$this->cVariants) {
			$variantIdName = [];
			foreach ($this->em->getRepository(Variant::class)->createQueryBuilder('v')
				         ->select('v.id, vt.name')
				         ->join('v.texts', 'vt', 'WITH', 'vt.lang = :lang')
				         ->setParameter('lang', $this->lang)->getQuery()->getArrayResult() as $v) {
				$name                    = strtolower($v['name']);
				$this->cVariants[$name]  = ['id' => $v['id'], 'values' => []];
				$variantIdName[$v['id']] = $name;
			}

			foreach ($this->em->getRepository(VariantValue::class)->createQueryBuilder('vv')
				         ->select('vv.id, vvt.name, IDENTITY(vv.variant) as variantId')
				         ->join('vv.texts', 'vvt', 'WITH', 'vvt.lang = :lang')
				         ->setParameter('lang', $this->lang)->getQuery()->getArrayResult() as $v) {
				$name                                                              = strtolower($v['name']);
				$this->cVariants[$variantIdName[$v['variantId']]]['values'][$name] = $v['id'];
			}
			unset($variantIdName);
		}

		return $this->cVariants;
	}

	/**
	 * @param Product $product
	 * @param array   $params
	 *
	 * @throws
	 */
	public function parametersProccess(&$product, $params)
	{
		$features = $this->getFeatures();
		$variants = $this->getVariants();

		foreach ($params as $key => $value) {
			if (!$value || !$key || is_numeric($key))
				continue;

			if (is_array($value)) {
				$this->parametersProccess($product, $value);
				continue;
			}

			if (mb_strtoupper($key) == $key)
				$key = ucfirst(mb_strtolower($key));

			$key   = $this->dictionaryResolver->findValue($key, $this->lang);
			$value = $this->dictionaryResolver->findValue($value, $this->lang);

			$lKey   = mb_strtolower($key);
			$lValue = mb_strtolower($value);

			if (!$lValue || !$lKey || $lValue == '' || $lKey == '')
				continue;

			if (isset($variants[$lKey])) {
				if (!isset($variants[$lKey]['values'][$lValue])) {
					$variantValue                              = $this->createVariantValue($variants[$lKey]['id'], $lValue);
					$this->cVariants[$lKey]['values'][$lValue] = $variantValue;
					$variants[$lKey]['values'][$lValue]        = $variantValue;
				}

				$this->addVariantValueToProduct($product, $variants[$lKey]['id'], $variants[$lKey]['values'][$lValue]);
			} else {
				if (!isset($features[$lKey])) {
					$feature = $this->createFeature($key);

					if (!$feature)
						continue;

					$this->cFeatures[$lKey]['id'] = $feature;
					$features[$lKey]['id']        = $feature;
				}

				if (!isset($features[$lKey]['values'][$lValue])) {
					$featureValue                              = $this->createFeatureValue($features[$lKey]['id'], $lValue);
					$this->cFeatures[$lKey]['values'][$lValue] = $featureValue;
					$features[$lKey]['values'][$lValue]        = $featureValue;
				}

				$this->addFeatureValueToProduct($product, $features[$lKey]['id'], $features[$lKey]['values'][$lValue]);
			}
		}
	}

	/**
	 * @param string $value
	 *
	 * @return Feature
	 */
	public function createFeature($value)
	{
		$feature = new Feature();
		$feature->setIsPublished(1);
		$feature->setUseAsFilter(0);
		$featureText       = new FeatureTexts($feature, $this->lang);
		$featureText->name = $value;

		$this->em->persist($feature)
			->persist($featureText);

		return $feature;
	}

	/**
	 * @param int|Feature $featureId
	 * @param string      $value
	 *
	 * @return FeatureValue
	 * @throws \Doctrine\ORM\ORMException
	 *
	 */
	public function createFeatureValue($featureId, $value)
	{
		$featureValue = new FeatureValue();
		$featureValue->setFeature(is_numeric($featureId) ? $this->em->getReference(Feature::class, $featureId) : $featureId);
		$featureValueText       = new FeatureValueTexts($featureValue, $this->lang);
		$featureValueText->name = $value;

		$this->em->persist($featureValue)
			->persist($featureValueText);

		return $featureValue;
	}

	/**
	 * @param Product          $product
	 * @param int|Feature      $feature
	 * @param int|FeatureValue $featureValue
	 *
	 * @return |null
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function addFeatureValueToProduct($product, $feature, $featureValue)
	{
		foreach ($product->getFeatureProducts() as $fp) {
			$id = is_numeric($feature) ? $feature : $feature->getId();
			if ($fp->feature->getId() == $id)
				return null;
			$id = is_numeric($featureValue) ? $featureValue : $featureValue->getId();
			if ($fp->featureValue->getId() == $id)
				return null;
		}

		$featureProduct = new FeatureProduct(
			$product,
			is_numeric($feature) ? $this->em->getReference(Feature::class, $feature) : $feature,
			is_numeric($featureValue) ? $this->em->getReference(FeatureValue::class, $featureValue) : $featureValue
		);

		$this->em->persist($featureProduct);

		if ($featureProduct->getFeature()->getIsPublished() === null || $featureProduct->getFeature()->getUseAsFilter() === null)
			$this->em->remove($featureProduct);
		else
			$product->addFeatureProduct($featureProduct);
	}

	/**
	 * @param int|VariantValue $variantId
	 * @param string           $value
	 *
	 * @return VariantValue
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function createVariantValue($variantId, $value)
	{
		$variantValue           = new VariantValue(is_numeric($variantId) ? $this->em->getReference(Variant::class, $variantId) : $variantId);
		$variantValueText       = new VariantValueText($variantValue, $this->lang);
		$variantValueText->name = $value;

		$this->em->persist($variantValue)
			->persist($variantValueText);

		return $variantValue;
	}

	/**
	 * @param Product          $product
	 * @param int|Variant      $variant
	 * @param int|VariantValue $variantValue
	 *
	 * @return |null
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function addVariantValueToProduct($product, $variant, $variantValue)
	{
		$exist = [];
		if ($product->getId()) {
			$tmp = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
				->select('pv.id as vId, IDENTITY(pvpc.variant) as id, IDENTITY(pvpc.value) as value')
				->where('pv.product = :productId')->setParameter('productId', $product->getId())
				->join('pv.productCombinations', 'pvpc')->getQuery()->getArrayResult();
		}

		foreach ($tmp as $t) {
			$a = is_numeric($variant) ? $variant : $variant->getId();
			$v = is_numeric($variantValue) ? $variantValue : $variantValue->getId();

			if ($t['id'] == $a && $t['value'] == $v) {
				$this->variantsForQuantity[] = $this->em->getReference(ProductVariant::class, $t['vId']);

				return null;
			}
		}

		$productVariant            = new ProductVariant($product);
		$productVariantCombination = new ProductVariantCombination(
			$productVariant,
			is_numeric($variantValue) ? $this->em->getReference(VariantValue::class, $variantValue) : $variantValue,
			is_numeric($variant) ? $this->em->getReference(Variant::class, $variant) : $variant
		);

		$this->variantsForQuantity[] = $productVariant;
		$this->em->persist($productVariant)
			->persist($productVariantCombination);
		$product->addProductVariant($productVariant);
	}

	/**
	 * @param Supplier       $supplier
	 * @param ProductVariant $productVariant
	 * @param int            $quantity
	 */
	public function setProductVariantQuantity($supplier, $productVariant, $quantity)
	{
		$entity = $this->em->getRepository(ProductVariantSupplier::class)->findOneBy([
			'supplier'       => $supplier,
			'productVariant' => $productVariant,
		]);

		if (!$entity)
			$entity = new ProductVariantSupplier($supplier, $productVariant);

		$entity->quantity = $quantity;
		$this->em->persist($entity);
	}

	/**
	 * @param Product $product
	 * @param float   $value
	 * @param string  $column
	 */
	public function setProductPrice(Product $product, float $value, string $column = 'price')
	{
		if ($product->getMoreDataValue('stopImportPrice') != true)
			$product->$column = $value;
	}
}
