<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\FrontModule\Model\Categories as FrontCategories;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;

/**
 * Class Categories
 * @package EshopCatalog\AdminModule\Model
 *
 * @method Category|object|null getReference($id)
 * @method Category[]|null getAll()
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseEntityService
{
	use TPublish;

	protected $entityClass = Category::class;

	/** @var CacheService */
	protected $cacheService;

	/** @var ITranslator */
	protected $translator;

	/** @var array */
	protected $cForTree;

	public function __construct(CacheService $cacheService, ITranslator $translator)
	{
		$this->cacheService = $cacheService;
		$this->translator   = $translator;
	}

	public function get($id): ?Category
	{
		return $this->getEr()->createQueryBuilder('c')->addSelect('ct')
			->leftJoin('c.categoryTexts', 'ct')
			->where('c.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param $id
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function removeCategory($id)
	{
		try {
			if ($category = $this->getEr()->find($id)) {
				$ct = $this->em->getRepository(Entities\CategoryTexts::class)->findBy(['id' => $id, 'lang' => 'cs']);
				$this->em->remove($ct);
				$this->em->flush();
				$this->em->remove($category);
				$this->em->flush();

				return true;
			}
		} catch (\Exception $e) {
		}

		return false;
	}

	public function cleanCacheDeep($categoryId)
	{
		$cat = $this->get($categoryId);

		if (!$cat)
			return false;

		$tags = ['category/' . $cat->getId(), 'hierarchy'];
		$loop = function($childs) use (&$loop, &$ids) {
			foreach ($childs as $child) {
				$tags[] = 'category/' . $child->getId();

				if ($child->children->toArray())
					$loop($child->children->toArray());
			}
		};

		$loop($cat->children->toArray());

		$this->cacheService->categoryCache->clean([Cache::TAGS => $tags]);
	}

	protected function prepareDataForTree(): array
	{
		if ($this->cForTree === null) {
			$roots      = [];
			$categories = [];
			$flat       = [];

			foreach ($this->getEr()->createQueryBuilder('c')
				         ->select('c.id, ct.name, c.isPublished, c.lvl, IDENTITY(c.parent) as parent')
				         ->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->groupBy('c.id')
				         ->orderBy('c.root')->addOrderBy('c.lft')->getQuery()->getResult() as $c) {
				if ($c['lvl'] == 0) {
					$roots[$c['id']] = [
						'id'   => $c['id'],
						'name' => $c['name'],
					];
					continue;
				}

				$name = !$c['isPublished'] ? '[x ' . $c['name'] . ' x]' : $c['name'];

				$categories[$c['id']] = $name;
				$flat[]               = [
					'id'     => $c['id'],
					'parent' => $c['parent'],
					'name'   => $name,
				];
			}

			$this->cForTree = [
				'roots'      => $roots,
				'categories' => $categories,
				'flat'       => $flat,
			];
		}

		return $this->cForTree;
	}

	public function getOptionsForSelect(): array { return $this->prepareDataForTree()['categories']; }

	public function getFlatTree(): array { return $this->prepareDataForTree()['flat']; }

	public function getTreeForSelect(): array
	{
		$data = $this->prepareDataForTree();

		$trees = [];
		foreach ($data['roots'] as $id => $root) {
			$trees[$id] = [
				'name'     => $root['name'],
				'children' => Arrays::buildTree($data['flat'], 'parent', 'id', $root['id']),
			];
		}

		return $trees;
	}
}
