<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\Entities\ExtraField;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\Criteria;
use EshopCatalog\AdminModule\Components\Categories\CategoryForm;
use EshopCatalog\AdminModule\Components\Categories\ICategoryFormFactory;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantSupplier;
use EshopCatalog\Model\Entities\VatRate;
use Gallery\Model\Entities\Album;
use Kdyby\Doctrine\QueryBuilder;
use Nette\Utils\Html;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\AdminModule\Model\Products;

/**
 * Class ProductsGrid
 * @package EshopCatalog\AdminModule\Components\Products
 */
class ProductsGrid extends BaseControl
{
	/** @var \EshopCatalog\AdminModule\Model\Products */
	protected $productServices;

	/** @var array */
	protected $quantities;

	/** @var array */
	protected $pvsInStock;

	/** @var array */
	protected $extraFields;

	/** @var array */
	protected $productCategories;

	/** @var Categories */
	protected $categories;

	/** @var ICategoryFormFactory */
	protected $categoryFormFactory;

	public function __construct(Products $products, Categories $categories, ICategoryFormFactory $categoryFormFactory)
	{
		$this->productServices     = $products;
		$this->categories          = $categories;
		$this->categoryFormFactory = $categoryFormFactory;
	}

	public function render()
	{
		$this->template->setFile(__DIR__ . '/ProductsGrid.latte');
		$this->template->render();
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleEditCategory(int $categoryId): void
	{
		$category = $this->categories->get($categoryId);

		if ($category) {
			$this['categoryForm']->categoryId = $categoryId;
			$this->template->modalTitle       = $this->t('eshopCatalog.title.editCategory') . ' ' . $category->getCategoryText()->getName();
			$this->template->modal            = 'categoryForm';
			$this->redrawControl('modal');
		} else {
			$this->getPresenter()->flashMessageDanger('eshopCatalog.category.notFound');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	public function handleDeleteDuplicity()
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->deleteDuplicity())
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.duplicityProductsDeleted');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.duplicityProductsDeleteFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false)
			->setRememberState()
			->setItemsPerPageList([20, 50, 100, 200, 500], false)
			->setDefaultPerPage(50);

		$grid->setTemplateFile(__DIR__ . '/ProductsGrid_grid.latte');

		foreach ($this->em->getRepository(ProductVariantSupplier::class)->createQueryBuilder('pvs')
			         ->select('pvs.quantity, IDENTITY(pv.product) as product')
			         ->join('pvs.productVariant', 'pv')->getQuery()->getArrayResult() as $row) {
			$id = $row['product'];

			if ($row['quantity'] > 0)
				$this->pvsInStock[$id] = 1;
			else if (!isset($this->pvsInStock[$id]))
				$this->pvsInStock[$id] = 0;
		}

		$qb = $this->productServices->getEr()->createQueryBuilder('p')
			->addSelect('pt, cd, cdt, cp, cpc, cpct, gal, imgs, manu, ptag, tags')
			->leftJoin('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->leftJoin('p.gallery', 'gal')
			->leftJoin('gal.images', 'imgs')
			->leftJoin('p.idCategoryDefault', 'cd')
			->leftJoin('cd.categoryTexts', 'cdt', 'WITH', 'cdt.lang = :lang')
			->leftJoin('p.categoryProducts', 'cp')
			->leftJoin('cp.category', 'cpc')
			->leftJoin('cpc.categoryTexts', 'cpct', 'WITH', 'cpct.lang = :lang')
			->leftJoin('p.manufacturer', 'manu')
			->leftJoin('p.productTags', 'ptag')
			->leftJoin('ptag.tag', 'tags')
			->orderBy('p.created', 'desc');
		$grid->setDataSource($qb);

		$grid->getDataSource()->onDataLoaded[] = function($items) {
			$ids      = array_map(function($row) { return $row->getId(); }, $items);
			$variants = [];

			foreach ($items as $row)
				$this->quantities[$row->getId()] = intval($row->quantity);

			if ($ids) {
				foreach ($this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')->select('IDENTITY(ps.product) as id, ps.quantity')
					         ->where('ps.product IN (:ids)')->setParameter('ids', $ids)->getQuery()->getArrayResult() as $v)
					$this->quantities[$v['id']] += intval($v['quantity']);
				foreach ($this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')->select('IDENTITY(pv.product) as id, pv.id as variantId, pv.quantity')
					         ->where('pv.product IN (:ids)')->setParameter('ids', $ids)->getQuery()->getArrayResult() as $v) {
					$this->quantities[$v['id']] += intval($v['quantity']);
					$variants[$v['id']][]       = $v['variantId'];
				}

				foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')->select('ct.name, IDENTITY(cp.product) as prod')
					         ->where('cp.product IN (:prod)')
					         ->join('cp.category', 'c')->join('c.categoryTexts', 'ct', 'with', 'ct.lang = :lang')
					         ->setParameters(['prod' => $ids,
					                          'lang' => $this->translator->getLocale()])->getQuery()->getArrayResult() as $row) {
					$this->productCategories[$row['prod']][] = $row['name'];
				}

				foreach ($this->productCategories as $k => $v)
					$this->productCategories[$k] = implode(', ', $v);
			}

			foreach ($variants as $productId => $vars) {
				foreach ($this->em->getRepository(ProductVariantSupplier::class)->createQueryBuilder('pvs')->select('IDENTITY(pvs.productVariant) as id, pvs.quantity')
					         ->where('pvs.productVariant IN (:ids)')->setParameter('ids', $vars)->getQuery()->getArrayResult() as $v)
					$this->quantities[$productId] += intval($v['quantity']);
			}

			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				         ->where('ef.sectionName = :secName')->setParameter('secName', Product::EXTRA_FIELD_SECTION)
				         ->getQuery()->getArrayResult() as $row)
				$this->extraFields[$row['sectionKey']][$row['key']] = $row['value'];
		};

		/**
		 * === Columns
		 */
		$grid->addColumnText('image', 'eshopCatalog.defaultGrid.image', '')->setRenderer(function($row) {
			if (!$row->gallery->images || !$row->gallery->images->first())
				return '';

			// TODO dalo by se předělat do nové ColumnImagePreview
			return Html::el('')
				->addHtml(Html::el('img', [
					'onMouseOver' => "showPicture('spt-{$row->getId()}', 1)",
					'onMouseOut'  => "showPicture('spt-{$row->getId()}', 0)",
					'src'         => $this->imagePipe->request($row->gallery->images->first()->getFilePath(), '50x40'),
				]))
				->addHtml(Html::el('img', [
					'id'    => 'spt-' . $row->getId(),
					'class' => 'show-picture-target',
					'src'   => $this->imagePipe->request($row->gallery->images->first()->getFilePath(), '400x400'),
				]));
		})->setAlign('center');
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(function(Product $row) {
			$link = Html::el('a', [
				'href' => $this->getPresenter()->link('Products:edit', [$row->getId()]),
			])->setText($row->getProductText()->name);

			$manufacturer = Html::el('div', [
				'class' => 'text-secondary',
			])->setText($this->t('eshopCatalog.defaultGrid.manufacturer') . ': ' . $row->getManufacturer()->name);

			return Html::el()->addHtml($link)->addHtml($manufacturer);
		});
		$grid->addColumnText('ean', 'eshopCatalog.defaultGrid.ean');
		$grid->addColumnText('code1', 'eshopCatalog.defaultGrid.code1');
		$grid->addColumnLink('defaultCategory', 'eshopCatalog.productGrid.defaultCategory', 'editCategory!', 'idCategoryDefault.categoryText.name',
			['categoryId' => 'idCategoryDefault.id'])->setClass('ajax');
		$grid->addColumnText('categories', 'eshopCatalog.productGrid.categories')->setRenderer(function($row) {
			return $this->productCategories[$row->getId()];
		});
		$grid->addColumnText('tags', 'eshopCatalog.productGrid.tags')->setRenderer(function($row) {
			$arr = [];
			foreach ($row->productTags->toArray() as $pt)
				$arr[] = $pt->tag->type;

			return implode(', ', $arr);
		});
		$grid->addColumnPrice('price', 'eshopCatalog.productGrid.price');
		$grid->addColumnNumber('quantity', 'eshopCatalog.productGrid.quantity')->setRenderer(function(Product $row) {
			return Config::load('pseudoWarehouse') && $row->unlimitedQuantity === 1 ? '∞' : $this->quantities[$row->getId()];
		});
		$grid->addColumnStatus('isPublished', 'eshopCatalog.productGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		/**
		 * === Filters
		 */
		$grid->getColumn('name')->setFilterText()->setCondition([$this, 'conditionGridName']);
		$grid->getColumn('ean')->setFilterText();
		$grid->getColumn('code1')->setFilterText();
		$grid->addFilterText('manufacturer', 'eshopCatalog.defaultGrid.manufacturer', 'manufacturer.name')
			->setCondition([$this, 'conditionGridManufacturer']);
		$grid->addFilterCheckboxNestedList('categoriesId', 'eshopCatalog.productGrid.defaultCategory', $this->categories->getFlatTree())
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('p.idCategoryDefault IN (:categoriesId) OR cpct.id IN (:categoriesId)')->setParameter('categoriesId', $value);
			});
		$grid->getColumn('defaultCategory')->setFilterText()->setCondition([$this, 'conditionGridCategory']);
		$grid->getColumn('categories')->setFilterText()->setCondition([$this, 'conditionGridCategories']);
		$grid->getColumn('tags')->setFilterText()->setCondition([$this, 'conditionGridTags']);
		$grid->addFilterSelect('inStock', 'eshopCatalog.defaultGrid.isInStock', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		])->setCondition([$this, 'conditionGridQuantity']);
		$grid->getColumn('quantity')->setFilterRange()->setCondition([$this, 'conditionGridQuantity']);
		$grid->getColumn('price')->setFilterRange()->setCondition(function(QueryBuilder $qb, $values) {
			if ($values->from)
				$qb->andWhere('p.price >= :priceFrom')->setParameter('priceFrom', $values->from);
			if ($values->to)
				$qb->andWhere('p.price <= :priceTo')->setParameter('priceTo', $values->to);
		});
		$grid->addFilterSelect('canApplyDiscount', 'eshopCatalog.productForm.discountDisabled', [
			'' => '',
			0  => $this->t('default.yes'),
			1  => $this->t('default.no'),
		], 'p.discountDisabled');
		if (Config::load('pseudoWarehouse')) {
			$grid->addFilterSelect('unlimitedQuantity', 'eshopCatalog.productForm.unlimitedQuantity', [
				'' => '',
				1  => $this->t('default.yes'),
				0  => $this->t('default.no'),
			], 'p.unlimitedQuantity');
		}
		$grid->getColumn('isPublished')->setFilterSelect([
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		], 'p.isPublished');

		$manufacturers = [];
		foreach ($this->em->getRepository(Manufacturer::class)->createQueryBuilder('m')
			         ->select('m.id, m.name')->orderBy('m.position')->getQuery()->getArrayResult() as $m)
			$manufacturers[$m['id']] = $m['name'];

		$vatRates = [];
		foreach ($this->em->getRepository(VatRate::class)->createQueryBuilder('vr')
			         ->select('vr.id, vr.name')->getQuery()->getArrayResult() as $vr)
			$vatRates[$vr['id']] = $vr['name'];

		/**
		 * === Action
		 */
		$grid->addAction('edit', '', 'Products:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		$grid->addGroupAction('eshopCatalog.productGrid.editCategories')
			->onSelect[] = [$this, 'gridEditCategories'];
		$grid->addGroupAction('eshopCatalog.productGrid.editVariants')
			->onSelect[] = [$this, 'gridEditVariants'];
		$grid->addGroupAction('eshopCatalog.productGrid.editFeatures')
			->onSelect[] = [$this, 'gridEditFeatures'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.editManufacturer', $manufacturers)
			->onSelect[] = [$this, 'gridEditManufacturer'];
		$grid->addGroupAction('eshopCatalog.productGrid.changePublished')
			->onSelect[] = [$this, 'gridChangePublished'];
		$grid->addGroupAction('eshopCatalog.productGrid.changeUnpublished')
			->onSelect[] = [$this, 'gridChangeUnpublished'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.changeVatRate', $vatRates)
			->onSelect[] = [$this, 'gridChangeVatRate'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.discountDisabled', [
			1 => 'default.yes',
			0 => 'default.no',
		])->onSelect[]   = [$this, 'gridDiscountDisabled'];

		// Columns prototype
		$imageTdElement          = $grid->getColumn('image')->getElementPrototype('td');
		$imageTdElement->class[] = 'w1';
		$imageTdElement->style   = 'padding: 0;';
		$grid->getColumn('price')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentCategoryForm(): CategoryForm
	{
		$control = $this->categoryFormFactory->create();

		if ($this->getParameter('categoryId'))
			$control->setCategory($this->getParameter('categoryId'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this['grid']->reload();
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this['grid']->reload();
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->productServices->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridCategories(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('cpct'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('cpct.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridTags(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('tags'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('tags.type', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridName(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		foreach (explode(',', $value) as $val)
			$criteria->orWhere(Criteria::expr()->contains('pt.name', trim($val)));

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridQuantity(QueryBuilder $qb, $value)
	{
		$sign = null;

		if (is_numeric($value))
			$sign = $value == 1 ? '>' : '<=';

		$arr = [];

		if (!in_array('ps', $qb->getAllAliases()))
			$qb->leftJoin('p.suppliers', 'ps');
		if (!in_array('pv', $qb->getAllAliases()))
			$qb->leftJoin('p.productVariants', 'pv');

		if ($sign) { // Ano/ne
			$arr[] = '(p.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR p.quantity IS NULL)');
			$arr[] = '(ps.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR ps.quantity IS NULL)');
			$arr[] = '(pv.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR pv.quantity IS NULL)');

			$pvsKey = $value == 1 ? 'yes' : 'no';
			$pvs    = [];
			foreach ($this->pvsInStock as $id => $stock)
				$pvs[$stock == 1 ? 'yes' : 'no'][] = $id;

			$where = implode($value == 1 ? ' OR ' : ' AND ', $arr);

			if (Config::load('pseudoWarehouse')) {
				$where .= ' OR p.unlimitedQuantity = 1';
			}
			if (isset($pvs[$pvsKey]))
				$where .= ' OR p.id IN (' . implode(',', $pvs[$pvsKey]) . ')';

			$qb->andWhere($where);
		} else { // Od do
			if ($value->from)
				$qb->andWhere('(ifnull(p.quantity, 0) + ifnull(ps.quantity, 0) + ifnull(pv.quantity, 0) >= :quantityFrom)')->setParameter('quantityFrom', $value->from);

			if ($value->to)
				$qb->andWhere('(ifnull(p.quantity, 0) + ifnull(ps.quantity, 0) + ifnull(pv.quantity, 0) <= :quantityTo)')->setParameter('quantityTo', $value->to);
		}
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridManufacturer(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('manu'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('manu.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridCategory(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('cdt'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('cdt.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditCategories(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:editCategories', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditVariants(array $ids)
	{
		$this->getPresenter()->redirect('Products:editVariants', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditFeatures(array $ids)
	{
		$this->getPresenter()->redirect('Products:editFeatures', implode('-', $ids));
	}

	/**
	 * @param array  $ids
	 * @param string $manufacturer
	 */
	public function gridEditManufacturer(array $ids, $manufacturer)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setManufacturer($ids, $manufacturer)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.manufacturerChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.manufacturerChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array  $ids
	 * @param string $vatRate
	 */
	public function gridChangeVatRate(array $ids, $vatRate)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setVatRate($ids, $vatRate)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.vatRateChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.vatRateChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridChangePublished(array $ids)
	{
		$presenter = $this->getPresenter();

		foreach ($ids as $id) {
			if (!$this->productServices->setPublish($id, 1)) {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
				break;
			}
		}

		$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridChangeUnpublished(array $ids)
	{
		$presenter = $this->getPresenter();


		foreach ($ids as $id) {
			if (!$this->productServices->setPublish($id, 0)) {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
				break;
			}
		}

		$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array $ids
	 * @param int   $v
	 */
	public function gridDiscountDisabled(array $ids, $v)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setDiscountDisabled($ids, $v))
			$presenter->flashMessageSuccess('default.saved');
		else
			$presenter->flashMessageDanger('default.error');

		$presenter->redrawControl('flashes');
	}
}
