<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Event\AfterFillDaoProductEvent;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use EshopCatalog\Model\Entities\ProductVariantSupplier;
use EshopOrders\Model\Entities\Customer;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Users\Model\Http\UserStorage;

/**
 * TODO možnost přesunutí do fasády
 *
 * Class Products
 * @package EshopCatalog\FrontModule\Model
 */
class Products extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'eshopCatalogProducts';

	protected $entityClass = Product::class;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var Albums @inject */
	public $albumsService;

	/** @var Manufacturers @inject */
	public $manufacturersService;

	/** @var ProductVariants */
	private $productVariantsSevice;

	/** @var FeatureProducts */
	private $featureProductsService;

	/** @var CacheService */
	protected $cacheService;

	/** @var UserStorage */
	protected $userStorage;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var Config */
	protected $config;

	/** @var callable[] */
	public $customQB = [];

	/** @var array */
	protected $cProductsIdInCategory = [], $cSearch, $cProductsIdBySearch = [], $cProductsIdAll = [], $cGroupCustomer = [],
		$cPrices = [], $cExtraFields, $cQuantity = [];

	/** @var callable[] */
	public $onAfterProductsGet = [];

	/** @var array */
	public $cacheDep = [
		Cache::TAGS   => ['products'],
		Cache::EXPIRE => '15 minutes',
	];

	public function __construct(ProductVariants $productVariants, FeatureProducts $featurePorducts, CacheService $cacheService,
	                            UserStorage $userStorage, Config $config, EventDispatcher $eventDispatcher)
	{
		$this->productVariantsSevice  = $productVariants;
		$this->featureProductsService = $featurePorducts;
		$this->cacheService           = $cacheService;
		$this->userStorage            = $userStorage;
		$this->config                 = $config;
		$this->eventDispatcher        = $eventDispatcher;

		$this->afterConstruct();
		$this->onAfterProductsGet[] = function($products) {
			/** @var Dao\Product[] $products */
			$this->loadPrices($products,
				$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);
		};
	}

	protected function afterConstruct() { }

	/**
	 * @param int|array $id
	 *
	 * @return array|mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getForNavigation($id)
	{
		$qb = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name')
			->andWhere(is_array($id) ? 'p.id IN (:id)' : 'p.id = :id')->setParameter('id', $id)
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->getQuery();

		return is_array($id) ? $qb->getArrayResult() : $qb->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}

	/**
	 * @param int $categoryId
	 *
	 * @return int[]
	 */
	public function getProductsIdInCategory($categoryId, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$lang    = $this->translator->getLocale();
		$keyBase = md5(serialize([$categoryId, $sort, $filter])) . '/' . $lang;

		if (isset($this->cProductsIdInCategory[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

			return $this->cProductsIdInCategory[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($categoryId, $sort, $filter, $lang) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'listing/' . $categoryId;
				$dep[Cache::TAGS][] = 'listing';
				$dep[Cache::EXPIRE] = '5 minutes';

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($lang))
					->withTexts()
					->inCategory($categoryId)
					->selectIds();

				if (Config::load('onlyStockProductsInList', true) == true)
					$query->onlyInStockOrSupplier();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdInCategory[$keyBase] = array_unique(array_merge($this->getRecommendedIds($categoryId, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdInCategory[$keyBase] = $ids;
		}

		if ($limit)
			return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

		return $this->cProductsIdInCategory[$keyBase];
	}

	/**
	 * @param int        $categoryId
	 * @param array|null $filter
	 *
	 * @return int
	 */
	public function getProductsInCategoryCount($categoryId, $filter = [])
	{
		return count($this->getProductsIdInCategory($categoryId, 0, null, null, $filter));
	}

	/**
	 * @param       $categoryId
	 * @param int   $start
	 * @param null  $limit
	 * @param null  $sort
	 * @param array $filter
	 * @param array $productParams
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getProductsInCategory($categoryId, $start = 0, $limit = null, $sort = null, $filter = [], $productParams = [])
	{
		$products = [];
		foreach ($this->getProductsIdInCategory($categoryId, $start, $limit, $sort, $filter) as $id) {
			$product = $this->get($id, $productParams ?: ['loadPrice' => false]);
			if ($product)
				$products[$id] = $product;
		}

		foreach ($this->onAfterProductsGet as $c)
			$c($products);

		return $products;
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdBySearch($q, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$q, $sort, $filter]));

		if (isset($this->cProductsIdBySearch[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

			return $this->cProductsIdBySearch[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($q, $start, $limit, $sort, $filter) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'listing/search';
				$dep[Cache::EXPIRE] = '5 minutes';

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->search($q)
					->selectIds()
					->selectInStockBool();

				$emptySort = is_null($sort);
				if (is_string($sort)) {
					$tmp  = explode(' ', $sort, 2);
					$sort = ['inStockBool' => 'DESC'] + [$tmp[0] => ($tmp[1] ?? 'ASC')];
				} else if (is_array($sort))
					$sort = ['inStockBool' => 'DESC'] + $sort;
				else
					$sort = ['inStockBool' => 'DESC', 'p.price' => 'ASC'];

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdBySearch[$keyBase] = array_unique(array_merge($this->getRecommendedIds($q, 0, null, null, $filter, 'search'), $ids));
			} else
				$this->cProductsIdBySearch[$keyBase] = $ids;
		}

		if ($limit)
			return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

		return $this->cProductsIdBySearch[$keyBase];
	}

	/**
	 * @param int        $categoryId
	 * @param array|null $filter
	 *
	 * @return int
	 */
	public function getProductsBySearchCount($q, $filter = [])
	{
		$key = 'searchCount' . '_' . md5(serialize([$q, $filter]));

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($categoryId, $filter) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'listing/search';
			$dep[Cache::EXPIRE] = '5 minutes';

			$query = (new ProductQuery($this->defaultLang->locale))
				->search($categoryId)
				->selectIds();

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			return count($qb->getQuery()->getArrayResult());
		});
	}

	/**
	 * @param      $q
	 * @param int  $start
	 * @param null $limit
	 * @param null $sort
	 * @param null $filter
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getProductsBySearch($q, $start = 0, $limit = null, $sort = null, $filter = null)
	{
		$products = [];
		foreach ($this->getProductsIdBySearch($q, $start, $limit, $sort, $filter) as $id) {
			$product = $this->get($id, ['loadPrice' => false]);
			if ($product)
				$products[$id] = $product;
		}

		foreach ($this->onAfterProductsGet as $c)
			$c($products);

		return $products;
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$sort, $filter]));

		if (isset($this->cProductsIdAll[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

			return $this->cProductsIdAll[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($sort, $filter) {
				$dep = $this->cacheDep;

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->selectIds();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdAll[$keyBase] = array_unique(array_merge($this->getRecommendedIds(null, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdAll[$keyBase] = $ids;;
		}

		if ($limit)
			return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

		return $this->cProductsIdAll[$key];
	}

	/**
	 * @param int   $start
	 * @param null  $limit
	 * @param null  $sort
	 * @param array $filter
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getProductsAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$products = [];
		foreach ($this->getProductsIdAll($start, $limit, $sort, $filter) as $id) {
			$product       = $this->get($id, ['loadPrice' => false]);
			$products[$id] = $product;
		}

		$this->loadQuantity($products);

		foreach ($this->onAfterProductsGet as $c)
			$c($products);

		return $products;
	}

	/**
	 * @param      $ids
	 * @param int  $start
	 * @param null $limit
	 * @param null $sort
	 * @param null $filter
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getProductsByIds($ids, $start = 0, $limit = null, $sort = null, $filter = null)
	{
		$products = [];

		$ids = array_slice($ids, $start, $limit);

		foreach ($ids as $id) {
			$product = $this->get($id, ['loadPrice' => false]);
			if ($product)
				$products[$id] = $product;
		}

		foreach ($this->onAfterProductsGet as $c)
			$c($products);

		return $products;
	}

	public function getRecommendedIds($q = null, $start = 0, $limit = null, $sort = null, $filter = [], $type = 'category')
	{
		$keyBase = md5('recommended_' . serialize([$q, $sort, $filter, $type]));
		$key     = "$keyBase-$start-$limit";

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($q, $start, $limit, $sort, $filter, $type) {
			$dep                = $this->cacheDep;
			$dep[Cache::EXPIRE] = '5 minutes';

			if (isset($filter['manu']))
				$dep[Cache::TAGS][] = 'manufacturers';
			if (isset($filter['features']))
				$dep[Cache::TAGS][] = 'features';

			$query = (new ProductQuery($this->defaultLang->locale))
				->withTag('tip')
				->selectIds();

			if (Config::load('onlyStockProductsInList', true) == true)
				$query->onlyInStockOrSupplier();

			switch ($type) {
				case 'category':
					$dep[Cache::TAGS][] = 'listing/' . $q;
					if ($q)
						$query->inCategory($q);
					break;
				case 'search':
					$dep[Cache::TAGS][] = 'listing/search';
					$query->search($q);
					break;
			}

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));
			$sort = 'p.price ASC';
			$this->prepareSort($query, $sort);

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$ids = [];
			foreach ($qb->getQuery()->setMaxResults($limit)->setFirstResult($start)->getScalarResult() as $v)
				$ids[] = $v['id'];

			return $ids;
		});
	}

	/**
	 * @param       $id
	 * @param array $params
	 *
	 * @return Dao\Product|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 *
	 * @deprecated Použijte getProducts nebo getProduct z productsFacade
	 *
	 */
	public function get($id, $params = [])
	{
		$locale = $params['lang'] ?? $this->translator->getLocale();
		$key    = 'product_' . $locale . '_' . $id;

		/** @var Dao\Product $product */
		$product = $this->cacheService->productCache->load($key, function(&$dep) use ($id, $locale) {
			$dep                 = $this->cacheDep;
			$dep[Cache::TAGS][]  = 'product/' . $id;
			$dep[Cache::EXPIRE]  = '1 month';
			$dep[Cache::SLIDING] = true;

			$query = (new ProductQuery($locale))
				->withCategories()
				->withTexts()
				->withVatRate()
				->byId($id);

			$qb = $query->getQueryBuilder($this->getEr());
			$qb->groupBy('p.id');

			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$result = $qb->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			if ($result) {
				$result                = $this->normalizeHydrateArray($result);
				$result['extraFields'] = $this->getAllExtraFields()[$result['id']] ?? [];
				$dao                   = $this->fillDao($result);

				if ($this->linkGenerator)
					$dao->setLink($this->linkGenerator->link('EshopCatalog:Front:Default:product', [
						'id'     => $dao->getId(),
						'locale' => $locale,
					]));

				return $dao;
			}

			return null;
		});

		if ($product) {
			if ($product->galleryId)
				$product->setGallery($this->albumsService->get($product->galleryId));

			if ($product->manufacturerId)
				$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));

			if (isset($params['loadPrice']) && $params['loadPrice'] == true)
				$this->loadPrice($product,
					$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);

			$this->eventDispatcher->dispatch(new AfterFillDaoProductEvent($product), Products::class . '::afterFillDao');

			return $product;
		}

		return null;
	}

	/**
	 * @param Dao\Product $product
	 */
	public function loadVariants(Dao\Product &$product)
	{
		$product->setVariants($this->productVariantsSevice->getProductVariantsForProduct($product->getId()));
	}

	/**
	 * @param $product
	 * @param $userId
	 *
	 * @return mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadPrice(&$product, $userId = null)
	{
		$products = [$product->getId() => $product];
		$this->loadPrices($products, $userId);

		return $products[$product->getId()];
	}

	/**
	 * @param $products
	 * @param $userId
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadPrices(&$products, $userId = null)
	{
		$ids = array_map(function($p) { return $p->getId(); }, $products);
		foreach ($this->getEr()->createQueryBuilder('p')->select('p.id, p.price, p.retailPrice')
			         ->andWhere('p.id IN (:ids)')->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $row) {
			$products[$row['id']]->setPrice((float) $row['price']);
			$products[$row['id']]->setRetailPrice((float) $row['retailPrice']);
		}

		if (!$userId || !$this->config->get('enablePriceLevels', false))
			return;

		if (!isset($this->cGroupCustomer[$userId])) {
			$this->cGroupCustomer[$userId] = $this->em->getRepository(Customer::class)->createQueryBuilder('c')->select('IDENTITY(c.groupCustomers) as group')
				->where('c.user = :user')->setParameter('user', $userId)->getQuery()->useResultCache(true, 60)->getOneOrNullResult();
		}
		$group = $this->cGroupCustomer[$userId] ?? null;

		if (!$group)
			return;

		$group  = $group['group'];
		$ids    = [];
		$prices = [];
		foreach ($products as $product) {
			if (isset($this->cPrices[$product->getId()])) {
				$prices[$product->getId()] = $this->cPrices[$product->getId()];
			} else {
				$ids[] = $product->getId();
			}
		}

		if ($ids) {
			$qb = $this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('ppl')
				->select('IDENTITY(ppl.productId) as product, ppl.price');

			if (count($ids) == 1)
				$qb->where('ppl.productId = :ids');
			else
				$qb->where('ppl.productId IN (:ids)');
			$qb->andWhere('ppl.groupId = :groupId')
				->setParameters(['ids' => $ids, 'groupId' => $group]);

			foreach ($qb->getQuery()->useResultCache(true, 60)->getArrayResult() as $row) {
				$this->cPrices[$row['product']] = $row['price'];
				$prices[$row['product']]        = $row['price'];
			}
		}

		foreach ($prices as $product => $price) {
			$products[$product]->retailPrice = $products[$product]->price;
			$products[$product]->price       = $price;
		}
	}

	/**
	 * @param int $idVariant
	 */
	public function getVariant(int $idVariant)
	{
		return $this->productVariantsSevice->get($idVariant);
	}

	/**
	 * @param Dao\Product[] $product
	 */
	public function loadFeatures(&$products)
	{
		if ($products instanceof Dao\Product)
			$products->setFeatures($this->featureProductsService->getFeaturesForProduct($products->getId()));
		else {
			foreach ($products as $k => $product)
				$products[$k]->setFeatures($this->featureProductsService->getFeaturesForProduct($product->getId()));
		}
	}

	/**
	 * @param Dao\Product $product
	 * @param             $limit
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadAlternative(Dao\Product &$product, $limit)
	{
		$data = $this->getProductsIdInCategory($product->defaultCategory);
		shuffle($data);

		$products = [];
		foreach (array_slice($data, 0, $limit) as $id) {
			if ($id == $product->getId())
				continue;
			$products[$id] = $this->get($id, ['loadPrice' => false]);
		}

		$this->loadQuantity($products);
		$this->loadPrices($products,
			$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);

		$product->setAlternatives($products);
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadQuantity(&$product)
	{
		if (is_array($product)) {
			foreach ($product as $p) {
				if (isset($this->cQuantity[$p->getId()]))
					$product[$p->getId()]->setQuantity($this->cQuantity[$p->getId()]);
				else
					$id[] = $p->getId();
			}
			$id = array_map(function($row) { return $row->getId(); }, $product);
		} else {
			if (isset($this->cQuantity[$product->getId()]))
				$product->setQuantity($this->cQuantity[$product->getId()]);
			else
				$id = $product->getId();
		}

		if (!isset($id))
			return;

		$query = (new ProductQuery($this->defaultLang->locale))
			->byId($id)
			->selectQuantity();

		$data = $query->getQueryBuilder($this->getEr())->groupBy('p.id')->getQuery()->getArrayResult();

		if (is_array($product)) {
			foreach ($data as $id => $row) {
				$q                    = intval($row['productQ']) + intval($row['suppliersQ']) + intval($row['variantsQ']) + intval($row['variantSuppliersQ']);
				$this->cQuantity[$id] = $q;
				$product[$id]->setQuantity($q);
			}
		} else if (isset($data[$id])) {
			$q                    = intval($data[$id]['productQ']) + intval($data[$id]['suppliersQ']) + intval($data[$id]['variantsQ']) + intval($data[$id]['variantSuppliersQ']);
			$this->cQuantity[$id] = $q;
			$product->setQuantity($q);
		}
	}

	/**
	 * @param array $product
	 *
	 * @return Dao\Product
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function fillDao($product): Dao\Product
	{
		$p = new Dao\Product($product['id']);
		$p->setQuantity((int) $product['quantity']);
		$p->setPrice((double) $product['price']);
		$p->setBasePrice((double) $product['price']);
		$p->setRetailPrice((double) $product['retailPrice']);
		$p->setName($product['productTexts']['name']);
		$p->setName2($product['productTexts']['name2']);
		$p->setShortDescription($product['productTexts']['shortDescription']);
		$p->setDescription($product['productTexts']['description']);
		$p->setSeo($product['productTexts']['seo']);
		$p->setCreated($product['created']);
		$p->setModified($product['modified']);
		$p->setDefaultCategory((int) $product['defaultCategory']);
		$p->setCategories($product['categories']);
		$p->setVatRate($product['vatRate']);
		$p->setEan($product['ean']);
		$p->setCode1($product['code1']);
		$p->setCode2($product['code2']);
		$p->setExtraFields($product['extraFields']);
		$p->setUnlimitedQuantity($product['unlimitedQuantity'] ? 1 : 0);

		$p->discountDisabled     = $product['discountDisabled'] ? true : false;
		$p->categoryGiftsAllowed = $product['categoryGiftsAllowed'] ? true : false;

		if ($product['gallery'])
			$p->setGalleryId($product['gallery']);

		if ($product['manufacturer'])
			$p->setManufacturerId($product['manufacturer']);

		return $p;
	}

	/**
	 * @param array $product
	 *
	 * @return array
	 */
	public function normalizeHydrateArray($product)
	{
		$result                    = $product[0];
		$result['gallery']         = $product['gallery'];
		$result['manufacturer']    = $product['manufacturer'];
		$result['vatRate']         = $product['vatRate'];
		$result['productTexts']    = $product[0]['productTexts'][$this->defaultLang->locale];
		$result['defaultCategory'] = $product['defaultCategory'];
		$result['categories']      = array_map(function($r) { return $r['id_category']; }, $product[0]['categoryProducts']);
		unset($result['categoryProducts']);

		return $result;
	}

	protected function prepareFilter($filter)
	{
		if (isset($filter['variants'])) {
			$vals = [];
			foreach ($filter['variants'] as $g => $v1) {
				$vals = array_merge($vals, $v1);
			}

			$tmp      = [];
			$variants = [];
			foreach ($this->em->getRepository(ProductVariantCombination::class)->createQueryBuilder('pvc')
				         ->addSelect('IDENTITY(pvc.variant) as variantId')
				         ->where('IDENTITY(pvc.value) IN (:variants)')->setParameter('variants', $vals)->getQuery()->getArrayResult() as $row) {
				$variants[$row['id_product_variant']][$row['variantId']][] = $row['id_variant_value'];
			}

			foreach ($variants as $variant => $groups) {
				foreach ($filter['variants'] as $selGroup => $selValues) {
					if (!isset($groups[$selGroup])) {
						unset($variants[$variant]);
						break;
					}

					$isIn = false;
					foreach ($selValues as $val) {
						if (in_array($val, $selValues)) {
							$isIn = true;
						}
					}

					if (!$isIn) {
						unset($variants[$variant]);
						break;
					}
				}
			}

			$filter['variants'] = $variants ? array_keys($variants) : [];
		}

		if (isset($filter['onlyStock']) && $filter['onlyStock'] === true) {
			$pvs = [];
			foreach ($this->em->getRepository(ProductVariantSupplier::class)->createQueryBuilder('pvs')
				         ->select('pvs.quantity, IDENTITY(pv.product) as product')
				         ->andWhere('pvs.quantity > 0')
				         ->join('pvs.productVariant', 'pv')->getQuery()->useResultCache(true, 60)->getArrayResult() as $row)
				$pvs[] = $row['product'];

			$filter['onlyStockPVS'] = $pvs;
		}

		if (isset($filter['features'])) {
			$count = count($filter['features']);
			$ids   = [];
			$qb    = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')->select('IDENTITY(fp.product) as product');
			$tmp   = [];

			foreach ($filter['features'] as $group => $vals) {
				$or  = [];
				$qb2 = clone $qb;
				foreach ($vals as $k => $v) {
					$or[] = "IDENTITY(fp.featureValue) = :fpfv_{$i}_{$k}";
					$qb2->setParameter("fpfv_{$i}_{$k}", $v);
				}
				$qb2->andWhere(implode(' OR ', $or));
				foreach ($qb2->getQuery()->useResultCache(true, 60)->getArrayResult() as $row)
					$tmp[$row['product']] = isset($tmp[$row['product']]) ? $tmp[$row['product']] + 1 : 1;
			}

			foreach ($tmp as $k => $v) {
				if ($v == $count)
					$ids[] = $k;
			}
			$tmp = null;

			$filter['features'] = empty($ids) ? 'empty' : $ids;
		}

		return $filter;
	}

	public function prepareSort(ProductQuery &$query, $sort)
	{
		if ($sort == 'recommended' || is_null($sort)) {
			$sort = 'p.price ASC';
		}

		if (is_string($sort)) {
			$sort = explode(' ', $sort, 2);
			$query->sortBy($sort[0], $sort[1] ?? null);
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v)
				$query->addSortBy($k, $v);
		}
	}

	public function getAllExtraFields(): array
	{
		if ($this->cExtraFields === null) {
			$this->cExtraFields = [];
			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				         ->select('ef.sectionKey, ef.key, ef.value')
				         ->where('ef.sectionName = :secName')
				         ->setParameter('secName', Product::EXTRA_FIELD_SECTION)->getQuery()->getScalarResult() as $row)
				$this->cExtraFields[$row['sectionKey']][$row['key']] = $row['value'];
		}

		return $this->cExtraFields;
	}
}
