<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Seller;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use Nette\Utils\Html;

class SellersGrid extends BaseControl
{
	/** @var Sellers */
	protected $sellersService;

	/** @var ISellerFormFactory */
	protected $sellerFormFactory;

	public function __construct(Sellers $sellersService, ISellerFormFactory $formFactory)
	{
		$this->sellersService    = $sellersService;
		$this->sellerFormFactory = $formFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd()
	{
		$this->template->modalTitle       = $this->t('eshopCatalog.title.newSeller');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-md';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setSeller($id);
		$this->template->modalTitle = $this->t('eshopCatalog.title.editSeller',
			['title' => $this['form']->getSeller()->title]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->getPresenter();

		if ($this->sellersService->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->sellersService->getEr()->createQueryBuilder('s')
			->addSelect('sites')
			->leftJoin('s.sites', 'sites');
		$grid->setDataSource($qb);

		$grid->addColumnText('title', 'eshopCatalog.seller.title');
		$grid->addColumnText('ic', 'eshopCatalog.seller.ic');
		$grid->addColumnText('dic', 'eshopCatalog.seller.dic');
		$grid->addColumnEmail('email', 'eshopCatalog.seller.email');
		$grid->addColumnPhone('phone', 'eshopCatalog.seller.phone');
		$grid->addColumnText('address', 'eshopCatalog.seller.address');
		$grid->addColumnText('city', 'eshopCatalog.seller.city');
		$grid->addColumnText('postal', 'eshopCatalog.seller.postal');
		$grid->addColumnText('country', 'eshopCatalog.seller.country');
		$grid->addColumnText('sites', 'eshopCatalog.seller.eshops')->setRenderer(function(Seller $row) {
			return Html::el()->setHtml(implode('<br>', $row->getSitesIdent()));
		});

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): SellerForm
	{
		$control = $this->sellerFormFactory->create();
		if ($this->getParameter('id'))
			$control->setSeller((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
