<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Repository\RelatedGroups;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\RelatedProduct;
use Nette\Localization\ITranslator;

class RelatedProductContainer
{
	/** @var ITranslator */
	protected ITranslator $translator;

	/** @var EntityManagerDecorator */
	protected EntityManagerDecorator $em;

	/** @var Products */
	protected Products $products;

	/** @var RelatedGroups */
	protected RelatedGroups $relatedGroups;

	/**
	 * RelatedProductContainer constructor.
	 *
	 * @param ITranslator            $translator
	 * @param EntityManagerDecorator $em
	 * @param Products               $products
	 * @param RelatedGroups          $relatedGroups
	 */
	public function __construct(ITranslator $translator, EntityManagerDecorator $em, Products $products, RelatedGroups $relatedGroups)
	{
		$this->translator    = $translator;
		$this->em            = $em;
		$this->products      = $products;
		$this->relatedGroups = $relatedGroups;
	}

	/**
	 * @param bool $multilang
	 *
	 * @return BaseContainer
	 */
	public function getContainer($multilang = false): BaseContainer
	{
		$container = new BaseContainer();
		$container->addText('product', 'eshopCatalog.productForm.findProducts')
			->setHtmlAttribute('class', ['productAutocomplete', 'typeahead'])
			->setHtmlAttribute('autocomplete', 'off')
			->setHtmlId('product-finder');
		$container->addCustomData('template', __DIR__ . '/RelatedProductContainer.latte');
		$container->addCustomData('groups', $this->relatedGroups->getOptionsForSelect());

		$container->monitor(ProductForm::class, function(ProductForm $control) use ($container) {
			$product = $control->product;
			$related = [];

			if ($product) {
				foreach ($product->getRelatedProducts()->toArray() as $row) {
					/** @var RelatedProduct $row */
					$related[$row->getProduct()->getId()] = [
						'name'  => $row->getProduct()->getText()->name,
						'group' => $row->getGroup()->getId(),
					];
				}
			}

			$container->addCustomData('related', $related);
		});

		return $container;
	}

	/**
	 * @param           $container
	 * @param Product[] $relatedProducts
	 */
	public function setDefaults(BaseContainer &$container, array $relatedProducts): void
	{
	}

	/**
	 * @param array   $data
	 * @param Product $product
	 */
	public function saveData(array $data, Product $product): void
	{
		$data           = $data['relatedProductsIds'];
		$currentRelated = $product->getRelatedProducts()->toArray();

		foreach (array_diff_key($currentRelated, $data) as $entity)
			$this->em->remove($entity);

		foreach (array_diff_key($data, $currentRelated) as $prodId => $vals) {
			if (!is_numeric($prodId))
				continue;

			$group   = $this->relatedGroups->getReference($vals['group']);
			$related = new RelatedProduct($product, $this->products->getReference($prodId), $group);

			$this->em->persist($related);
		}
	}
}
