<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Core\AdminModule\Model\Sites;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTexts;
use Gallery\Model\Entities\AlbumText;
use Import\AdminModule\Model\ProcessingType\ProcessingType;
use Import\Model\Entities\Import;
use Import\Model\ImportDebugger;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Localization\ITranslator;
use Nette\Utils\Strings;
use Tracy\Debugger;

/**
 * Class ProcessPairing
 * @package EshopCatalog\AdminModule\Model\Import
 */
class ProcessPairing extends ProcessingType
{
	public string $title = 'eshopCatalog.import.processingType.pairing';

	public string $name = 'eshopCatalog.pairing';

	protected EntityManagerDecorator $em;

	protected ITranslator $translator;

	protected ImportHelper $importHelper;

	protected Sites $sites;

	public array $lastProductsRun = [];

	/**
	 * ProcessPairing constructor.
	 *
	 * @param EntityManagerDecorator $em
	 */
	public function __construct(EntityManagerDecorator $em, ITranslator $translator, ImportHelper $importHelper, Sites $sites)
	{
		$this->em           = $em;
		$this->translator   = $translator;
		$this->importHelper = $importHelper;
		$this->sites        = $sites;
	}

	/**
	 * @param Import $import
	 * @param array  $data
	 *
	 * @return array|string
	 * @throws \Exception
	 */
	public function run($import, $data)
	{
		$error = null;
		set_time_limit(1200);
		Debugger::$showBar = false;

		$pairing = $import->data;
		$result  = [
			'created'  => 0,
			'updated'  => 0,
			'notFound' => 0,
			'message'  => '',
		];

		$availabilities = [];
		foreach ($this->em->getRepository(Availability::class)->createQueryBuilder('a')->select('a.id, a.ident, a.position')
			         ->orderBy('a.position')->getQuery()->getArrayResult() as $row)
			$availabilities[$row['id']] = [
				'id'       => $row['id'],
				'ident'    => $row['ident'],
				'position' => $row['position'],
			];
		$supplier            = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
		$syncBy              = $import->getSyncOpt('syncBy');
		$soldOutAvailability = $this->em->getConnection()->fetchColumn("SELECT id FROM eshop_catalog__availability WHERE ident = ?", [Availability::SOLD_OUT]);
		$syncBySupplier      = $syncBy === 'supplierCode';

		$lang     = $this->translator->getDefaultLocale();
		$exists   = [];
		$imported = [];
		// Zjisteni existujicich produktu prirazenych dodavateli
		if ($syncBySupplier) {
			foreach ($this->em->getConnection()->fetchAll("SELECT p.id, ps.code FROM eshop_catalog__product_supplier ps
				INNER JOIN eshop_catalog__product p on ps.id_product = p.id
				WHERE ps.id_supplier = ?", [$supplier->getId()]) as $v) {
				$exists[$v['code']] = $v['id'];
			}
		} else {
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.' . $syncBy)
				         ->select('p.id, p.' . $syncBy)
				         ->join('p.suppliers', 'ps', 'WITH', 'ps.supplier = :supplier')->setParameter('supplier', $supplier->getId())
				         ->getQuery()->getResult() as $v)
				$exists[$v[$syncBy]] = $v['id'];
		}

		// Srovnani podle synchronizacniho pole a vyfiltrovani zvolenych vyrobcu
		$tmp                  = [];
		$allowedManufacturers = $pairing['allowedManufacturers']['value'] ?? null;
		if (count($allowedManufacturers) === 0)
			$allowedManufacturers = null;

		foreach ($data['data'] as $row) {
			if (!$allowedManufacturers || in_array($row[$pairing['manufacturer']['key']], $allowedManufacturers))
				$tmp[(string) $row[$pairing[$syncBy]['key']]] = $row;
		}

		$data['data'] = $tmp;
		$tmp          = null;

		// Nenalezení v importu nastavit počet ks na 0
		$notFound = array_diff_key($exists, $data['data']);
		if ($notFound) {
			foreach ($notFound as $p) {
				$this->em->getConnection()->exec('UPDATE eshop_catalog__product_supplier SET quantity = 0 WHERE id_product = ' . $p . ' AND id_supplier = ' . $supplier->getId());
				$result['notFound']++;
			}

			$notFound = null;
		}

		$chunks = array_chunk($data['data'], 25, true);
		$data   = null;

		$total            = 0;
		$totalLimit       = 20000;
		$galleries        = [];
		$entitiesForClean = [];

		foreach ($chunks as $chunk) {
			$this->em->beginTransaction();
			try {
				$i        = 0;
				$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
				foreach ($chunk as $rowKey => $row) {
					$exist = false;

					if (isset($exists[$row[$pairing[$syncBy]['key']]])) {
						/** @var Product $product */
						$product         = $this->em->getRepository(Product::class)->find($exists[$row[$pairing[$syncBy]['key']]]);
						$productTexts    = $product->getTexts()->toArray();
						$productSupplier = $product->getSupplier($supplier->getId());
						$exist           = true;
					}

					if (!$exist) {
						if (isset($pairing['ean']['key']) && isset($row[$pairing['ean']['key']])) {
							$product = $this->importHelper->findByEan($row[$pairing['ean']['key']]);
							if ($product) {
								$productTexts    = $product->getTexts()->toArray();
								$productSupplier = $product->getSupplier($supplier->getId());
								if (!$productSupplier)
									$productSupplier = new ProductSupplier($product, $supplier);
								$exist = true;
							}
						}

						if (!$exist && $import->getSyncOpt('onlyUpdate') == 1)
							continue;

						if (!$exist) {
							$product         = new Product();
							$productTexts    = $product->getTexts()->toArray();
							$productSupplier = new ProductSupplier($product, $supplier);
						}
					}

					$this->em->persist($product);
					$this->em->flush($product);

					$pVals = ['ean', 'code1', 'code2', 'vatRate', 'price' => 'priceVat', 'retailPrice'];
					foreach ($pVals as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						if ($v == 'vatRate') {
							$product->$k = $this->importHelper->getVatRate((int) $this->fixString($row[$pairing[$v]['key']]) ?: $pairing[$v]['fixedValue']);
						} else if (in_array($k, ['price', 'retailPrice'])) {
							$modify          = $pairing[$v]['modifyInt'];
							$priceWithoutVat = $pairing[$v]['withoutVat'] ? true : false;
							$price           = (float) $this->fixString($row[$pairing[$v]['key']]);
							if ($modify)
								$price = round($price * (float) $modify, 2);
							if ($priceWithoutVat)
								$price = round($price * $product->getVateRate()->getModifier(), 0);

							$this->importHelper->setProductPrice($product, $price, $k);
						} else {
							if (isset($pairing[$v]['key'])) {
								$product->$k = $this->fixString($row[$pairing[$v]['key']]);
							} else if (isset($pairing[$v]['fixedValue']))
								$product->$k = $this->fixString($pairing[$v]['fixedValue']);
						}
					}

					$pValsBool = ['isPublished', 'isAssort'];
					foreach ($pValsBool as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						if (isset($row[$pairing[$v]['key']]))
							if (!isset($pairing[$v]['key']))
								$product->$k = 0;
							else {
								$val         = (float) $this->fixString($row[$pairing[$v]['key']]);
								$compare     = (float) $pairing[$v]['value'];
								$product->$k = $this->importHelper->checkValueIs($pairing[$v]['valueIs'], $val, $compare) ? 1 : 0;
							}
						else if (isset($pairing[$v]['fixedValue']))
							$product->$k = (int) $this->fixString($pairing[$v]['fixedValue']);
					}

					$ptVals       = ['name', 'description', 'shortDescription'];
					$textsUpdated = false;
					foreach ($ptVals as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						$lastText = null;
						foreach ($pairing[$v]['key'] as $kLang => $vKey) {
							$text = $productTexts[$kLang];

							if (!$text) {
								$text                 = new ProductTexts($product, $kLang);
								$productTexts[$kLang] = $text;
							}

							$textVal = '';
							if ($vKey)
								$textVal = $this->fixString($row[$vKey]);
							else if (isset($pairing[$v]['fixedValue']))
								$textVal = $this->fixString($pairing[$v]['fixedValue']);

							if (!$textVal && $lastText && $lastText->$k)
								$textVal = $lastText->$k;

							$text->$k = $textVal;

							if ($v == 'name' && $text->$k) {
								$text->setAlias($text->$k);
							}

							if ($v == 'description' && $text->description) {
								$text->setShortDescription((string) $productTexts->description);
							}
							$lastText = $text;
						}

						$textsUpdated = true;
					}

					if ($textsUpdated) {
						$this->em->persist($product);
						foreach ($productTexts as $pt) {
							$this->em->persist($pt);
						}
					}

					$pSVals = ['code' => 'supplierCode'];
					foreach ($pSVals as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (isset($row[$pairing[$v]['key']]))
							$productSupplier->$k = $this->fixString($row[$pairing[$v]['key']]);
						else if (isset($pairing[$v]['fixedValue']))
							$productSupplier->$k = $this->fixString($pairing[$v]['fixedValue']);
					}

					// TODO dodělat nefixní hodnotu
					//				if (isset($pairing['category']['key']) && $row[$pairing['category']['key']] && false)
					//					$productSupplier->quantity = $this->importHelper->getManufacturer($this->fixString($row[$pairing['quantity']['key']]));

					if (isset($pairing['manufacturer']) && (!$exist || $pairing['manufacturer']['applyOn'] != 'new'))
						if (isset($row[$pairing['manufacturer']['key']])) {
							$man = $this->fixString($row[$pairing['manufacturer']['key']]);
							if ($man && $man != '') {
								$product->setManufacturer($this->importHelper->getManufacturer($man));
							} else
								$product->setManufacturer(null);
						} else if (isset($pairing['manufacturer']['fixedValue']))
							$product->setManufacturer($this->importHelper->getManufacturer($pairing['manufacturer']['fixedValue']));

					if (isset($pairing['inStock']) && (!$exist || $pairing['inStock']['applyOn'] != 'new'))
						$product->inStock = $product->quantity > 0;

					if (!$exist || $pairing['category'] && $pairing['category']['applyOn'] != 'new') {
						$baseCat = $this->importHelper->findCategory($import->getSyncOpt('baseCategory'));

						if ($baseCat) {
							$inSite = new ProductInSite($product, $this->sites->get($baseCat->getRoot()->getCategoryText()->alias));
							$inSite->setActive(0);
							$inSite->category = $baseCat;

							$this->em->persist($inSite);
						}

						if (isset($pairing['category']['fixedValue'])) {
							$cat = $this->importHelper->findCategory($pairing['category']['fixedValue']);
							if ($cat)
								$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $cat));
						} else {
							$pCat       = $pairing['category'];
							$importCats = null;

							if (isset($row[$pCat['key']])) {
								if ($pCat['separator'])
									$importCats = explode($pCat['separator'], $row[$pCat['key']]);
								else
									$importCats = [$row[$pCat['key']]];

								if ($pCat['deepSeparator']) {
									foreach ($importCats as $icK => $icV) {
										$importCats[$icK] = explode($pCat['deepSeparator'], $icV);
									}
								}
							}

							if ($baseCat && $importCats) {
								foreach ($importCats as $ic) {
									$deepC = $this->importHelper->getDeepCategory($baseCat, $ic, $lang);
									if ($deepC)
										$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $deepC));
								}
							}
						}
					}

					// Parametry
					if (isset($pairing['params']['key']) && (!$exist || $pairing['params']['applyOn'] != 'new')) {
						$params = [];
						foreach ($pairing['params']['key'] as $ik) {
							if (is_array($row[$ik]))
								foreach ($row[$ik] as $ikk => $ikv)
									$params[$ikk] = $ikv;
							elseif ($row[$ik])
								$params[$ik] = $row[$ik];
						}

						if ($params)
							$this->importHelper->parametersProccess($product, $params, $lang);
					}

					// Dostupnost
					$setQuantity = true;
					$quantity    = null;

					if (isset($row[$pairing['quantity']['key']]))
						$quantity = $this->fixString($row[$pairing['quantity']['key']]);
					else if (isset($pairing['quantity']['fixedValue']))
						$quantity = $this->fixString($pairing['quantity']['fixedValue']);

					// Pokud není nastaven počet skladem, zkontroluje se hodnota
					if ($pairing['inStock']['key']) {
						$inStock  = false;
						$iv       = $row[$pairing['inStock']['key']];
						$ic       = $pairing['inStock']['value'];
						$inStock  = $this->importHelper->checkValueIs($pairing['inStock']['valueIs'], $iv, $ic);
						$quantity = 0;
						if ($inStock)
							$quantity = $pairing['quantity']['fixedValue'] ?? 1;
					}

					if ($setQuantity && $quantity !== null) {
						$productSupplier->quantity = (int) $quantity;
						$setQuantity               = false;
					}

					// Dostupnost 2
					$compare = $pairing['idAvailability']['value'];
					if ($compare === '')
						$compare = $pairing['idAvailability']['custom'];
					// Pokud je skladem tak se nastavi skladem
					if ($pairing['idAvailability']['inStock']
						&& $this->importHelper->checkValueIs($pairing['idAvailability']['valueIs'], (int) $row[$pairing['idAvailability']['key']], (int) $compare)) {
						// Jeste kontrola - produkt nesmi byt skladem a mit nastaveno skladem
						$selAvailability = $availabilities[$pairing['idAvailability']['inStock']] ?? null;

						if ($selAvailability && $product->quantity <= 0 && $product->getAvailability()->getPosition() > $selAvailability['position'])
							$product->setAvailability($this->em->getReference(Availability::class, $pairing['idAvailability']['inStock']));
					} else if ($pairing['idAvailability']['soldOut']) {
						// Jinak zkontrolovat sklad a nastavit
						$maxSupplierStock = (int) $this->em->getConnection()
							->fetchColumn("SELECT max(quantity) FROM eshop_catalog__product_supplier WHERE id_product = ?", [$product->getId()]);
						if ($maxSupplierStock === 0 && $product->quantity <= 0)
							$product->setAvailability($this->em->getReference(Availability::class, $pairing['idAvailability']['soldOut']));
					}

					if (ImportDebugger::$allowImagesDownload) {
						if (isset($pairing['images']) && (!$exist || $pairing['images']['applyOn'] != 'new')) {
							$gallery = $this->importHelper->getGallery($product);

							foreach ($gallery->getImages()->toArray() as $k => $img) {
								if (!file_exists($img->getFile())) {
									$this->em->remove($img);
									$gallery->getImages()->remove($k);
								}
							}

							$galleryText = $gallery->getText($lang);
							if (!$galleryText)
								$galleryText = new AlbumText($gallery, $lang);

							$galleryText->title = $productTexts->name;
							$this->em->persist($galleryText);

							// Zjištění prefixu pokud je nastaven
							$urlPrefix = (string) ($pairing['imagesUrlPrefix']['key'] ?? $pairing['imagesUrlPrefix']['fixedValue']);

							$product->setGallery($gallery);
							$galleries[] = $gallery;

							// Získání obrázků z importu a přidání do galerie
							foreach ($pairing['images']['key'] as $k) {
								foreach ($this->importHelper->getImages($row[$k], $urlPrefix) as $img) {
									$this->importHelper->addImage($gallery, $img);
								}
							}
						}
					}

					if ($productSupplier) {
						if (!$productSupplier->getSupplier())
							$productSupplier->setSupplier($supplier);
						$this->em->persist($productSupplier);
						$entitiesForClean[] = $productSupplier;
					}

					if (!$product->getAvailability())
						$product->setAvailability($this->em->getReference(Availability::class, $soldOutAvailability));

					$product->validateCreated();
					if ($product) {
						$this->em->persist($product);
						$entitiesForClean[] = $product;
					}
					if ($productTexts) {
						foreach ($productTexts as $pt) {
							$this->em->persist($pt);
							$entitiesForClean[] = $pt;
						}
					}
					if ($exist)
						$result['updated']++;
					else
						$result['created']++;
					$i++;
					$total++;

					if (!$exist)
						$exists[$product->$syncBy] = $product;

					$imported[$product->$syncBy] = $product;

					if ($i == 20) {
						$i = 0;
					}

					if ($total >= $totalLimit) {
						break;
					}

					$this->em->flush();
					$this->importHelper->moveGalleryImage($galleries);
					$galleries = [];
				}

				$this->em->flush();
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->commit();
				$this->em->clear();
				$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
				$this->importHelper->moveGalleryImage($galleries);
				$galleries                   = [];
				$this->importHelper->onError = [];
				if ($total >= $totalLimit)
					break;
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->rollback();
				Debugger::log($e, 'import');
				$result['message'] = $e->getMessage();
				$result['product'] = $product->getId() . ' / ' . $product->code1;
				$error             = $e;

				foreach ($this->importHelper->onError as $onError)
					$onError();

				echo $e->getMessage();
				Debugger::$showBar = true;
				dump($e);
				die();

				break;
			}
		}
		Debugger::log($result, 'import');

		return $result;
	}

	/**
	 * TODO udělat globální třídu
	 *
	 * @param $str
	 *
	 * @return string
	 */
	private function fixString($str)
	{
		if (!is_string($str))
			return $str;
		$str = htmlspecialchars_decode($str, ENT_QUOTES);
		$str = html_entity_decode($str);

		return $str;
	}
}
