<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Entities\ExtraField;
use EshopCatalog\Model\Config;
use Nette\Application\LinkGenerator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Event\DaoEvent;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\FrontModule\Model\Dao;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Nette\Caching\Cache;
use Symfony\Component\EventDispatcher\EventDispatcher;

/**
 * class Categories
 * @package EshopCatalog\Model
 *
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'eshopCatalogCategories';

	const MODE_ESHOP    = 'eshop';
	const MODE_CHECKOUT = 'checkout';

	protected static string $mode = self::MODE_ESHOP;

	public static $allowGenerateLink = true;

	protected $entityClass = Category::class;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	public $cacheDep = [
		Cache::TAGS   => ['categories'],
		Cache::EXPIRE => '1 week',
	];

	/** @var array */
	protected $cCats, $cStructured, $cNavs, $cPath, $cQueryPath, $cSiteRoot;

	protected ?array $cFullUrl = null;

	/** @var bool */
	protected $cacheExist;

	/** @var Sites */
	protected $sitesService;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	public function __construct(CacheService $cacheService, Sites $sites, EventDispatcher $eventDispatcher)
	{
		$this->cacheService    = $cacheService;
		$this->eventDispatcher = $eventDispatcher;
		$this->sitesService    = $sites;
	}

	protected function cacheExist()
	{
		if ($this->cacheExist === null)
			$this->cacheExist = count(glob(TMP_DIR . '/cache/_' . self::CACHE_NAMESPACE . '/*')) > 1;

		return $this->cacheExist;
	}

	public function getRootIdForSite(?string $siteIdent = null): ?int
	{
		if ($siteIdent === null)
			$siteIdent = $this->sitesService->getCurrentSite()->getIdent();

		return $this->getAllRootIds()[$siteIdent] ?? null;
	}

	public function getAllRootIds(): array
	{
		if ($this->cSiteRoot === null) {
			$this->cSiteRoot = [];
			foreach ($this->getEr()->createQueryBuilder('c')
				         ->select('IDENTITY(ct.id) as id, ct.alias')
				         ->innerJoin('c.categoryTexts', 'ct')
				         ->andWhere('c.lvl = 0')
				         ->getQuery()->getArrayResult() as $row)
				$this->cSiteRoot[$row['alias']] = (int) $row['id'];
		}

		return $this->cSiteRoot;
	}

	public function getStructured(?int $rootId = null, ?string $lang = null): array
	{
		$lang   = $lang ?: $this->translator->getLocale();
		$rootId = $rootId ?: $this->getRootIdForSite();
		$cKey   = $rootId . '-' . $lang;

		if (!isset($this->cStructured[$cKey])) {
			$fullUrls = Config::load('allowCategoryFullUrlField', false) ? $this->getCategoryFullUrl() : [];
			/** @var Dao\Category[] $flat */
			$flat = [];
			/** @var Dao\Category[][] $grouped */
			$grouped = [];

			$published = $this->getEr()->createQueryBuilder('c')
				->addSelect('ct, IDENTITY(c.parent) as parent')
				->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				->where('c.isPublished = 1')
				->andWhere('c.root = :root')
				->setParameters([
					'lang' => $lang,
					'root' => $rootId,
				])
				->addOrderBy('c.lft')
				->getQuery()->getResult(Query::HYDRATE_ARRAY);

			foreach ($published as $v) {
				$cat           = $v[0];
				$cat['texts']  = $cat['categoryTexts'][$lang];
				$cat['parent'] = $v['parent'];

				$dao = $this->fillDao($cat);
				if ($fullUrls && $fullUrls[$dao->getId()]) {
					$dao->setAttrs(['originAlias' => $dao->alias]);
					$dao->alias = $fullUrls[$dao->getId()];
				}

				$grouped[$cat['parent']][$cat['id']] = $dao;
				$flat[$cat['id']]                    = &$grouped[$cat['parent']][$cat['id']];
			}

			$this->em->clear([Category::class, CategoryTexts::class]);

			$createTree = function(&$list, $parent) use (&$createTree) {
				$tree = [];
				foreach ($parent as $k => $l) {
					if (isset($list[$l->id])) {
						if ($l->lvl >= 1)
							foreach ($list[$l->id] as $kk => $v) {
								$list[$l->id][$kk]->setParent($parent[$k]);
							}
						$l->setChild($createTree($list, $list[$l->id]));
					}

					$tree[] = $l;
				}

				return $tree;
			};

			$baseCat = $this->getEr()->createQueryBuilder('c')->select('c.id')
				->where('c.lvl = 0')->setMaxResults(1)->getQuery()->getSingleScalarResult();

			if (!$baseCat)
				return [];

			$dao     = (new Dao\Category());
			$dao->id = $rootId;
			$dao->setParentId(0);

			$grouped[0][] = $dao;

			/** @var Dao\Category[] $tree */
			$tree = $createTree($grouped, $grouped[0])[0]->getChild();

			foreach ($flat as $k => &$v) {
				$v->link = $this->generateLink($v->id);
			}

			$this->cStructured[$cKey] = $tree;
		}

		return $this->cStructured[$cKey];
	}

	/**
	 * @param int|null    $rootId
	 * @param string|null $lang
	 *
	 * @return Dao\Category[]
	 */
	public function getCategories(?int $rootId = null, ?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if (!$rootId && self::$mode == self::MODE_ESHOP)
			$rootId = $this->getRootIdForSite();

		$cKey = $lang . '-' . ($rootId ?: 'all');

		if ($this->cCats[$cKey] === null) {
			$result = [];

			$loop = function(array $arr) use (&$loop, &$result) {
				/** @var Dao\Category[] $arr */
				foreach ($arr as $v) {
					$result[$v->id] = $v;

					if ($v->getChild())
						$loop($v->getChild());
				}
			};

			if (!$rootId) {
				foreach ($this->getAllRootIds() as $rootId)
					$loop($this->getStructured($rootId, $lang));
			} else
				$loop($this->getStructured($rootId, $lang));

			$this->cCats[$cKey] = $result;
		}

		return $this->cCats[$cKey];
	}

	protected function generateLink(int $id): ?string
	{
		return $this->linkGenerator ? $this->linkGenerator->link('EshopCatalog:Front:Default:category', ['id' => $id]) : '';
	}

	/**
	 * @param Dao\Category $category
	 *
	 * @return Dao\Category[]
	 */
	public function getPath(Dao\Category $category): array
	{
		if (!$this->cPath[$category->id]) {
			$path = [$category];
			$p    = $category->getParent();

			while ($p) {
				$path[] = $p;
				$p      = $p->getParent();
			}
			$this->cPath[$category->id] = array_reverse($path);
		}

		return $this->cPath[$category->id];
	}

	public function getQueryPath(int $id): array
	{
		$lang = $this->translator->getLocale();
		$key  = 'path/' . $id . '/' . $lang;

		if (!isset($this->cQueryPath[$lang])) {
			foreach ($this->getEr()->createQueryBuilder('c')
				         ->select('c.id, IDENTITY(c.parent) as parent, ct.alias')
				         ->andWhere('c.isPublished = 1 OR c.lvl = 0')
				         ->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				         ->setParameter('lang', $lang)
				         ->groupBy('c.id')
				         ->getQuery()->getArrayResult() as $row)
				$this->cQueryPath[$lang][$row['id']] = $row;
		}

		if (!$this->cQueryPath[$key]) {
			$flat = $this->cQueryPath[$lang];
			$path = [];
			$i    = 0;
			$cat  = $flat[$id];

			while ($cat && $i < 20) {
				$path[$cat['id']] = $cat;
				$cat              = $flat[$cat['parent']] ?? null;
			}

			$this->cQueryPath[$key] = array_reverse($path, true);
		}

		return $this->cQueryPath[$key];
	}

	public function get(int $id): ?Dao\Category
	{
		return $this->getCategories()[$id] ?? null;
	}

	/**
	 * @param string $alias
	 *
	 * @return Dao\Category[]
	 */
	public function findByAlias(string $alias): array
	{
		$result = [];

		foreach ($this->getCategories() as $c) {
			if ($c->alias == $alias)
				$result[] = $c;
		}

		return $result;
	}

	public function findNavigationId(int $categoryId): ?int
	{
		if ($this->cNavs === null) {
			$this->cNavs = $this->em->getRepository(Navigation::class)->createQueryBuilder('n')
				->select('n.id, n.componentParams')
				->where('n.componentType = :type')->setParameter('type', 'eshopCatalog.navigation.category')
				->getQuery()->getArrayResult();
		}

		foreach ($this->cNavs as $row) {
			if ($row['componentParams']['category'] == $categoryId)
				return (int) $row['id'];
		}

		return null;
	}

	/**
	 * @param Dao\Category      $category
	 * @param DaoNavigationItem $navigation
	 *
	 * @return Dao\Category[]
	 */
	public function getBreadcrumb(Dao\Category $category, DaoNavigationItem $navigation): array
	{
		$path = $this->getPath($category);

		$path[0]->name = $navigation->title;

		return $path;
	}

	/**
	 * @param int|null $catId
	 *
	 * @return array|string|null
	 */
	public function getCategoryFullUrl(?int $catId = null)
	{
		if ($this->cFullUrl === null) {
			$this->cFullUrl = [];
			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				         ->select('ef.sectionKey, ef.value')
				         ->where('ef.sectionName = :sn')
				         ->andWhere('ef.key = :key')
				         ->setParameters([
					         'sn'  => \EshopCatalog\Model\Entities\Category::EXTRA_FIELD_SECTION,
					         'key' => 'fullUrl',
				         ])
				         ->getQuery()->getArrayResult() as $row)
				$this->cFullUrl[(int) $row['sectionKey']] = $row['value'];
		}

		return $catId ? $this->cFullUrl[$catId] ?? null : $this->cFullUrl;
	}

	protected function fillDao($category)
	{
		/** @var Dao\Category $c */
		$c                    = (new Dao\Category());
		$c->id                = (int) $category['id'];
		$c->name              = $category['texts']['name'];
		$c->nameH1            = $category['texts']['nameH1'];
		$c->alias             = $category['texts']['alias'];
		$c->shortDescription  = $category['texts']['shortDescription'];
		$c->description       = (string) $category['texts']['description'];
		$c->image             = $category['image'];
		$c->lvl               = (int) $category['lvl'];
		$c->created           = $category['created'];
		$c->modified          = $category['modified'] ?: $category['created'];
		$c->filtersFromParent = (int) $category['filtersFromParent'];
		$c->rod               = $category['rod'];
		$c->setSeo($category['texts']['seo']);
		$c->setAttrs($category['attrs'] ?: []);

		$c->canProductsAddToCart = $category['canProductsAddToCart'] ? true : false;
		if ($category['canProductsAddToCart'] === null)
			$c->canProductsAddToCart = true;

		if ($category['parent'])
			$c->setParentId($category['parent'] ? (int) $category['parent'] : null);

		$this->eventDispatcher->dispatch(new DaoEvent($c, $category), Categories::class . '::fillDao');

		return $c;
	}

	public static function setMode(string $mode): void
	{
		self::$mode = $mode;
		Products::setMode(Products::MODE_CHECKOUT);
	}
}
