<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\RelatedProduct;
use Gallery\FrontModule\Model\Dao\Album;
use Nette\Application\LinkGenerator;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use Core\Model\Event\EventDispatcher;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;
use Users\Model\Http\UserStorage;
use EshopCatalog\FrontModule\Model\Dao;

class ProductsFacade
{
	const MODE_ESHOP    = 'eshop';
	const MODE_CHECKOUT = 'checkout';

	protected static string $mode = self::MODE_ESHOP;

	/** @var ITranslator */
	protected $translator;

	/** @var CacheService */
	protected $cacheService;

	/** @var Products */
	public $productsService;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var UserStorage */
	protected $userStorage;

	/** @var Albums */
	protected $albumsService;

	/** @var Manufacturers */
	protected $manufacturersService;

	/** @var Tags */
	protected $tagsService;

	/** @var Sites */
	protected $sitesService;

	/** @var Categories */
	protected $categoriesService;

	/** @var Dao\Product[] */
	protected $cProducts, $cVariants;

	public function __construct(ITranslator $translator, CacheService $cacheService, Products $products, EventDispatcher $eventDispatcher,
	                            UserStorage $userStorage, Albums $albums, Manufacturers $manufacturers, Tags $tags, Sites $sites, Categories $categories)
	{
		$this->translator           = $translator;
		$this->cacheService         = $cacheService;
		$this->productsService      = $products;
		$this->eventDispatcher      = $eventDispatcher;
		$this->userStorage          = $userStorage;
		$this->albumsService        = $albums;
		$this->manufacturersService = $manufacturers;
		$this->tagsService          = $tags;
		$this->sitesService         = $sites;
		$this->categoriesService    = $categories;
	}

	protected function loadBase(array $ids)
	{
		$whereIds = [];
		$loaded   = [];
		$result   = [];
		$locale   = $this->translator->getLocale();

		foreach ($ids as $k => $id) {
			if (isset($this->cProducts[$id])) {
				$loaded[$id] = $this->cProducts[$id];
				unset($ids[$k]);
			}
		}

		$keys = [];
		foreach ($ids as $id)
			$keys[] = 'product/' . $id . '/' . $locale;

		// Načtení z cache
		if ($keys) {
			foreach ($this->cacheService->productCache->bulkLoad($keys) as $key => $product) {
				$tmp = explode('/', $key);
				$id  = $tmp[1];

				if ($product)
					$result[$id] = $product;
				else
					$whereIds[] = $id;
			}
		}

		// Vytvoření získání produktů co nejsou v cache
		if (!empty($whereIds)) {
			$query = (new ProductQuery($locale))
				->withCategories()
				->withTexts()
				->withVatRate()
				->withVariantParent()
				->byId($whereIds);

			$qb = $query->getQueryBuilder($this->productsService->getEr());
			$qb->groupBy('p.id');

			foreach ($this->productsService->customQB as $cqb)
				$cqb($qb);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$tmp                = $this->productsService->normalizeHydrateArray($row);
				$tmp['extraFields'] = $this->productsService->getAllExtraFields()[$tmp['id']] ?? [];
				$dao                = $this->productsService->fillDao($tmp);

				if ($this->linkGenerator)
					$dao->link = $this->linkGenerator->link('EshopCatalog:Front:Default:product', [
						'id'     => $dao->getId(),
						'locale' => $locale,
					]);

				$cacheDep                = $this->productsService->cacheDep;
				$cacheDep[Cache::TAGS][] = 'product/' . $dao->getId();
				$cacheDep[Cache::EXPIRE] = '1 week';
				$this->cacheService->productCache->save('product/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				$result[$tmp['id']] = $dao;
			}
		}

		return [
			'prepared' => $result,
			'loaded'   => $loaded,
		];
	}

	/**
	 * Vrátí DAO produktů z cache nebo získá data z db a uloží do cache
	 *
	 * @param array       $ids
	 * @param string|null $siteIdent
	 *
	 * @return Dao\Product[]
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProducts(array $ids, ?string $siteIdent = null): array
	{
		$baseIds = $ids;
		$data    = $this->loadBase($ids);

		/** @var Dao\Product[] $loaded */
		$loaded = $data['loaded'];

		/** @var Dao\Product[] $result */
		$result = $data['prepared'];

		if (!empty($result)) {
			$variants         = [];
			$variantParentIds = [];
			$variantIds       = [];

			foreach ($result as $product) {
				if ($product->variantId)
					$variantIds[$product->getId()] = $product->variantId;
			}

			if (!empty($variantIds)) {
				foreach ($this->productsService->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					         ->select('IDENTITY(pv.product) as product, pv.variantId, pv.isDefault')
					         ->where('pv.variantId IN (:ids)')->setParameter('ids', array_unique($variantIds))
					         ->getQuery()->getArrayResult() as $row) {
					if ($row['isDefault'] == 1)
						$variantParentIds[$row['variantId']] = $row['product'];
					$variantIds[$row['variantId']][] = $row['product'];
					$variants[]                      = $row['product'];
				}

				if (!empty($variants)) {
					$tmp    = $this->loadBase(array_unique($variants));
					$loaded += $tmp['loaded'];
					$result += $tmp['prepared'];
				}
			}

			foreach ($result as $product)
				if (isset($variantParentIds[$product->variantId]))
					$product->variantOf = (int) $variantParentIds[$product->variantId];

			$this->productsService->loadFresh($result, $this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);
			$this->productsService->loadFeatures($result);
			$this->tagsService->loadTagsToProduct($result);

			for ($i = 0; $i <= 1; $i++) {
				foreach ($result as $id => $product) {
					if (!$product->isActive) {
						unset($result[$id]);
						continue;
					}

					if ($i === 0 && $product->variantOf && $product->variantOf !== $product->getId())
						continue;
					else if ($i === 1 && !$product->variantOf)
						continue;

					if ($product->galleryId) {
						/** @var Album $gallery */
						$gallery = $this->albumsService->get($product->galleryId);

						if ($product->helperData['variantImage']) {
							$img = $gallery->getImages()[$product->helperData['variantImage']];

							if ($img)
								$gallery->setCover($img);
						}

						$product->setGallery($gallery);
					}

					if ($product->manufacturerId)
						$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));

					if (!$product->defaultCategoryId && $this->cProducts[$product->variantOf])
						$product->defaultCategoryId = $this->cProducts[$product->variantOf]->defaultCategoryId;

					if ($product->defaultCategoryId) {
						$product->defaultCategory = $this->categoriesService->get($product->defaultCategoryId);
						$product->canAddToCart    = $product->defaultCategory->canProductsAddToCart ? true : false;
					}

					$this->cProducts[$id] = $product;
					$loaded[$id]          = $product;
					$result[$id]          = null;
				}
			}

			$this->eventDispatcher->dispatch(new ProductsEvent($loaded), Products::class . '::afterFillDao');

			$allLoaded = $loaded;

			foreach ($allLoaded as $id => $product)
				$this->cProducts[$id] = $product;

			foreach (array_diff(array_keys($loaded), $baseIds) as $v)
				unset($loaded[$v]);

			if (self::$mode != self::MODE_CHECKOUT)
				foreach ($loaded as $id => $product) {
					if (!$product->variantId)
						continue;

					foreach ($variantIds[$product->variantId] as $v)
						if ($allLoaded[$v])
							$product->variants[$v] = $allLoaded[$v];

					foreach ($product->variants as $k => $v)
						$product->variants[$k]->variants = &$product->variants;
				}
		}

		return $loaded;
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Product|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProduct(int $id, ?string $siteIdent = null): ?Dao\Product
	{
		return $this->getProducts([$id], $siteIdent)[$id] ?? null;
	}

	public function loadAlternative(Dao\Product &$product, $limit)
	{
		$data       = $this->productsService->getProductsIdInCategory($product->defaultCategoryId);
		$variantIds = array_keys($product->variants);

		foreach ($data as $k => $v) {
			if ($product->getId() == $v || in_array($v, $variantIds))
				unset($data[$k]);
		}

		shuffle($data);
		$products = $this->getProducts(array_slice($data, 0, $limit));

		$product->setAlternatives($products);
	}

	public function loadRelated(Dao\Product &$product, $limit)
	{
		if (!Config::load('allowRelatedProducts') || !empty($product->getRelated()))
			return;

		$related    = [];
		$ids        = [];
		$groupedIds = [];
		foreach ($this->productsService->em->getRepository(RelatedProduct::class)->createQueryBuilder('rp')
			         ->select('IDENTITY(rp.product) as product, g.id as groupId, gt.name as groupName')
			         ->innerJoin('rp.group', 'g', Join::WITH, 'g.isPublished = 1')
			         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
			         ->andWhere('rp.origin = :id')
			         ->orderBy('g.position')
			         ->setParameters([
				         'lang' => $this->translator->getLocale(),
				         'id'   => $product->getId(),
			         ])->getQuery()->getArrayResult() as $row) {
			if (!isset($related[$row['groupId']]))
				$related[$row['groupId']] = new Dao\RelatedGroup($row['groupId'], $row['groupName']);
			$ids[]                         = $row['product'];
			$groupedIds[$row['groupId']][] = $row['product'];
		}

		if ($ids) {
			$products = $this->getProducts($ids);

			foreach ($groupedIds as $groupId => $ids) {
				shuffle($ids);
				$i = 0;

				foreach ($ids as $id) {
					if (!isset($products[$id]))
						continue;

					$related[$groupId]->addProduct($products[$id]);
					$i++;

					if ($i >= $limit)
						break;
				}
			}

			$product->setRelated($related);
		}
	}

	public static function setMode(string $mode): void
	{
		self::$mode = $mode;
		Products::setMode(Products::MODE_CHECKOUT);
		Categories::setMode(Categories::MODE_CHECKOUT);
	}
}
