<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Presenters;

use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Components\CartAddForm;
use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Components\IProductsFilterFactory;
use EshopCatalog\FrontModule\Components\IProductsListFactory;
use EshopCatalog\FrontModule\Components\ProductPreview;
use EshopCatalog\FrontModule\Components\ProductsFilter;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Event\CategoryHeadEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use Gallery\FrontModule\Model\Albums;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Paths;
use Spatie\SchemaOrg\Schema;

class DefaultPresenter extends BasePresenter
{
	/** @var Category */
	public $category;

	/** @var Product */
	public $product;

	/** @var TemplatePage */
	protected $templatePage;

	/** @var ProductsFacade @inject */
	public $productsFacade;

	/** @var Categories */
	protected $categoriesService;

	/** @var Products */
	protected $productsService;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var Paths */
	protected $pathsService;

	/** @var Albums */
	protected $albumsService;

	/** @var Tags */
	protected $tagsService;

	/** @var Config */
	protected $config;

	/** @var Currencies */
	protected $currencies;

	public function __construct(Config $config, Categories $categories, Products $products, TemplatePages $templatePages, Paths $paths, Albums $albums, Tags $tags, Currencies $currencies)
	{
		$this->config               = $config;
		$this->categoriesService    = $categories;
		$this->productsService      = $products;
		$this->templatePagesService = $templatePages;
		$this->pathsService         = $paths;
		$this->albumsService        = $albums;
		$this->tagsService          = $tags;
		$this->currencies           = $currencies;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateHome']);

		$this->template->pageClass          .= ' page-eshopcatalog-default';
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
	}

	public function actionProduct(int $id): void
	{
		$this->product = $product = $this->productsFacade->getProduct($id);

		if (!$product)
			$this->error('eshopCatalog.product.notFound');

		$products           = [
			$product->getId() => $product,
		];
		$this->templatePage = $templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateProduct']);

		$this->productsFacade->loadAlternative($product, $this->config->get('productDetailAlternativeLimit', 4));
		$this->productsService->loadFeatures($products);
		$this->productsService->loadDocuments($products);
		$this->tagsService->loadTagsToProduct($products);

		$product = $products[$product->getId()];
		if ($product->alternative)
			$this->tagsService->loadTagsToProduct($product->alternative);

		$this->productsFacade->loadRelated($product, $this->config->get('productDetailRelatedLimit', 4));

		$this->setTitle($product->getSeo('title') ?: $product->getName());
		$this->seoDataFromEntity = $product->getSeo();

		if ($product->getQuantity() <= 0)
			$this->seoDataFromEntity['robots'] = 'noindex';
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));

		$url = $this->link('//this');
		// Meta
		$this['meta']
			->setMeta('title', $product->getName())
			->setMeta('og:title', $product->getName(), 'property')
			->setMeta('og:type', 'og:product', 'property')
			->setMeta('og:description', Strings::truncate(strip_tags($product->getDescription()), 255), 'property')
			->setMeta('og:url', $url, 'property')
			->setMeta('product:plural_title', $product->getName(), 'property')
			->setMeta('product:price:amount', strval($product->getBasePrice()), 'property')
			->setMeta('product:price:currency', $this->currencies->getCurrent()->getCode(), 'property')
			->setMeta('product:condition', $product->condition, 'property')
			->setMeta('product:availability', $product->getAvailability()->getOpenGraphText(), 'property');

		// Schema
		$schema = Schema::product()
			->productID($product->getId())
			->name($product->getName())
			->category($product->defaultCategory->name)
			->description(Strings::truncate(strip_tags($product->getDescription()), 255))
			->offers([
				Schema::offer()
					->price($product->getBasePrice())
					->priceCurrency($this->currencies->getCurrent()->getCode())
					->itemCondition($product->condition)
					->availability($product->getAvailability()->getSchemaUrl())
					->url($url),
			])
			->sku($product->getQuantity())
			->url($url);

		if ($product->getEan())
			$schema->gtin13($product->getEan());

		if ($product->getGallery() && $product->getGallery()->getCover()) {
			$imgUrl = $this->imagePipe->request($product->getGallery()->getCover()->getFilePath(), '1200x1200', 'fit');
			$schema->image($imgUrl);

			$this['meta']
				->setMeta('og:image', $imgUrl, 'property')
				->setMeta('image', $imgUrl);
		}

		if ($product->getManufacturer()) {
			$schema->brand(Schema::brand()
				->name($product->getManufacturer()->name)
				->logo($product->getManufacturer()->logo)
			);
			$schema->manufacturer(Schema::organization()
				->name($product->getManufacturer()->name)
				->logo($product->getManufacturer()->logo)
			);
		}
		$this['dataSchemaControl']->addSchema($schema);
	}

	public function actionCategory(int $id): void
	{
		$this->category = $category = $this->categoriesService->get($id);

		if (!$category)
			$this->error('eshopCatalog.category.notFound');

		$this->templatePage = $templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templatePage']);

		$this->category          = $category;
		$this->seoDataFromEntity = $category->getSeo();
		$this->templatePage      = $templatePage;

		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
	}

	public function actionSearch($q)
	{
		$templatePage       = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateResult']);
		$this->templatePage = $templatePage;

		$this['list']->setProductsBySearch((string) $q)
			->setFilterComponent($this['productsFilter']);

		$this->template->q                  = $q;
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-search';

		$this->setTitle($this->translator->translate('eshopCatalogFront.search.searchQuery', null, ['q' => $q]));
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));

		if (!$this->isAjax() && empty($this['list']->getProducts())) {
			$this->getActiveNavigation()->setSeo('robots', 'noindex, nofollow');
		}

		if (Config::load('allowSearchInContent', false)) {
			$this->template->alowSearchInContent = true;
			$this->template->contentSeachLink    = $this->link(':Core:Front:Default:search', ['q' => $q]);
		}
	}

	/*******************************************************************************************************************
	 * ============================== Render
	 */

	public function renderProduct()
	{
		$product      = $this->product;
		$templatePage = $this->templatePage;

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-product product-' . $id;
		$this->template->product            = $product;

		if ($product->defaultCategoryId) {
			$category = $this->categoriesService->get((int) $product->defaultCategoryId);

			if (!$category)
				$category = $this->categoriesService->get((int) array_values($product->categories)[0]);

			if ($category) {
				$this->template->breadcrumb += $this->getBreadcrumb($category);
			}
		}
	}

	public function renderCategory()
	{
		$category     = $this->category;
		$templatePage = $this->templatePage;

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-category';
		$this->template->category           = $category;
		$this->template->breadcrumb         = $this->getBreadcrumb($category);

		$this->processCategory();

		if (!$this->isAjax() && empty($this['list']->getProducts())) {
			$this->seoDataFromEntity['robots'] = 'noindex, nofollow';
		}

		if (!$category->getSeo('description'))
			$this['meta']->setMeta('description', Strings::truncate(str_replace('&nbsp;', ' ', strip_tags($category->description)), 130));
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentCartAddForm(ICartAddFormFactory $factory): CartAddForm
	{
		$control = $factory->create();

		if ($this->product)
			$control->setProduct($this->product);

		return $control;
	}

	protected function createComponentAddProductToCartForm(ICartAddFormFactory $factory): Multiplier
	{
		return new Multiplier(function($productId) use ($factory) : CartAddForm {
			$control = $factory->create();
			$control->setProduct($this->productsFacade->getProduct((int) $productId));

			return $control;
		});
	}

	protected function createComponentList(IProductsListFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentProductsFilter(IProductsFilterFactory $factory)
	{
		$control = $factory->create();

		if ($this->isAjax())
			$this->redrawControl('categoryHead');

		$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function() use ($control) {
			$this->redrawControl('categoryHead');
			$this['list']->redrawControl('listWrap');
		});

		return $control;
	}

	protected function createComponentProductPreview(IProductPreviewFactory $factory): Multiplier
	{
		return new Multiplier(function(string $id) use ($factory): ProductPreview {
			$control = $factory->create();

			$control->setProductById((int) $id);

			return $control;
		});
	}

	/*******************************************************************************************************************
	 * ==============================
	 */

	protected function getBreadcrumb(Category $category): array
	{
		$path    = $this->categoriesService->getPath($category);
		$baseNav = null;

		foreach ($path as $k => $cat) {
			$tmp = $this->categoriesService->findNavigationId($cat->id);

			if (!$tmp)
				continue;

			$tmp = $this->navigationsService->getNavigation($tmp);
			if ($tmp) {
				$baseNav = $tmp;
				break;
			}
		}

		return $baseNav ? $this->categoriesService->getBreadcrumb($category, $baseNav) : [];
	}

	protected function processCategory()
	{
		$this['list']->setCategory($this->category->getId())
			->setFilterComponent($this['productsFilter']);
		$paginator    = $this['list']->getPaginator();
		$title        = $this->category->getSeo('title') ?: $this->category->name;
		$description  = $this->category->description;
		$categoryName = $this->category->getNameH1();

		$this->eventDispatcher->dispatch(
			new CategoryHeadEvent($title, $categoryName, $description, $this['productsFilter']->getFilterValues(), $this->category),
			'eshopCatalog.default.actionCategory.head'
		);

		$titleSuffix = null;
		if ($paginator->getPage() > 1) {
			$titleSuffix = $this->translator->translate('eshopCatalogFront.pageXOfY', [
				'x' => $paginator->getPage(),
				'y' => $paginator->getPageCount(),
			]);
			$title       .= ' ' . $titleSuffix;
			$description = '';
		}

		$this->setTitle($title);
		$this->template->categoryName = $categoryName ?: $this->category->getNameH1();
		$this->template->description  = $description;
		$this->template->titleSuffix  = $titleSuffix;
		$this->template->pageTitle    = trim($categoryName . ' ' . $titleSuffix);
		$this->template->page         = $paginator->getPage();
	}
}
