<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Nette\Caching\Cache;
use Nette\SmartObject;

class ProductInSiteListener implements EventSubscriberInterface
{
	use SmartObject;

	/** @var array */
	protected static $cleared = [];

	/** @var CacheService */
	protected $cacheService;

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param ProductInSite      $productInSite
	 * @param LifecycleEventArgs $event
	 */
	public function postHandler(ProductInSite $productInSite, LifecycleEventArgs $event)
	{
		if (!$productInSite->category)
			return;

		$categoryId = $productInSite->category->getId();
		if (!self::$cleared[$categoryId]) {
			bdump("product in site '{$categoryId}' post update / remove");
			$this->cacheService->clean('default', [
				Cache::TAGS => ['listing/' . $categoryId, 'listing/search'],
			]);
		}

		self::$cleared[$categoryId] = true;
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param ProductInSite      $productInSite
	 * @param LifecycleEventArgs $args
	 */
	public function onPreFlush($productInSite, $args)
	{
		foreach ($args->getEntityManager()->getUnitOfWork()->getScheduledEntityInsertions() as $entity) {
			if ($entity instanceof ProductInSite && $entity->category && !self::$cleared[$entity->category->getId()]) {
				$categoryId = $entity->category->getId();
				bdump("product in site '{$categoryId}' pre flush");
				$this->cacheService->clean('default', [
					Cache::TAGS => ['listing/' . $categoryId, 'listing/search'],
				]);

				self::$cleared[$categoryId] = true;
			}
		}
	}
}
