<?php declare(strict_types = 1);

namespace EshopCatalog\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\ControllerPath;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;


/**
 * Class CategoriesController
 * @ControllerPath("/category")
 */
class CategoriesController extends BaseController
{
	protected Categories $categoriesDaoService;

	protected Products $productsDaoService;

	protected ProductsFacade $productsFacade;

	public function __construct(Categories $categoriesDaoService, Products $productsDaoService, ProductsFacade $productsFacade)
	{
		$this->categoriesDaoService = $categoriesDaoService;
		$this->productsDaoService   = $productsDaoService;
		$this->productsFacade       = $productsFacade;

		ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response)
	{
		$result = [];

		foreach ($this->categoriesDaoService->getAllRootIds() as $siteIdent => $id)
			$result[$siteIdent] = $this->categoriesDaoService->getStructured($id);

		return $result;
	}

	/**
	 * TODO
	 * Nekdy az bude chodit lepsi DAO dodelat.
	 *
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/{id}/subcategories")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", description="category id", deprecated=true)
	 * })
	 */
	public function subcategories(ApiRequest $request, ApiResponse $response)
	{
		$id  = $request->getParameter('id');
		$cat = $this->categoriesDaoService->get((int) $id);

		return $cat ? array_values($cat->getChild()) : [];
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", description="category id")
	 * })
	 */
	public function category(ApiRequest $request, ApiResponse $response)
	{
		$category = $this->categoriesDaoService->get(intval($request->getParameter('id')));
		if (!$category) {
			$this->sendError($response, "Category with this id doesn´t exist");
		}

		return $response->writeJsonBody(['data' => (array) $category]);
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 * @Path("/{id}/products")
	 *
	 * @return array
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", description="Category id"),
	 *      @RequestParameter(name="offset", type="int", description="Start", in="query", required=false),
	 *      @RequestParameter(name="limit", type="int", description="End", in="query", required=false),
	 *      @RequestParameter(name="sort", type="string", description="Sort", in="query", required=false)
	 * })
	 */
	public function categoryProducts(ApiRequest $request, ApiResponse $response)
	{

		$id     = $request->getParameter('id');
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', 0);
		$sort   = $request->getQueryParam('sort', null);

		$productIds = $this->productsDaoService->getProductsIdInCategory($id, $offset, $limit, $sort);
		$products   = $productIds ? $this->productsFacade->getProducts($productIds) : [];

		if (!$products) {
			$this->sendError($response, "Category with this id doesn´t exist or the request failed");
		}

		return array_values($products);
	}
}
