<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Entities\TSeo;
use EshopCatalog\Model\Config;
use EshopCatalog\FrontModule\Model\Dao\Manufacturer;
use EshopGifts\FrontModule\Model\Dao\Gift;
use Gallery\FrontModule\Model\Dao\Album;
use Nette\Utils\DateTime;

class Product
{
	use TSeo;

	/** @var int */
	public $id;

	/** @var int */
	public $quantity;

	/**
	 * Základní cena bez slev
	 *
	 * @var float
	 */
	public $basePrice;

	/** @var float */
	public $price;

	/** @var float */
	public $retailPrice;

	/** @var float */
	public $basePriceInBaseCurrency;

	/** @var float */
	public $priceInBaseCurrency;

	/** @var float */
	public $retailPriceInBaseCurrency;

	/** @var int */
	public $manufacturerId;

	/** @var Manufacturer */
	private $manufacturer;

	/** @var string */
	public $name;

	/** @var string */
	public $name2;

	/** @var string */
	public $shortDescription;

	/** @var string */
	public $description;

	/** @var DateTime */
	public $created;

	/** @var DateTime */
	public $modified;

	/** @var int */
	public $galleryId;

	/** @var Album */
	public $gallery;

	/** @var FeatureProduct[] */
	public $features;

	/** @var RelatedGroup[] */
	protected array $related = [];

	/** @var self[] */
	public $alternative;

	/** @var int */
	public $defaultCategoryId;

	/** @var Category */
	public $defaultCategory;

	/** @var int[] */
	public $categories;

	/** @var Tag[] */
	public $tags;

	/** @var string */
	public $link = '#';

	/** @var int */
	public $vatRate;

	/** @var string */
	protected $ean;

	/** @var string */
	public $code1;

	/** @var string */
	protected $code2;

	/** @var Availability */
	protected $availability;

	/** @var int */
	public $unlimitedQuantity = 0;

	/** @var array */
	protected $extraFields = [];

	/** @var bool */
	public $discountDisabled = false;

	/** @var bool */
	public $categoryGiftsAllowed = true;

	/** @var Gift[] */
	public $gifts = [];

	public bool $isActive = true;

	public bool $canAddToCart = true;

	public bool $isAssort = false;

	/** @var array|self[] */
	public array $variants = [];

	public ?int $variantId = null;

	public ?int $variantOf = null;

	public array $variantProductIds = [];

	public array $helperData = [];

	/** @var FeatureProduct[] */
	protected array $cVariantDifferences = [];

	public string $condition = 'new';

	/** @var Document[] */
	protected array $documents = [];

	/** @var bool */
	public bool $disablePickUpSpedition = false;

	/** @var bool */
	public bool $isOversize = false;

	public function __construct($id)
	{
		$this->id = $id;
	}

	/**
	 * @return int
	 */
	public function getId()
	{
		return $this->id;
	}

	/*******
	 * == Quantity
	 */

	public function getQuantity(): int
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			return 500;

		return (int) $this->quantity;
	}

	public function setQuantity(int $quantity): self
	{
		$this->quantity = $quantity;

		return $this;
	}

	/**
	 * Vrátí text pro překlad s počtem ks skladem
	 *
	 * @return string
	 */
	public function getQuantityText(): string
	{
		$q = $this->getQuantity();

		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			$text = 'inStock';
		else if ($q <= 0)
			$text = 'soldOut';
		else if ($q <= 4)
			$text = 'lastInStock1';
		else if ($q <= 15)
			$text = 'lastInStock2';
		else
			$text = 'inStock';

		return 'eshopCatalogFront.product.' . $text;
	}

	/**
	 * Vrátí stav poštu ks skladem
	 *
	 * @return string
	 */
	public function getQuantityColor(): string
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			return 'green';

		$q = $this->getQuantity();
		if ($q <= 0)
			return 'red';

		if ($q <= 15)
			return 'yellow';

		return 'green';
	}

	/*******
	 * == Price
	 */

	public function getPrice(): float { return (float) $this->price; }

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function getBasePrice(): float { return (float) $this->basePrice; }

	public function setBasePrice(float $price): self
	{
		$this->basePrice = $price;

		return $this;
	}

	public function getPriceWithoutVat(): float { return ($this->getPrice() / (1 + ($this->vatRate / 100))); }

	public function getRetailPrice(): float
	{
		return (float) $this->retailPrice;
	}

	/**
	 * @param float $retailPrice
	 */
	public function setRetailPrice(float $retailPrice)
	{
		$this->retailPrice = $retailPrice;
	}

	public function getRetailPriceWithoutVat(): float { return ($this->getRetailPrice() / (1 + ($this->vatRate / 100))); }

	/**
	 * @return Manufacturer
	 */
	public function getManufacturer()
	{
		return $this->manufacturer;
	}

	/**
	 * @param mixed $manufacturer
	 */
	public function setManufacturer($manufacturer)
	{
		$this->manufacturer = $manufacturer;
	}

	/**
	 * @return string
	 */
	public function getName()
	{
		$str = $this->name;

		if (Config::load('addManufacturerBeforeProductName') && $this->getManufacturer())
			$str = trim($this->getManufacturer()->name . ' ' . $str);

		return $str;
	}

	/**
	 * @param string $name
	 */
	public function setName($name)
	{
		$this->name = $name;
	}

	public function setName2($name)
	{
		$this->name2 = $name;
	}

	public function getName2(): ?string { return Config::load('enableProductName2') ? $this->name2 : $this->name; }

	/**
	 * @return string
	 */
	public function getDescription()
	{
		return $this->description;
	}

	/**
	 * @param string $description
	 */
	public function setDescription($description)
	{
		$this->description = $description;
	}

	/**
	 * @return Album|null
	 */
	public function getGallery()
	{
		return $this->gallery;
	}

	/**
	 * @param mixed $gallery
	 */
	public function setGallery($gallery)
	{
		$this->gallery = $gallery;
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeatures($features)
	{
		$this->features = $features;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeatures() { return $this->features; }

	public function getFeatureById(int $id)
	{
		foreach ($this->getFeatures() as $feature) {
			if ($feature->idFeature === $id)
				return $feature;
		}

		return null;
	}

	public function getGroupedFeatures(): array
	{
		/** @var FeatureProduct[] $result */
		$result = [];

		foreach ($this->getFeatures() as $feature) {
			if (!isset($result[$feature->idFeature]))
				$result[$feature->idFeature] = $feature;
			else
				$result[$feature->idFeature]->value = trim($result[$feature->idFeature]->value) . ', ' . trim($feature->value);
		}

		return $result;
	}

	/**
	 * @param string $type
	 *
	 * @return Tag|null
	 */
	public function getTag($type) { return $this->tags[$type] ?? null; }

	/**
	 * @param self[] $array
	 *
	 * @return $this
	 */
	public function setAlternatives($array)
	{
		$this->alternative = $array;

		return $this;
	}

	/** @return int */
	public function getVatRate()
	{
		return $this->vatRate;
	}

	/**
	 * @param int $vatRate
	 *
	 * @return Product
	 */
	public function setVatRate($vatRate): Product
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	/** @return string */
	public function getEan()
	{
		return $this->ean;
	}

	/**
	 * @param string $ean
	 *
	 * @return Product
	 */
	public function setEan($ean): Product
	{
		$this->ean = $ean;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getCode1()
	{
		return $this->code1;
	}

	/**
	 * @param $code1
	 *
	 * @return Product
	 */
	public function setCode1($code1): Product
	{
		$this->code1 = $code1;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getCode2()
	{
		return $this->code2;
	}

	/**
	 * @param $code2
	 *
	 * @return Product
	 */
	public function setCode2($code2): Product
	{
		$this->code2 = $code2;

		return $this;
	}

	public function setExtraFields(array $array): self
	{
		$this->extraFields = $array;

		return $this;
	}

	public function getExtraFields(): array { return $this->extraFields; }

	/**
	 * @param string $key
	 * @param mixed  $default
	 *
	 * @return mixed|null
	 */
	public function getExtraField(string $key, $default = null) { return $this->extraFields[$key] ?? $default; }

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function getAvailability(): Availability { return $this->availability; }

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	/**
	 * ====== Variants
	 */

	public function getSortedVariantsByFeature(?int $variantId = null)
	{
		if (!$this->features)
			return $this->variants;

		if ($variantId === null)
			$variantId = array_values($this->features)[0]->idFeature;

		$tmp = [];
		$not = [];

		foreach ($this->variants as $variant) {
			$feature = $variant->getFeatureById($variantId);

			if ($feature) {
				$tmp[$feature->valuePosition] = $variant;
			} else {
				$not[] = $variant;
			}
		}
		ksort($tmp);

		return array_merge($tmp, $not);
	}

	public function hasVariants(): bool { return $this->variants ? true : false; }

	public function getVariantDifferences(): array
	{
		if (empty($this->variants))
			return [];

		if (empty($this->cVariantDifferences)) {
			$used     = [];
			$features = [];
			foreach ($this->features as $v)
				$features[$v->getKey()] = $v;

			foreach ($this->variants as $v)
				foreach ($v->features as $f)
					$used[$f->getKey()]++;

			foreach ($used as $k => $v)
				if ($v === count($this->variants))
					unset($features[$k]);

			if (count($features) === 1) {
				uksort($this->variants, function($a, $b) {
					$a = array_values($this->variants[$a]->features)[0];
					$b = array_values($this->variants[$b]->features)[0];

					return $a->valuePosition <=> $b->valuePosition;
				});
			}

			$this->cVariantDifferences = $features;
		}

		return $this->cVariantDifferences;
	}

	public function getFromPrice(): float
	{
		$min = $this->getPrice();

		foreach ($this->variants as $v)
			if ($v->getPrice() < $min)
				$min = $v->getPrice();

		return $min;
	}

	public function getFromPriceWithoutVat(): float
	{
		$min = $this->getPriceWithoutVat();

		foreach ($this->variants as $v)
			if ($v->getPriceWithoutVat() < $min)
				$min = $v->getPriceWithoutVat();

		return $min;
	}

	/**
	 * @return self[]
	 */
	public function getRelated(): array
	{
		return is_array($this->related) ? $this->related : [];
	}

	/**
	 * @param Product[] $products
	 */
	public function setRelated(array $products): void
	{
		$this->related = $products;
	}

	/**
	 * @param Document $document
	 */
	public function addDocument(Document $document): void { $this->documents[] = $document; }

	/**
	 * @return array|Document[]
	 */
	public function getDocuments(): array { return $this->documents; }
}
