<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Lang\DefaultLang;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Dao\FilterItem;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Caching\Cache;
use Nette\SmartObject;

/**
 * TODO misto hledání podle productsId používat třeba kategorie
 *
 * Class FilterService
 * @package EshopCatalog\FrontModule\Model
 */
class FilterService
{
	use SmartObject;

	const CACHE_NAMESPACE = 'eshopCatalogFilter';

	/** @var EntityManagerDecorator */
	protected $em;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var Features */
	protected $featuresService;

	/** @var Config */
	protected $config;

	/** @var array */
	protected $cProductsData;

	/** @var array */
	public static $cacheTags = [];

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => ['filters'],
		Cache::EXPIRE => '5 minutes',
	];

	/** @var array */
	protected $cFeatures, $cFeatureValues;

	/**
	 * FilterService constructor.
	 *
	 * @param EntityManagerDecorator $em
	 * @param DefaultLang            $defaultLang
	 * @param CacheService           $cacheService
	 * @param Features               $features
	 * @param Config                 $config
	 */
	public function __construct(EntityManagerDecorator $em, DefaultLang $defaultLang, CacheService $cacheService, Features $features, Config $config)
	{
		$this->em              = $em;
		$this->defaultLang     = $defaultLang;
		$this->cacheService    = $cacheService;
		$this->featuresService = $features;
		$this->config          = $config;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterGroup[]
	 */
	public function getFeatures(array $productIds = [])
	{
		$groups = [];
		$qb     = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
			->select('IDENTITY(fv.feature) as featureId, IDENTITY(fp.featureValue) as valueId')
			->join('fp.featureValue', 'fv', 'WITH', 'fv.isPublished = 1')
			->join('fv.feature', 'f', 'WITH', 'f.useAsFilter = 1')
			->orderBy('f.position')->addOrderBy('fv.position');

		if ($productIds) {
			$qb->where('fp.product IN (:products)')
				->setParameter('products', $productIds);
		}

		$data = $qb->getQuery()->getScalarResult();

		foreach ($data as $row) {
			if (!isset($groups[$row['featureId']])) {
				$tmp = $this->featuresService->getFeatureById($row['featureId']);

				if ($tmp && $tmp['name']) {
					$tmpV       = new FilterGroup();
					$tmpV->id   = $tmp['id'];
					$tmpV->name = $tmp['name'];
					$tmpV->type = 'feature';

					$groups[$row['featureId']] = $tmpV;
				}
			}

			$g = $groups[$row['featureId']];

			if (!isset($g->items[$row['valueId']])) {
				$tmp = $this->featuresService->getFeatureValueById($row['valueId']);

				if ($tmp && $tmp['name']) {
					$tmpV        = new FilterItem();
					$tmpV->id    = $tmp['id'];
					$tmpV->value = $tmp['name'];
					$tmpV->type  = 'featureValue';
					$tmpV->rodM  = $tmp['rodM'];
					$tmpV->rodZ  = $tmp['rodZ'];
					$tmpV->rodS  = $tmp['rodS'];

					$g->items[$row['valueId']] = $tmpV;
				}
			}
		}

		return $groups;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getMinMaxPrice($productIds)
	{
		$key = 'minMax_' . md5(serialize($productIds));

		$data = [];
		$tmp  = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->select('MIN(p.price), MAX(p.price)')
			->where('p.id IN (:ids)')->setParameter('ids', $productIds)
			->getQuery()->useResultCache(true, 60)->getScalarResult();
		if (isset($tmp[0]))
			$data = ['min' => $tmp[0][1], 'max' => $tmp[0][2]];

		return $data;
	}

	public function getManufacturers($productIds)
	{
		$key = 'manu_' . md5(serialize($productIds));

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($productIds) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'manufacturers';

			if (self::$cacheTags['manu'])
				$dep[Cache::TAGS][] = self::$cacheTags['manu'];

			$toUppercase = $this->config->get('filterManufacturersUppercase', false);
			$data        = [];
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('m.id, m.name')
				         ->join('p.manufacturer', 'm')->where('p.id IN (:ids)')->andWhere('m.name != \'\'')
				         ->andWhere('m.isPublished = 1')
				         ->setParameter('ids', $productIds)->groupBy('m.id')->orderBy('m.position')
				         ->getQuery()->getScalarResult() as $row) {
				$data[$row['id']]        = new FilterItem();
				$data[$row['id']]->id    = $row['id'];
				$data[$row['id']]->value = $toUppercase ? mb_strtoupper($row['name']) : $row['name'];
			}

			return $data;
		});
	}

	/**
	 * @param int  $id
	 * @param bool $deep
	 *
	 * @return array
	 */
	public function getCategoryFilters(int $id, $deep = true)
	{
		$key  = 'categoryFilters_' . $id . ($deep ? '_deep' : '');
		$data = [];

		$features = $this->cacheService->categoryCache->load($key, function(&$dep) use ($id, $deep) {
			$dep                = $this->cacheDep;
			$dep[Cache::EXPIRE] = '14 days';
			$dep[Cache::TAGS][] = 'category/' . $id;

			$find = function(int $id) {
				return $this->em->getRepository(CategoryFilter::class)->createQueryBuilder('f')
					->select('IDENTITY(f.feature) as featureId')
					->andWhere('f.category = :category')->setParameters(['category' => $id])
					->orderBy('f.position')
					->getQuery()->getArrayResult();
			};

			$features = $find($id);

			if (!$features && $deep) {
				try {
					$stmt = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM eshop_catalog__category WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, eshop_catalog__category h WHERE @r <> 0) T1 
						JOIN eshop_catalog__category T2 ON T1._id = T2.id ORDER BY T1.lvl DESC
						");
					$stmt->execute();
					$tmp = $stmt->fetchAll();
				} catch (\Exception $e) {
					$tmp = [];
				}
				array_shift($tmp);
				array_pop($tmp);
				$categories = [];
				foreach ($tmp as $row) {
					$dep[Cache::TAGS][] = 'category/' . $row['id'];
					$categories[]       = $row['id'];
				}

				while (!empty($categories) && empty($features)) {
					$id       = array_pop($categories);
					$features = $find((int) $id);

					if ($features)
						break;
				}
			}

			return array_map(function($row) { return $row['featureId']; }, $features);
		});

		if (!$features)
			return $data;

		foreach ($features as $featureId) {
			$tmp = $this->featuresService->getFeatureById($featureId);

			$group       = new FilterGroup();
			$group->id   = (int) $tmp['id'];
			$group->name = $tmp['name'];
			$group->type = 'feature';

			$data[$featureId] = $group;

			foreach ($this->featuresService->getValuesByFeatureId((int) $featureId) as $tmp) {
				$item        = new FilterItem();
				$item->id    = (int) $tmp['id'];
				$item->value = $tmp['name'];
				$item->type  = 'featureValue';

				$data[$featureId]->items[$tmp['id']] = $item;
			}
		}

		return $data;
	}

	public function getProductsData(array $productsIds, string $cacheKey)
	{
		if (!$this->cProductsData) {
			$result = [
				'f' => [],
				'm' => [],
				'p' => [],
			];

			// TODO cenové hladiny a jiné slevy
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id as id, p.price as price, IDENTITY(fp.featureValue) as f, IDENTITY(p.manufacturer) as m')
				         ->leftJoin('p.featureProducts', 'fp')
				         ->where('p.id IN (:ids)')->setParameter('ids', $productsIds)
				         ->getQuery()->enableResultCache(60)->getScalarResult() as $v) {

				$result['p'][$v['id']]['price'] = $v['price'];
				if ($v['f']) {
					$result['f'][$v['f']][]       = $v['id'];
					$result['p'][$v['id']]['f'][] = $v['f'];
				}

				if ($v['m']) {
					$result['m'][$v['m']][$v['id']] = $v['id'];
					$result['p'][$v['id']]['m']     = $v['m'];
				} else {
					$result['m'][0][$v['id']]   = $v['id'];
					$result['p'][$v['id']]['m'] = $v['m'];
				}
			}
			$this->cProductsData = $result;
		}

		return $this->cProductsData;
	}
}
