<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\Supplier;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\AlbumText;
use Import\AdminModule\Model\ProcessingType\ProcessingType;
use Import\Model\Entities\Import;
use Import\Model\Entities\ImportHistory;
use Import\Model\ImportDebugger;
use Kdyby\Doctrine\EntityManager;
use Nette\Localization\ITranslator;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Strings;
use Tracy\Debugger;
/**
 * Class ProcessPairing
 * @package EshopCatalog\AdminModule\Model\Import
 */
class ProcessPairing extends ProcessingType
{
	/** @var string */
	public $title = 'eshopCatalog.import.processingType.pairing';

	/** @var string */
	public $name = 'eshopCatalog.pairing';

	/** @var EntityManager */
	protected $em;

	/** @var ITranslator */
	protected $translator;

	/** @var ImportHelper */
	protected $importHelper;

	/** @var array */
	public $lastProductsRun = [];

	/**
	 * ProcessPairing constructor.
	 *
	 * @param EntityManager $em
	 */
	public function __construct(EntityManager $em, ITranslator $translator, ImportHelper $importHelper)
	{
		$this->em           = $em;
		$this->translator   = $translator;
		$this->importHelper = $importHelper;
	}

	/**
	 * @param Import $import
	 * @param array  $data
	 *
	 * @return string
	 */
	public function run($import, $data)
	{
		$error = null;
		set_time_limit(600);

		$pairing = $import->data;
		$result  = [
			'created'  => 0,
			'updated'  => 0,
			'notFound' => 0,
			'message'  => '',
		];

		$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
		$syncBy   = $import->getSyncOpt('syncBy');


		$langKey = null;
		$lang    = $this->translator->getDefaultLocale();
		if (isset($pairing['lang']['key']))
			$langKey = $pairing['lang']['key'];
		else if (isset($pairing['lang']['fixedValue']))
			$lang = $pairing['lang']['fixedValue'];
		$this->importHelper->lang = $lang;

		$tmp      = $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.' . $syncBy)
			->select('p.id, p.' . $syncBy)
			->join('p.suppliers', 'ps', 'WITH', 'ps.supplier = :supplier')->setParameter('supplier', $supplier->getId())
			->getQuery()->getResult();
		$exists   = [];
		$imported = [];

		foreach ($tmp as $v)
			$exists[$v[$syncBy]] = $v['id'];
		unset($tmp);

		$tmp = [];
		foreach ($data['data'] as $row) {
			$tmp[(string) $row[$pairing[$syncBy]['key']]] = $row;
		}
		$data['data'] = $tmp;
		$tmp          = null;

		// Nenalezení v importu nastavit počet ks na 0
		$notFound = array_diff_key($exists, $data['data']);
		if ($notFound) {
			foreach ($notFound as $p) {
				$this->em->getConnection()->exec('UPDATE eshop_catalog__product_supplier SET quantity = 0 WHERE id_product = ' . $p . ' AND id_supplier = ' . $supplier->getId());
				$result['notFound']++;
			}

			$notFound = null;
		}

		$chunks = array_chunk($data['data'], 25, true);
		$data   = null;

		$total            = 0;
		$totalLimit       = 10000;
		$galleries        = [];
		$entitiesForClean = [];

		foreach ($chunks as $chunk) {
			$this->em->beginTransaction();
			try {
				$i        = 0;
				$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
				foreach ($chunk as $rowKey => $row) {
					try {
						if ($langKey)
							$lang = $row[$langKey];
						$exist = false;

						if (isset($exists[$row[$pairing[$syncBy]['key']]])) {
							/** @var Product $product */
							$product         = $this->em->getRepository(Product::class)->find($exists[$row[$pairing[$syncBy]['key']]]);
							$productTexts    = $product->getProductText($lang);
							$productSupplier = $product->getSupplier($supplier->getId());
							$exist           = true;
						}

						if (!$exist && $import->getSyncOpt('onlyUpdate') == 1)
							continue;

						if (!$exist) {
							if (isset($pairing['ean']['key']) && isset($row[$pairing['ean']['key']])) {
								$product = $this->importHelper->findByEan($row[$pairing['ean']['key']]);
								if ($product) {
									$productTexts = $product->getProductText($lang);
									if (!$productTexts)
										$productTexts = new ProductTexts($product, $lang);
									$productSupplier = $product->getSupplier($supplier->getId());
									if (!$productSupplier)
										$productSupplier = new ProductSupplier($product, $supplier);
									$exist = true;
								}
							}

							if (!$exist) {
								$product         = new Product();
								$productTexts    = new ProductTexts($product, $lang);
								$productSupplier = new ProductSupplier($product, $supplier);
							}
						}

						// TODO hodit do jednoho pole a jednoho cyklu
						$pVals     = ['ean', 'code1', 'code2', 'price' => 'priceVat', 'vatRate', 'retailPrice'];
						$pValsBool = ['isPublished'];
						$ptVals    = ['name', 'description'];
						$pSVals    = ['code' => 'supplierCode'];

						foreach ($pVals as $k => $v) {
							$k = is_string($k) ? $k : $v;

							if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
								continue;

							if ($v == 'vatRate') {
								$product->$k = $this->importHelper->getVatRate((int) $this->fixString($row[$pairing[$v]['key']]) ?: $pairing[$v]['fixedValue']);
							} else if (in_array($k, ['price', 'retailPrice'])) {
								$this->importHelper->setProductPrice($product, (float) $this->fixString($row[$pairing[$v]['key']]), $k);
							} else {
								if (isset($pairing[$v]['key'])) {
									$product->$k = $this->fixString($row[$pairing[$v]['key']]);
								} else if (isset($pairing[$v]['fixedValue']))
									$product->$k = $this->fixString($pairing[$v]['fixedValue']);
							}
						}

						foreach ($pValsBool as $k => $v) {
							$k = is_string($k) ? $k : $v;

							if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
								continue;

							if (isset($row[$pairing[$v]['key']]))
								if (!isset($pairing[$v]['key']))
									$product->$k = 0;
								else
									$product->$k = $this->fixString($row[$pairing[$v]['key']]) == $pairing[$v]['value'] ? 1 : 0;
							else if (isset($pairing[$v]['fixedValue']))
								$product->$k = $this->fixString($pairing[$v]['fixedValue']);
						}

						foreach ($ptVals as $k => $v) {
							$k = is_string($k) ? $k : $v;

							if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
								continue;

							if (isset($row[$pairing[$v]['key']]))
								$productTexts->$k = $this->fixString($row[$pairing[$v]['key']]);
							else if (isset($pairing[$v]['fixedValue']))
								$productTexts->$k = $this->fixString($pairing[$v]['fixedValue']);

							if ($v == 'description' && $productTexts->description) {
								$productTexts->setShortDescription((string)Strings::truncate($productTexts->description, 255));
							}
						}

						foreach ($pSVals as $k => $v) {
							$k = is_string($k) ? $k : $v;

							if (isset($row[$pairing[$v]['key']]))
								$productSupplier->$k = $this->fixString($row[$pairing[$v]['key']]);
							else if (isset($pairing[$v]['fixedValue']))
								$productSupplier->$k = $this->fixString($pairing[$v]['fixedValue']);
						}

						// TODO dodělat nefixní hodnotu
						//				if (isset($pairing['category']['key']) && $row[$pairing['category']['key']] && false)
						//					$productSupplier->quantity = $this->importHelper->getManufacturer($this->fixString($row[$pairing['quantity']['key']]));

						if (isset($pairing['manufacturer']) && (!$exist || $pairing['manufacturer']['applyOn'] != 'new'))
							if (isset($row[$pairing['manufacturer']['key']])) {
								$man = $this->fixString($row[$pairing['manufacturer']['key']]);
								if ($man && $man != '') {
									$product->setManufacturer($this->importHelper->getManufacturer($man));
								} else
									$product->setManufacturer(null);
							} else if (isset($pairing['manufacturer']['fixedValue']))
								$product->manufacturer = $this->importHelper->getManufacturer($pairing['manufacturer']['fixedValue']);

						if (isset($pairing['inStock']) && (!$exist || $pairing['inStock']['applyOn'] != 'new'))
							$product->inStock = $product->quantity > 0;

						if (!$exist || $pairing['category'] && $pairing['category']['applyOn'] != 'new') {
							$baseCat = $this->importHelper->findCategory($import->getSyncOpt('baseCategory'));

							if ($baseCat)
								$product->idCategoryDefault = $baseCat;

							if (isset($pairing['category']['fixedValue'])) {
								$cat = $this->importHelper->findCategory($pairing['category']['fixedValue']);
								if ($cat)
									$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $cat));
							} else {
								$pCat       = $pairing['category'];
								$importCats = null;

								if (isset($row[$pCat['key']])) {
									if ($pCat['separator'])
										$importCats = explode($pCat['separator'], $row[$pCat['key']]);
									else
										$importCats = [$row[$pCat['key']]];

									if ($pCat['deepSeparator']) {
										foreach ($importCats as $icK => $icV) {
											$importCats[$icK] = explode($pCat['deepSeparator'], $icV);
										}
									}
								}

								if ($baseCat && $importCats) {
									foreach ($importCats as $ic) {
										$deepC = $this->importHelper->getDeepCategory($baseCat, $ic, $lang);
										if ($deepC)
											$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $deepC));
									}
								}
							}
						}

						if (isset($pairing['params']['key']) && (!$exist || $pairing['params']['applyOn'] != 'new')) {
							$params = [];
							foreach ($pairing['params']['key'] as $ik) {
								if (is_array($row[$ik]))
									foreach ($row[$ik] as $ikk => $ikv)
										$params[$ikk] = $ikv;
								elseif ($row[$ik])
									$params[$ik] = $row[$ik];
							}

							$this->importHelper->parametersProccess($product, $params);
						}

						$setQuantity = true;
						$quantity    = null;

						if (isset($row[$pairing['quantity']['key']]))
							$quantity = $this->fixString($row[$pairing['quantity']['key']]);
						else if (isset($pairing['quantity']['fixedValue']))
							$quantity = $this->fixString($pairing['quantity']['fixedValue']);

						// Pokud není nastaven počet skladem, zkontroluje se hodnota
						if ($pairing['inStock']['key']) {
							$inStock = false;
							$iv      = $row[$pairing['inStock']['key']];
							$ic      = $pairing['inStock']['value'];
							switch ($pairing['inStock']['valueIs']) {
								case '<':
									$inStock = $iv < $ic;
									break;
								case '<=':
									$inStock = $iv <= $ic;
									break;
								case '=':
									$inStock = $iv == $ic;
									break;
								case '>=':
									$inStock = $iv >= $ic;
									break;
								case '>':
									$inStock = $iv > $ic;
									break;
							}

							$quantity = 0;
							if ($inStock)
								$quantity = $pairing['quantity']['fixedValue'] ?? 1;
						}

						if (!empty($this->importHelper->variantsForQuantity) && $quantity !== null) {
							$setQuantity = false;
							foreach ($this->importHelper->variantsForQuantity as $variant)
								$this->importHelper->setProductVariantQuantity($supplier, $variant, $quantity);

							$this->importHelper->variantsForQuantity = [];
						}

						if ($setQuantity && $quantity !== null) {
							$productSupplier->quantity = $quantity;
							$setQuantity               = false;
						}

						if (ImportDebugger::$allowImagesDownload) {
							if (isset($pairing['images']) && (!$exist || $pairing['images']['applyOn'] != 'new')) {
								$gallery = $this->importHelper->getGallery($product);

								foreach ($gallery->getImages()->toArray() as $k => $img) {
									if (!file_exists($img->getFile())) {
										$this->em->remove($img);
										$gallery->getImages()->remove($k);
									}
								}

								$galleryText = $gallery->getText($lang);
								if (!$galleryText)
									$galleryText = new AlbumText($gallery, $lang);

								$galleryText->title = $productTexts->name;
								$this->em->persist($galleryText);

								$urlPrefix = (string) ($pairing['imagesUrlPrefix']['key'] ?? $pairing['imagesUrlPrefix']['fixedValue']);

								$product->gallery = $gallery;
								$galleries[]      = $gallery;

								$imgs = [];
								foreach ($pairing['images']['key'] as $ik) {
									if (is_object($row[$ik]) || is_array($row[$ik]))
										foreach ($row[$ik] as $ikv)
											$imgs[] = $urlPrefix . (string) $ikv;
									else
										$imgs[] = $urlPrefix . (string) $row[$ik];
								}

								foreach ($imgs as $img) {
									if ($img)
										$this->importHelper->addImage($gallery, $img);
								}
							}
						}

						if ($productSupplier) {
							if (!$productSupplier->getSupplier())
								$productSupplier->setSupplier($supplier);
							$this->em->persist($productSupplier);
							$entitiesForClean[] = $productSupplier;
						}
						$product->validateCreated();
						if ($product) {
							$this->em->persist($product);
							$entitiesForClean[] = $product;
						}
						if ($productTexts) {
							$this->em->persist($productTexts);
							$entitiesForClean[] = $productTexts;
						}
						if ($exist)
							$result['updated']++;
						else
							$result['created']++;
						$i++;
						$total++;

						if (!$exist)
							$exists[$product->$syncBy] = $product;

						$imported[$product->$syncBy] = $product;

						if ($i == 20) {
							$i = 0;
						}

						if ($total >= $totalLimit) {
							break;
						}

						$this->em->flush();
						$this->importHelper->moveGalleryImage($galleries);
						$galleries = [];
					} catch (\Exception $e) {
						$this->em = $this->em::create($this->em->getConnection(), $this->em->getConfiguration());
					}
				}

				$this->em->flush();
				$this->em->commit();
				$this->em->clear();
				$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
				$this->importHelper->moveGalleryImage($galleries);
				$galleries                   = [];
				$this->importHelper->onError = [];
				if ($total >= $totalLimit)
					break;
			} catch (\Exception $e) {
				$this->em->rollback();
				$result['message'] = $e->getMessage();
				$result['product'] = $product->getId() . ' / ' . $product->code1;
				$error             = $e;

				foreach ($this->importHelper->onError as $onError)
					$onError();

				if (!$this->em->isOpen()) {
					$this->em = $this->em::create($this->em->getConnection(), $this->em->getConfiguration());
				}
				break;
			}
		}

		return $result;
	}

	/**
	 * TODO udělat globální třídu
	 *
	 * @param $str
	 *
	 * @return string
	 */
	private function fixString($str)
	{
		if (!is_string($str))
			return $str;
		$str = htmlspecialchars_decode($str, ENT_QUOTES);
		$str = html_entity_decode($str);

		return $str;
	}
}
