<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\VatRate;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Nette\Utils\FileSystem;

/**
 * Class Products
 * @package EshopCatalog\AdminModule\Model
 *
 * @method Product|null|object = getReference($id)
 * @method Product[]|null getAll()
 * @method Product|null get($id)
 */
class Products extends BaseEntityService
{
	use TPublish;

	protected $entityClass = Product::class;

	/**
	 * @param array|int $ids
	 * @param int       $manufacturerId
	 *
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function setManufacturer($ids, $manufacturerId)
	{
		try {
			$manufacturer = $this->em->getReference(Manufacturer::class, $manufacturerId);
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setManufacturer($manufacturer);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/**
	 * @param array|int $ids
	 * @param int       $vatRateId
	 *
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function setVatRate($ids, $vatRateId)
	{
		try {
			$vatRate = $this->em->getReference(VatRate::class, $vatRateId);
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setVateRate($vatRate);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/** Vrati seznam produktu, ktere obsahuji dany retezec
	 * @return ArrayCollection|Product[]
	 */
	public function getByTerm($term = null)
	{
		if (is_null($term)) {
			return new ArrayCollection();
		}
		$query = $this->getEr()->createQueryBuilder('p', 'p.id')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vr')
			->setParameter('lang', 'cs')//TODO ITranslator
			->orWhere('pt.name LIKE :term')->setParameter('term', '%' . $term . '%')
			->orWhere('p.code1 LIKE :term')
			->orderBy('pt.name', 'ASC');

		$products = $query->getQuery()->getResult();

		return $products;
	}

	public function deleteDuplicity()
	{
		try {
			set_time_limit(600);
			$data = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name as name, p.code1, p.code2, count(p) as q')
				->leftJoin('p.productTexts', 'pt')
				->having('count(p) > 1')->groupBy('pt.name, p.code1, p.code2')->getQuery()->getResult();

			foreach ($data as $row) {
				$products = $this->getEr()->createQueryBuilder('p')->where('pt.name = :name AND p.code1 = :code1')
					->join('p.productTexts', 'pt')->orderBy('p.id')
					->setParameters(['code1' => $row['code1'], 'name' => $row['name']]);

				if ($row['code2'] == null) {
					$products->andWhere('p.code2 IS NULL');
				} else {
					$products->andWhere('p.code2 = :code2')->setParameter('code2', $row['code2']);
				}

				$products = $products->getQuery()->getResult();

				if (count($products) >= 2) {
					array_pop($products);

					foreach ($products as $product) {
						if ($product->gallery) {
							FileSystem::delete(WWW_DIR . $product->gallery->generatePath());

							$this->em->createQuery('DELETE FROM ' . Image::class . ' i WHERE i.album = :album')->execute(['album' => $product->gallery->getId()]);
							$this->em->createQuery('DELETE FROM ' . Album::class . ' a WHERE a.id = :album')->execute(['album' => $product->gallery->getId()]);
						}

						$this->em->remove($product);
						$this->em->flush();
					}

					$this->em->clear();
				}
			}
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}
}
