<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Caching\Cache;

/**
 * Class FeatureProducts
 * @package EshopCatalog\FrontModule\Model
 */
class FeatureProducts extends BaseFrontEntityService
{

	protected $entityClass = FeatureProduct::class;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['featureProduct'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function __construct(CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	/**
	 * @param int $idProduct
	 *
	 * @return null|Dao\FeatureProduct
	 */
	public function getFeaturesForProduct($idProduct)
	{
		return $this->cacheService->productCache->load('productFeatures_' . $idProduct, function(&$dep) use ($idProduct) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'product/' . $idProduct;

			$result = $this->getEr()->createQueryBuilder('fp')
				->select('IDENTITY(ft) as featureId, ft.name as featureName, fv.id as featureValueId, fvt.name as featureValueName')
				->join('fp.feature', 'f')
				->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				->join('fp.featureValue', 'fv')
				->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				->setParameter('lang', $this->defaultLang->locale)
				->andWhere('fp.product = :id')->setParameter('id', $idProduct)
				->getQuery()->getArrayResult();

			$emptyKeys = [];
			if ($result) {
				$arr = [];
				foreach ($result as $fp) {
					if (!$fp['featureName']) {
						$emptyKeys[] = $fp['featureUd'];
						continue;
					}

					$dep[Cache::TAGS][]    = 'feature/' . $fp['featureId'];
					$fp['productId']       = $idProduct;
					$arr[$fp['featureId']] = $this->fillDao($fp);
				}

				return $arr;
			}

			return [];
		});
	}

	/**
	 * @param array $arr
	 *
	 * @return Dao\FeatureProduct
	 */
	protected function fillDao($arr): Dao\FeatureProduct
	{
		$fp                 = new Dao\FeatureProduct();
		$fp->idProduct      = $arr['productId'];
		$fp->idFeature      = $arr['featureId'];
		$fp->name           = $arr['featureName'];
		$fp->idFeatureValue = $arr['featureValueId'];
		$fp->value          = $arr['featureValueName'];

		return $fp;
	}
}
