<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use EshopCatalog\Model\Entities\ProductVariantSupplier;
use EshopCatalog\Model\Entities\Tag;
use Kdyby\Doctrine\QueryBuilder;
use Kdyby\Doctrine\QueryObject;
use Kdyby\Persistence\Queryable;

class ProductQuery extends QueryObject
{
	/** @var array|\Closure[] */
	private $filter = [];

	/** @var array|\Closure[] */
	private $select = [];

	/** @var string */
	protected $lang;

	public function __construct($lang)
	{
		$this->lang = $lang;
	}

	public function skipId($id)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			$qb->andWhere('p.id != :notId')->setParameter('notId', $id);
		};

		return $this;
	}

	public function withTexts($addSelect = true)
	{
		$this->filter[] = function(QueryBuilder $qb) {
			$qb->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
				->setParameter('lang', $this->lang);
		};
		if ($addSelect)
			$this->select[] = function(QueryBuilder $qb) {
				$qb->addSelect('pt');
			};

		return $this;
	}

	public function withCategories()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->addSelect('cp, IDENTITY(p.idCategoryDefault) as defaultCategory')
				->leftJoin('p.categoryProducts', 'cp');
		};

		return $this;
	}
	
	public function withVatRate()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->addSelect('vr.rate as vatRate')
				->leftJoin('p.vatRate', 'vr');
		};
		
		return $this;
	}

	public function withTags()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->leftJoin('p.productTags', 'ptags');
		};

		return $this;
	}

	public function withTag($tag)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($tag) {
			$qb->innerJoin(Tag::class, 'tag', 'WITH', 'tag.type = :tagType')
				->setParameter('tagType', $tag)
				->innerJoin('p.productTags', 'pTags', 'WITH', 'pTags.tag = tag.id')
				->groupBy('p.id');
		};

		return $this;
	}

	public function withoutTag($tag)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($tag) {
			$qb->leftJoin('p.productTags', 'pTags')
				->leftJoin('pTags.tag', 'tags')
				->andWhere('pTags IS NULL OR tags.type != :withoutTag')->setParameter('withoutTag', $tag)
				->groupBy('p.id');
		};

		return $this;
	}

	public function inCategory($id)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			if (is_array($id))
				$qb->andWhere('cp.category IN (:categoryId) OR p.idCategoryDefault IN (:categoryId)');
			else
				$qb->andWhere('cp.category = :categoryId OR p.idCategoryDefault = :categoryId');
			$qb->setParameter('categoryId', $id)
				->innerJoin('p.categoryProducts', 'cp');
		};

		return $this;
	}

	public function search($query)
	{
		$this->withTexts(false);

		$this->filter[] = function(QueryBuilder $qb) use ($query) {
			$qb->leftJoin('p.manufacturer', 'manu');
			if (!in_array('cp', $qb->getAllAliases())) {
				$qb->leftJoin('p.categoryProducts', 'cp');
				$qb->leftJoin('cp.category', 'c');
			}
			if (!in_array('catD', $qb->getAllAliases())) {
				$qb->leftJoin('p.idCategoryDefault', 'catD');
			}

			foreach (explode(' ', $query) as $k => $word) {
				$word = trim((string) $word, ',. ');
				$qb->andWhere("CONCAT(manu.name, ' ', pt.name, ' ', p.code1) LIKE :search_" . $k)->setParameter('search_' . $k, strtolower("%$word%"));
			}
			$qb->andWhere('c.isPublished = 1 OR c IS NULL')->andWhere('catD.isPublished = 1 OR catD IS NULL');
		};

		return $this;
	}

	public function onlyInStockOrSupplier($pvsIds = null)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($pvs) {
			$arr[] = 'p.quantity > 0';

			if (!in_array('ps', $qb->getAllAliases()))
				$qb->leftJoin('p.suppliers', 'ps');
			$arr[] = 'ps.quantity > 0';

			if (!in_array('pv', $qb->getAllAliases()))
				$qb->leftJoin('p.productVariants', 'pv');
			$arr[] = 'pv.quantity > 0';

			if ($pvs)
				$arr[] = 'p.id IN (' . implode(',', $pvs) . ')';

			$qb->andWhere(implode(' OR ', $arr));
		};

		return $this;
	}

	public function useFilter($filter)
	{
		if (isset($filter['variants']))
			$this->filterVariants($filter['variants']);
		if (isset($filter['features']))
			$this->filterFeatures($filter['features']);
		if (isset($filter['priceRange']['min']))
			$this->filterPriceMin($filter['priceRange']['min']);
		if (isset($filter['priceRange']['max']))
			$this->filterPriceMax($filter['priceRange']['max']);
		if (isset($filter['manu']))
			$this->filterManu($filter['manu']);
		if (isset($filter['onlyStock']) && $filter['onlyStock'] === true)
			$this->onlyInStockOrSupplier($filter['onlyStockPVS'] ?? null);

		return $this;
	}

	public function filterPriceMin($min)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($min) {
			$qb->andWhere('p.price >= :priceMin')->setParameter('priceMin', $min);
		};

		return $this;
	}

	public function filterPriceMax($max)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($max) {
			$qb->andWhere('p.price <= :priceMax')->setParameter('priceMax', $max);
		};

		return $this;
	}

	public function filterVariants($ids)
	{
		//		if ($ids)
		$this->filter[] = function(QueryBuilder $qb) use ($ids) {
			$qb->join('p.productVariants', 'pv')
				->andWhere('pv.id IN (:pvIds)')->setParameter('pvIds', $ids);
		};

		return $this;
	}

	public function filterFeatures($ids)
	{
		if ($ids)
			$this->filter[] = function(QueryBuilder $qb) use ($ids) {
				if ($ids)
					$qb->andWhere('p.id IN (:featureP)')->setParameter('featureP', $ids);
			};

		return $this;
	}

	public function filterManu($ids)
	{
		if ($ids) {
			$this->filter[] = function(QueryBuilder $qb) use ($ids) {
				$qb->andWhere('p.manufacturer ' . (is_array($ids) ? 'IN (:ids)' : '= :ids'))->setParameter('ids', $ids);
			};
		}

		return $this;
	}

	public function selectIds()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->select('p.id')->groupBy('p.id');
		};

		return $this;
	}

	public function selectQuantity()
	{
		$this->filter[] = function(QueryBuilder $qb) {
			$qb->leftJoin('p.suppliers', 'ps')
				->leftJoin('p.productVariants', 'pv')
				->leftJoin(ProductVariantSupplier::class, 'pvs', 'WITH', 'pvs.productVariant = pv.id');
		};
		$this->select[] = function(QueryBuilder $qb) {
			$qb->select('p.id, p.quantity as productQ, SUM(ps.quantity) as suppliersQ, SUM(pv.quantity) as variantsQ, SUM(pvs.quantity) as variantSuppliersQ');
		};

		return $this;
	}

	/**
	 * @param int|int[] $id
	 *
	 * @return $this
	 */
	public function byId($id)
	{
		$this->select[] = function(QueryBuilder $qb) use ($id) {
			if (is_array($id))
				$qb->andWhere('p.id IN (:id)');
			else
				$qb->andWhere('p.id = :id');
			$qb->setParameter('id', $id);
		};

		return $this;
	}

	public function sortBy($sort = null, $direction = null)
	{
		if ($sort) {
			$this->select[] = function(QueryBuilder $qb) use ($sort, $direction) {
				if (strpos($sort, 'pt') === 0 && !in_array('pt', $qb->getAllAliases()))
					$qb->innerJoin('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')->setParameter('lang', $this->lang);
				$qb->orderBy($sort, $direction);
			};
		}

		return $this;
	}

	/**
	 * @param \Kdyby\Persistence\Queryable $repository
	 *
	 * @return \Doctrine\ORM\Query|\Doctrine\ORM\QueryBuilder
	 */
	protected function doCreateQuery(Queryable $repository)
	{
		$qb = $this->createBasicDql($repository);

		foreach ($this->select as $modifier) {
			$modifier($qb);
		}

		return $qb->addOrderBy('p.id', 'DESC');
	}

	protected function doCreateCountQuery(Queryable $repository)
	{
		return $this->createBasicDql($repository)->select('count(p.id)');
	}

	private function createBasicDql(Queryable $repository)
	{
		$qb = $repository->createQueryBuilder('p', 'p.id')
			->addSelect('IDENTITY(p.gallery) as gallery')
			->addSelect('IDENTITY(p.manufacturer) as manufacturer')
			->andWhere('p.isPublished = 1');

		foreach ($this->filter as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	public function getQueryBuilder(Queryable $repository)
	{
		return $this->doCreateQuery($repository);
	}
}
