<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\FeaturesTypeService;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Helpers\GoogleExportEnums;
use EshopProductsComparison\DI\EshopProductsComparisonExtension;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\AdminModule\Model\Features;

class FeatureForm extends BaseControl
{
	public ?Feature $feature = null;

	public ?FeatureTexts $featureTexts = null;

	protected Features            $featureServices;
	protected FeaturesTypeService $featuresTypeService;
	protected CacheService        $cacheService;

	public function __construct(
		Features            $features,
		FeaturesTypeService $featuresTypeService,
		CacheService        $cacheService
	)
	{
		$this->featureServices     = $features;
		$this->featuresTypeService = $featuresTypeService;
		$this->cacheService        = $cacheService;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.featureForm.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addText('unit', 'eshopCatalog.feature.unit')
			->setMaxLength(10)
			->setNullable();
		$form->addBool('isPublished', 'eshopCatalog.featureForm.isPublished')
			->setDefaultValue(1);
		$form->addBool('useAsFilter', 'eshopCatalog.featureForm.useAsFilter')
			->setDefaultValue(1);
		$form->addBool('useForVariantDiff', 'eshopCatalog.featureForm.useForVariantDiff')
			->setDefaultValue(1);

		if (Config::load('features.allowType')) {
			$form->addSelect('type', 'eshopCatalog.featureForm.type', [
				Feature::TYPE_CHECK => 'eshopCatalog.feature.type.check',
				Feature::TYPE_RANGE => 'eshopCatalog.feature.type.range',
			])
				->setDescription('eshopCatalog.feature.typeChangeWarning');
			$form->addText('decimals', 'eshopCatalog.feature.decimals')
				->setHtmlType('number')
				->setHtmlAttribute('step', 1)
				->setHtmlAttribute('min', 0)
				->setHtmlAttribute('max', 5)
				->setDefaultValue(0);
		}

		$form->addSelect('valueType', 'eshopCatalog.featureForm.valueType', [
			Feature::VALUE_TYPE_TEXT  => 'eshopCatalog.feature.valueType.text',
			Feature::VALUE_TYPE_COLOR => 'eshopCatalog.feature.valueType.color',
		])->setDefaultValue(Feature::VALUE_TYPE_TEXT);

		if (Config::load('features.allowDescription')) {
			$form->addText('productTabTitle', 'eshopCatalog.featureForm.productTabTitle')
				->setIsMultilanguage()
				->setMaxLength(255);
		}

		if (class_exists(EshopProductsComparisonExtension::class)) {
			$form->addBool('exportHeureka', 'eshopCatalog.featureForm.exportHeureka')->setDefaultValue(1);
			$form->addText('heurekaName', 'eshopCatalog.featureForm.heurekaName')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addBool('exportZbozi', 'eshopCatalog.featureForm.exportZbozi')->setDefaultValue(1);
			$form->addText('zboziName', 'eshopCatalog.featureForm.zboziName')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addBool('exportGoogle', 'eshopCatalog.featureForm.exportGoogle')->setDefaultValue(1);
			$form->addText('googleName', 'eshopCatalog.featureForm.googleName')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addSelect('googleType', 'eshopCatalog.featureForm.googleType', ['' => null] + GoogleExportEnums::getListForSelect());
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			FeatureForm::class . '::createForm'
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$oldType = $this->feature ? $this->feature->type : null;
			$feature = $this->feature ?: new Feature();

			$feature->setIsPublished($values->isPublished);
			$feature->setUseAsFilter($values->useAsFilter);
			$feature->type              = $values->type ?: Feature::TYPE_CHECK;
			$feature->unit              = $values->unit ?: null;
			$feature->decimals          = (int) $values->decimals ?: 0;
			$feature->exportHeureka     = $values->exportHeureka ?: 1;
			$feature->exportZbozi       = $values->exportZbozi ?: 1;
			$feature->exportGoogle      = $values->exportGoogle ?: 1;
			$feature->googleType        = $values->googleType ?: null;
			$feature->useForVariantDiff = (int) $values->useForVariantDiff;
			$feature->valueType         = $values->valueType ?: Feature::VALUE_TYPE_TEXT;

			$langValues = $form->convertMultilangValuesToArray();
			$texts      = $feature->getFeatureTexts()->toArray();

			foreach ($langValues as $lang => $vals) {
				if ($vals['name'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text              = $texts[$lang] ?? new FeatureTexts($feature, $lang);
				$text->name        = $vals['name'];
				$text->heurekaName = $vals['heurekaName'];
				$text->zboziName   = $vals['zboziName'];
				$text->googleName  = $vals['googleName'];

				if (Config::load('features.allowDescription'))
					$text->productTabTitle = $vals['productTabTitle'];

				$this->em->persist($text);
				$texts[$lang] = $text;
			}

			$feature->setFeatureTexts($texts);
			$this->em->persist($feature);

			// Změna typu hodnot
			if (Config::load('features.allowType') && $oldType && $oldType != $this->feature->type) {
				set_time_limit(5 * 60);
				if (!$this->featuresTypeService->changeType($oldType, $this->feature->type, $this->feature->getId()))
					throw new \Exception('Change feature type failed');
			}

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $feature;
			$this->eventDispatcher->dispatch($event, FeatureForm::class . '::formSuccess');

			$this->em->persist($feature);

			$this->em->flush();
			$this->em->commit();

			$form->addCustomData('featureId', $feature->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.featureForm.featureSaved');

			$this->cacheService->defaultCache->clean([
				Cache::Tags => [\EshopCatalog\FrontModule\Model\Features::CACHE_NAMESPACE],
			]);

			return true;
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}

		return false;
	}

	public function setFeature($id)
	{
		$this->feature = $this->featureServices->get($id);

		if ($this->feature) {
			$form = $this['form'];
			$d    = [
				'isPublished'       => $this->feature->getIsPublished(),
				'useAsFilter'       => $this->feature->getUseAsFilter(),
				'type'              => $this->feature->type,
				'unit'              => $this->feature->unit,
				'decimals'          => $this->feature->decimals,
				'exportHeureka'     => $this->feature->exportHeureka,
				'exportZbozi'       => $this->feature->exportZbozi,
				'exportGoogle'      => $this->feature->exportGoogle,
				'googleType'        => $this->feature->googleType,
				'useForVariantDiff' => $this->feature->useForVariantDiff,
				'valueType'         => $this->feature->valueType,
			];
			$form->setDefaults($d);

			$mTexts = $form->prepareMultilangTextsForForm([
				'name',
				'productTabTitle',
				'heurekaName',
				'zboziName',
				'googleName',
			], $this->feature->getFeatureTexts()->toArray());
			$form->setDefaults($mTexts['texts']);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->feature), FeatureForm::class . '::setFeature');
		} else
			throw new InvalidArgumentException();
	}
}

