<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Manufacturers;

use Core\Model\UI\BaseControl;
use EshopCatalog\AdminModule\Model\Manufacturers;

/**
 * Class ManufacturersGrid
 * @package EshopCatalog\AdminModule\Components\Manufacturers
 */
class ManufacturersGrid extends BaseControl
{
	/** @var Manufacturers */
	protected Manufacturers $manufacturerServices;

	public function __construct(Manufacturers $manufacturers)
	{
		$this->manufacturerServices = $manufacturers;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();
		if ($this->manufacturerServices->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleMergeAll(): void
	{
		$this->manufacturerServices->mergeAll();
		$this->redirect('this');
	}

	public function handleRepairPositions(): void
	{
		$this->manufacturerServices->repairPositions();
		$this->redirect('this');
	}

	public function handleSortAlphabetically(): void
	{
		$this->manufacturerServices->sortAlphabetically();
		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$qb = $this->manufacturerServices->getEr()->createQueryBuilder('m')->addSelect('mt')
			->leftJoin('m.texts', 'mt', 'WITH', 'mt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('m.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('manufacturersGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('name', 'eshopCatalog.defaultGrid.name', 'ManufacturersSuppliers:editManufacturer');
		$grid->addColumnStatus('isPublished', 'eshopCatalog.defaultGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('name', '');

		// Actions
		$grid->addAction('edit', '', 'ManufacturersSuppliers:editManufacturer')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->manufacturerServices->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->manufacturerServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else
			$this->redirect('this');
	}
}
