<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\AdminModule\Model\Redirects;
use Core\Model\Entities\EntityRepository;
use Core\Model\Entities\ExtraField;
use Core\Model\Entities\Redirect;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductDocument;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductPrice;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductPriceLevelCountry;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVideo;
use EshopCatalog\Model\Entities\RelatedProduct;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\VatRate;
use EshopProductsComparison\Model\Entities\ProductExport;
use EshopStock\DI\EshopStockExtension;
use Gallery\AdminModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Nette\Application\BadRequestException;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\Http\Url;
use Nette\Localization\Translator;
use Nette\Utils\FileSystem;
use EshopCatalog\Model\Products as BaseProducts;
use Tracy\Debugger;

/**
 * @method Product|null|object getReference($id)
 * @method Product[]|null getAll()
 * @method Product|null get($id)
 * @method EntityRepository getEr()
 */
class Products extends BaseEntityService
{
	use TPublish;

	const SESSION_SECTION = 'EshopCatalog/Admin/Products';

	protected $entityClass = Product::class;

	protected BaseProducts   $baseProductsService;
	protected Albums         $albums;
	protected Translator     $translator;
	protected CacheService   $cacheService;
	protected Redirects      $redirects;
	protected ProductsFacade $productsFacade;
	protected LinkGenerator  $linkGenerator;
	protected Sites          $sites;

	public function __construct(BaseProducts  $baseProductsService, Translator $translator, Albums $albums,
	                            CacheService  $cacheService, Redirects $redirects, ProductsFacade $productsFacade,
	                            LinkGenerator $linkGenerator, Sites $sites)
	{
		$this->baseProductsService = $baseProductsService;
		$this->translator          = $translator;
		$this->albums              = $albums;
		$this->cacheService        = $cacheService;
		$this->redirects           = $redirects;
		$this->productsFacade      = $productsFacade;
		$this->linkGenerator       = $linkGenerator;
		$this->sites               = $sites;
	}

	/**
	 * @param array|int $ids
	 * @param int       $manufacturerId
	 *
	 * @return bool
	 */
	public function setManufacturer($ids, $manufacturerId)
	{
		try {
			$manufacturer = $this->em->getReference(Manufacturer::class, $manufacturerId);
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setManufacturer($manufacturer);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/**
	 * @param int[]|int $ids
	 * @param int       $vatRateId
	 *
	 * @return bool
	 */
	public function setVatRate($ids, $vatRateId)
	{
		try {
			$vatRate = $this->em->getReference(VatRate::class, $vatRateId);
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setVateRate($vatRate);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function setAvailabilityAfterSoldOut(array $ids, $av): bool
	{
		try {
			$availability = $av ? $this->em->getReference(Availability::class, $av) : null;
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->availabilityAfterSoldOut = $availability;
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/** Vrati seznam produktu, ktere obsahuji dany retezec
	 *
	 * @param string|null $term
	 * @param int  $hydrate
	 *
	 * @return ArrayCollection|int|mixed|string
	 */
	public function getByTerm($term = null, $hydrate = Query::HYDRATE_OBJECT)
	{
		if (is_null($term))
			return new ArrayCollection();

		$qb = $this->getEr()->createQueryBuilder('p', 'p.id')
			->addSelect('pt')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vr')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('pt.name', 'ASC');

		if ($term) {
			$qb->orWhere('pt.name LIKE :term')
				->orWhere('p.code1 LIKE :term')
				->orWhere('p.code2 LIKE :term')
				->orWhere('p.ean LIKE :term')
				->orWhere('p.id LIKE :term')
				->setParameter('term', '%' . $term . '%');
		}

		$qb->andWhere('p.isDeleted = 0')
			->andWhere('p.disableListing = 0');

		return $qb->getQuery()->getResult($hydrate);
	}

	/**
	 * TODO REMOVE?
	 */
	public function getByCode(string $codeKey, string $code): ?Product
	{
		return $this->getEr()->createQueryBuilder('p')
			->addSelect('pt')
			->leftJoin('p.productTexts', 'pt')
			->where('p.' . $codeKey . ' = :code')
			->andWhere('p.isDeleted = 0')
			->setParameter('code', $code)
			->setMaxResults(1)
			->getQuery()->getOneOrNullResult();
	}

	public function deleteDuplicity()
	{
		try {
			set_time_limit(600);
			$data = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name as name, p.code1, p.code2, count(p) as q')
				->leftJoin('p.productTexts', 'pt')
				->andWhere('p.isDeleted = 0')
				->having('count(p) > 1')->groupBy('pt.name, p.code1, p.code2')->getQuery()->getResult();

			foreach ($data as $row) {
				$products = $this->getEr()->createQueryBuilder('p')->where('pt.name = :name AND p.code1 = :code1')
					->join('p.productTexts', 'pt')->orderBy('p.id')
					->andWhere('p.isDeleted = 0')
					->setParameters(['code1' => $row['code1'], 'name' => $row['name']]);

				if ($row['code2'] == null) {
					$products->andWhere('p.code2 IS NULL');
				} else {
					$products->andWhere('p.code2 = :code2')->setParameter('code2', $row['code2']);
				}

				$products = $products->getQuery()->getResult();

				if (count($products) >= 2) {
					array_pop($products);

					foreach ($products as $product) {
						/** @var Product $product */
						if ($product->getGallery()) {
							FileSystem::delete(WWW_DIR . $product->getGallery()->generatePath());

							$this->em->createQuery('DELETE FROM ' . Image::class . ' i WHERE i.album = :album')->execute(['album' => $product->getGallery()->getId()]);
							$this->em->createQuery('DELETE FROM ' . Album::class . ' a WHERE a.id = :album')->execute(['album' => $product->getGallery()->getId()]);
						}

						$this->em->remove($product);
						$this->em->flush();
					}

					$this->em->clear();
				}
			}
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	public function setDiscountDisabled(array $ids, $value): bool
	{
		try {
			foreach ($ids as $id) {
				/** @var Product $product */
				$product = $this->getReference($id);

				$product->discountDisabled = $value ? 1 : 0;
				$this->em->persist($product);
			}

			$this->em->flush();
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	/**
	 * @param array $ids
	 * @param int   $tagId
	 *
	 * @return bool
	 */
	public function setTag(array $ids, int $tagId): bool
	{
		try {
			$tag        = $this->em->getReference(Tag::class, $tagId);
			$activeTags = [];
			foreach ($this->em->getRepository(ProductTag::class)->createQueryBuilder('pt')
				         ->select('IDENTITY(pt.product) as product, IDENTITY(pt.tag) as tag')
				         ->where('pt.product IN (:ids)')
				         ->andWhere('pt.tag = :tag')
				         ->setParameters([
					         'ids' => $ids,
					         'tag' => $tagId,
				         ])->getQuery()->getArrayResult() as $row) {
				$activeTags[$row['product']] = $row['tag'];
			}

			foreach ($ids as $id) {
				if (isset($activeTags[$id]))
					continue;

				$product = $this->getReference($id);

				$productTag = new ProductTag($product, $tag);
				$this->em->persist($productTag);
			}

			$this->em->flush();

			$this->cacheService->defaultCache->clean([Cache::TAGS => ['productsByTag']]);
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	public function createVariant(int $id): Product
	{
		$product = $this->get($id);

		if (!$product->isVariant()) {
			$baseProductVariant                 = new ProductVariant($product, $this->getLastVariantId() + 1);
			$baseProductVariant->isDefault      = 1;
			$baseProductVariant->createdDefault = $product->getCreated();

			$this->em->persist($baseProductVariant);
		} else {
			if ($product->isVariant()->isDefault === 1)
				$baseProductVariant = $product->isVariant();
			else {
				foreach ($product->variants->toArray() as $v)
					if ($v->isDefault === 1) {
						$product            = $v->product;
						$baseProductVariant = $v;
						break;
					}
			}
		}

		if (!isset($baseProductVariant))
			throw new BadRequestException();

		$variant = new Product();

		$texts = [];
		foreach ($product->getTexts()->toArray() as $text) {
			$this->em->detach($text);
			/** @var ProductTexts $text */
			$text->setProduct($variant);
			$texts[] = $text;
		}

		foreach ($product->getProductTags()->toArray() as $tag) {
			$this->em->detach($tag);
			$variant->addProductTag($tag);
			$this->em->persist($tag);
		}

		$variant->setProductTexts($texts);

		if (!$product->getGallery()) {
			$album = new Album(UPLOADS_PATH . '/products');
			$this->em->persist($album);
			$product->setGallery($album);
		}

		$variant->setGallery($product->getGallery());
		$variant->setVateRate($product->getVateRate());

		if (Config::load('enableCountryPrices')) {
			$variant->prices              = new ArrayCollection($product->clonePrices($variant));
			$variant->priceLevelCountries = new ArrayCollection($product->clonePriceLevelCounties($variant));
		}

		$this->em->persist($variant);

		$productVariant                 = new ProductVariant($variant, $baseProductVariant->getVariantId());
		$productVariant->createdDefault = $baseProductVariant->createdDefault;
		$this->em->persist($productVariant);

		$this->em->flush();

		return $variant;
	}

	public function removeFromVariants(Product $product): bool
	{
		if ($product->getGallery() && $product->getVariantParent()->getId()
			&& $product->getGallery()->getId() === $product->getVariantParent()->getGallery()->getId()) {
			$newGallery = $this->albums->cloneAlbum($product->getGallery());
			$product->setGallery($newGallery);

			$this->em->persist($product);
			$this->em->flush();
		}

		$variantId = $product->isVariant()->getVariantId();

		$allVariants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
			->where('pv.variantId = :variantId')->setParameter('variantId', $variantId)
			->getQuery()->getResult();

		if (count($allVariants) == 2) {
			foreach ($allVariants as $v)
				$this->em->remove($v);
			$this->em->flush();
		} else {
			$this->em->remove($product->isVariant());
			$this->em->flush();
		}

		return true;
	}

	public function setAsMainVariant(int $prodId): bool
	{
		try {
			/** @var ProductVariant $variant */
			$variant = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
				->where('pv.product = :id')
				->setParameter('id', $prodId)
				->getQuery()->getOneOrNullResult();

			if (!$variant)
				return false;

			/** @var ProductVariant $parent */
			$parent = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
				->where('pv.variantId = :variantId')
				->andWhere('pv.isDefault = 1')
				->setParameter('variantId', $variant->getVariantId())
				->getQuery()->getOneOrNullResult();

			if (!$parent)
				return false;

			$variant->isDefault = 1;
			$parent->isDefault  = 0;

			$this->em->persist($variant);
			$this->em->persist($parent);
			$this->em->flush();

			$parentId = $parent->getProduct()->getId();
			$this->em->createQueryBuilder()->delete(ProductInSite::class, 'pis')
				->where('pis.product = ' . $prodId)->getQuery()->execute();
			$this->em->createQueryBuilder()->update(ProductInSite::class, 'pis')
				->set('pis.product', $prodId)
				->where('pis.product = ' . $parentId)->getQuery()->execute();
			$this->em->createQueryBuilder()->delete(ProductInSite::class, 'pis')
				->where('pis.product = ' . $parentId)->getQuery()->execute();

			$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
				->where('cp.product = ' . $prodId)->getQuery()->execute();
			$this->em->createQueryBuilder()->update(CategoryProduct::class, ' cp')
				->set('cp.product', $prodId)
				->where('cp.product = ' . $parentId)->getQuery()->execute();
			$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
				->where('cp.product = ' . $parentId)->getQuery()->execute();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function setAsVariantFor(int $prodId, int $parentId): bool
	{
		if ($prodId === $parentId)
			return false;

		$product = null;
		$parent  = null;
		foreach ($this->getEr()->createQueryBuilder('p')->addSelect('isVar')
			         ->where('p.id IN (:ids)')->setParameter('ids', [$parentId, $prodId])
			         ->andWhere('p.isDeleted = 0')
			         ->leftJoin('p.isVariant', 'isVar')
			         ->getQuery()->getResult() as $row) {
			/** @var Product $row */
			if ($row->getId() == $prodId)
				$product = $row;
			else
				$parent = $row;
		}

		if (!$product || !$parent)
			return false;

		if ($parent->isVariant() && $parent->isVariant()->isDefault === 0)
			return false;

		if (!$parent->isVariant()) {
			$parentVariant                 = new ProductVariant($parent, $this->getLastVariantId() + 1);
			$parentVariant->createdDefault = $parent->getCreated();
			$parentVariant->isDefault      = 1;

			$this->em->persist($parentVariant);
		} else {
			$parentVariant = $parent->isVariant();
		}

		if (!$product->getGallery()) {
			if (!$parent->getGallery()) {
				$album = new Album(UPLOADS_PATH . '/products');
				$this->em->persist($album);
				$parent->setGallery($album);
				$this->em->persist($parent);
			}

			$product->setGallery($parent->getGallery());
		}

		$product->setVateRate($parent->getVateRate());

		if (!$product->isVariant()) {
			$productVariant = new ProductVariant($product, $parentVariant->getVariantId());
			/** @var ProductTexts $prodText */
			$prodText = $product->getText();

			$productVariant->useName             = $prodText->name ? 1 : 0;
			$productVariant->useName2            = $prodText->name2 ? 1 : 0;
			$productVariant->usePrice            = $product->price ? 1 : 0;
			$productVariant->usePurchasePrice    = $product->purchasePrice ? 1 : 0;
			$productVariant->useShortDescription = $prodText->shortDescription ? 1 : 0;
			$productVariant->useDescription      = $prodText->description ? 1 : 0;
			$productVariant->usePriceLevels      = $product->getPriceLevels()->count() ? 1 : 0;
			$productVariant->createdDefault      = $parentVariant->createdDefault;

			if (Config::load('product.allowRetailPrice')) {
				$productVariant->useRetailPrice = $product->retailPrice ? 1 : 0;
			}
		} else {
			if ($product->isVariant()->isDefault === 1) {
				throw new \Exception('eshopCatalog.product.productIsVariantParent');
			}

			$productVariant = $product->isVariant();
			$productVariant->setVariantId($parentVariant->getVariantId());
		}

		$this->em->persist($productVariant);

		foreach ($product->getCategoryProducts()->toArray() as $row) {
			$this->em->remove($row);
		}

		foreach ($product->sites->toArray() as $row) {
			$this->em->remove($row);
		}

		$this->em->persist($product);

		$this->em->flush();

		$this->cacheService->productCache->clean([
			Cache::Tags => ['variants'],
		]);

		return true;
	}

	public function getLastVariantId(): int
	{
		$r = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('v')
			->select('MAX(v.variantId) as variantId')
			->getQuery()->setMaxResults(1)->getArrayResult();

		return isset($r[0]) ? (int) $r[0]['variantId'] : 0;
	}

	public function duplicateProduct(int $productId): ?int
	{
		$this->em->beginTransaction();
		try {
			$originProduct = $this->get($productId);
			$newProduct    = clone $originProduct;
			$this->em->persist($newProduct);

			// Texty
			foreach ($originProduct->getTexts()->toArray() as $text) {
				/** @var ProductTexts $text */
				$entity = clone $text;
				$entity->setProduct($newProduct);
				$this->em->persist($entity);
			}

			// Kategorie
			foreach ($originProduct->getCategoryProducts()->toArray() as $cp) {
				/** @var CategoryProduct $cp */
				$entity = new CategoryProduct($newProduct, $cp->getCategory());
				$this->em->persist($entity);
			}

			// Features
			foreach ($originProduct->getFeatureProducts()->toArray() as $fp) {
				/** @var FeatureProduct $fp */
				$entity = new FeatureProduct($newProduct, $fp->getFeatureValue());
				$this->em->persist($entity);
			}

			// Dynamic features
			foreach ($originProduct->dynamicFeatures as $df) {
				$entity = new DynamicFeatureProduct($newProduct, $df->getFeature(), $df->value);
				$this->em->persist($entity);
			}

			// Tags
			foreach ($originProduct->getProductTags()->toArray() as $tag) {
				/** @var ProductTag $tag */
				$entity            = new ProductTag($newProduct, $tag->getTag());
				$entity->validFrom = $tag->validFrom;
				$entity->validTo   = $tag->validTo;
				$this->em->persist($entity);
			}

			// Gallery
			$originGallery = $originProduct->getGallery();
			if ($originGallery) {
				$gallery = new Album($originGallery->basePath);
				$this->em->persist($gallery);

				foreach ($originGallery->getImages()->toArray() as $img) {
					/** @var Image $img */
					$image = clone $img;
					$image->setAlbum($gallery);
					$image->setCloneOf($img);
					$image->clear();
					$img->setLastUse();

					$this->em->persist($img);
					$this->em->persist($image);
				}

				$newProduct->setGallery($gallery);
			}

			// Price levels
			foreach ($originProduct->getPriceLevels()->toArray() as $pl) {
				/** @var ProductPriceLevel $pl */
				$entity        = new ProductPriceLevel($newProduct, $pl->getGroupId());
				$entity->price = $pl->price;
				$this->em->persist($entity);
			}

			// Sites
			foreach ($originProduct->sites->toArray() as $site) {
				/** @var ProductInSite $site */
				$entity = new ProductInSite($newProduct, $site->getSite());
				$entity->setActive($site->isActive() ? 1 : 0);
				$entity->category = $site->category;
				$this->em->persist($entity);
			}

			// Documents
			foreach ($originProduct->documents->toArray() as $doc) {
				/** @var ProductDocument $doc */
				$entity = new ProductDocument($doc->lang, $doc->name, $doc->file, $newProduct);
				$this->em->persist($entity);
			}

			// RelatedProducts
			foreach ($originProduct->getRelatedProducts()->toArray() as $rl) {
				/** @var RelatedProduct $rl */
				$entity = new RelatedProduct($newProduct, $rl->getProduct(), $rl->getGroup());
				$this->em->persist($entity);
			}

			if (Config::load('enableCountryPrices')) {
				$newProduct->prices              = new ArrayCollection($originProduct->clonePrices($newProduct));
				$newProduct->priceLevelCountries = new ArrayCollection($originProduct->clonePriceLevelCounties($newProduct));
			}

			$this->em->persist($newProduct);
			$this->em->flush();
			$this->em->commit();

			return $newProduct->getId();
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			Debugger::log($e, 'duplicateProduct');
		}

		return null;
	}

	/**
	 * @param int[]|int $prodIds
	 *
	 * @return bool
	 */
	public function remove($prodIds): bool
	{
		try {
			$allIds = is_array($prodIds) ? $prodIds : [$prodIds];

			ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
			ProductQuery::$ignoreProductPublish = true;
			$this->productsFacade->clearTemp();

			$soldOutAv = $this->em->getRepository(Availability::class)->findOneBy(['ident' => Availability::SOLD_OUT]);

			foreach (array_chunk($allIds, 250) as $ids) {
				$this->em->beginTransaction();
				$products = $this->productsFacade->getProducts($ids);

				$productsInCategories = [];
				foreach ($this->em->createQueryBuilder()->select('IDENTITY(pis.product) as product, IDENTITY(pis.site) as site, IDENTITY(pis.category) as category')
					         ->from(ProductInSite::class, 'pis')
					         ->where('pis.product IN (:ids)')
					         ->setParameters([
						         'ids' => $ids,
					         ])->getQuery()->getArrayResult() as $row) {
					$productsInCategories[$row['product']][$row['site']] = $row['category'];
				}

				$existRedirects = [];
				foreach ($this->em->createQueryBuilder()
					         ->select('r.id, r.relationValue, r.siteIdent, r.lang')
					         ->from(Redirect::class, 'r')
					         ->where('r.package = :package')
					         ->andWhere('r.relationKey = :relationKey')
					         ->andWhere('r.relationValue IN (:relationValue)')
					         ->setParameters([
						         'package'       => 'EshopCatalog',
						         'relationKey'   => 'Product',
						         'relationValue' => $ids,
					         ])->getQuery()->getArrayResult() as $row) {
					$existRedirects[$row['siteIdent']][$row['lang']][$row['relationValue']] = $row['id'];
				}

				foreach ($products as $product) {
					$urlsByLang = [];

					foreach ($product->inSites as $siteIdent) {
						$site                        = $this->sites->getSites(false)[$siteIdent];
						Sites::$currentIdentOverride = $site->getIdent();

						foreach ($site->getDomains() as $domain) {
							$lang = $domain->getLang();
							if (!$domain->isActive || isset($existRedirects[$siteIdent][$lang][$product->getId()])) {
								continue;
							}

							if (!isset($urlsByLang[$lang])) {
								Sites::$currentLangOverride = $lang;
								$url                        = new Url($this->linkGenerator->link('EshopCatalog:Front:Default:product', [
									'id'     => $product->getId(),
									'locale' => $lang,
								]));

								$urlsByLang[$lang] = ltrim($url->getPath(), '/');
							}

							$catUrl = $this->linkGenerator->link('EshopCatalog:Front:Default:category', [
								'id'     => $productsInCategories[$product->getId()][$siteIdent],
								'locale' => $lang,
							]);
							$catUrl = new Url($catUrl);
							$catUrl = ltrim($catUrl->getPath(), '/');

							if ($urlsByLang[$lang] !== $catUrl) {
								$redirect                = new Redirect($product->name, $urlsByLang[$lang], $catUrl);
								$redirect->package       = 'EshopCatalog';
								$redirect->relationKey   = 'Product';
								$redirect->relationValue = $product->getId();
								$redirect->siteIdent     = $siteIdent;
								$redirect->lang          = $lang;

								$this->em->persist($redirect);
							}
						}
					}

					$this->em->flush();

					Sites::$currentLangOverride  = null;
					Sites::$currentIdentOverride = null;
				}

				foreach ($ids as $id) {
					$entity = $this->get($id);

					if (!$entity) {
						return false;
					}

					if (class_exists(EshopStockExtension::class)) {
						foreach ([
							         ProductPrice::class,
							         ProductPriceLevelCountry::class,
							         DynamicFeatureProduct::class,
							         ProductTag::class,
							         ProductInSite::class,
							         ProductVariant::class,
							         ProductDocument::class,
							         ProductVideo::class,
							         ProductSpedition::class,
							         CategoryProduct::class,
							         FeatureProduct::class,
						         ] as $class) {
							$this->em->createQueryBuilder()->delete($class, 'e')
								->where('e.product = :product')
								->setParameter('product', $id)
								->getQuery()->execute();
						}

						$this->em->createQueryBuilder()->delete(ProductPriceLevel::class, 'e')
							->where('e.productId = :product')
							->setParameter('product', $id)
							->getQuery()->execute();

						$this->em->createQueryBuilder()->delete(RelatedProduct::class, 'e')
							->where('e.origin = :product')
							->setParameter('product', $id)
							->getQuery()->execute();

						$this->em->createQueryBuilder()->delete(ExtraField::class, 'e')
							->where('e.sectionName = :sectionName')
							->andWhere('e.sectionKey = :sectionKey')
							->setParameters([
								'sectionName' => Product::EXTRA_FIELD_SECTION,
								'sectionKey'  => $id,
							])->getQuery()->execute();

						if (class_exists(ProductExport::class)) {
							$this->em->createQueryBuilder()->delete(ProductExport::class, 'e')
								->where('e.id = :product')
								->setParameter('product', $id)
								->getQuery()->execute();
						}

						$entity->setAvailability($soldOutAv);
						$entity->setMoreData([]);
						$entity->availabilityAfterSoldOut = $soldOutAv;
						$entity->isPublished              = 0;
						$entity->position                 = null;
						$entity->countryOfOrigin          = null;

						$entity->isDeleted = 1;
						$this->em->persist($entity);

						if ($entity->getGallery()) {
							$this->albums->removeAlbum($entity->getGallery());
						}

						$this->em->persist($entity);
					} else {
						$this->em->remove($entity);
					}

					$this->em->flush($entity);
				}

				$this->em->commit();

				$cache = new Cache($this->cacheStorage, Redirect::CACHE_NAMESPACE);
				$cache->clean([Cache::All => true]);
			}
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			return false;
		}

		return true;
	}

	public function updateVariantCategories(Product $product, int $variantId)
	{
		$conn     = $this->em->getConnection();
		$variants = [];

		$conn->beginTransaction();
		try {
			$categories        = [];
			$variantCategories = [];

			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(pv.product) as id')
				         ->from(ProductVariant::class, 'pv')
				         ->where('pv.variantId = :variantId')
				         ->andWhere('pv.isDefault = :isDefault')
				         ->setParameters([
					         'variantId' => $variantId,
					         'isDefault' => 0,
				         ])->getQuery()->getArrayResult() as $row) {
				$variants[]                    = $row['id'];
				$variantCategories[$row['id']] = [];
			}

			if (!$variants)
				return;

			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id_product, id_category FROM eshop_catalog__category_product WHERE id_product IN (" . implode(',', [$product->getId()] + $variants) . ")") as $row) {
				if ($row['id_product'] == $product->getId()) {
					$categories[] = $row['id_category'];
				} else {
					$variantCategories[$row['id_product']][] = $row['id_category'];
				}
			}

			$ins = [];
			foreach ($variantCategories as $variantId => $variantCats) {
				foreach (array_diff($categories, $variantCats) as $catId)
					$ins[] = "($variantId, $catId)";
				foreach (array_diff($variantCats, $categories) as $catId)
					$conn->executeQuery("DELETE FROM `eshop_catalog__category_product` WHERE `id_product` = {$variantId} AND `id_category` = {$catId}");
			}

			if ($ins) {
				$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_catalog__category_product (id_product, id_category) VALUES " . implode(', ', $ins));
			}

			/** @var ProductInSite[][] $variantsSites */
			$variantsSites = [];
			/** @var ProductInSite[] $productSites */
			$productSites = [];
			foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('pis')
				         ->where('pis.product IN (' . implode(',', $variants) . ')')
				         ->getQuery()->getResult() as $row) {
				/** @var ProductInSite $row */
				$variantsSites[$row->getProduct()->getId()][$row->getSite()->getIdent()] = $row;
			}

			foreach ($product->sites as $site) {
				$productSites[$site->getSite()->getIdent()] = $site;
			}

			// Add
			foreach ($variants as $variantId) {
				foreach (array_diff_key($productSites, $variantsSites[$variantId] ?? []) as $v) {
					/** @var ProductInSite $v */
					$pis           = new ProductInSite(
						$this->em->getReference(Product::class, $variantId),
						$v->getSite(),
					);
					$pis->category = $v->category;
					$pis->setActive($v->isActive());

					$this->em->persist($pis);
				}
			}

			$this->em->flush();
			$conn->commit();

			$this->cacheService->productCache->clean([
				Cache::Tags => ['variants'],
			]);
		} catch (\Exception $e) {
			if ($conn->isTransactionActive())
				$conn->rollBack();
		}
	}

	public function allowImportPrice(array $ids, bool $allow): bool
	{
		if (empty($ids)) {
			return false;
		}

		try {
			$iterator = $this->em->getRepository(Product::class)->createQueryBuilder('p')
				->where('p.id IN (' . implode(',', $ids) . ')')
				->getIterator();

			$i = 0;
			while (($product = $iterator->next()) !== false) {
				bdump($product);
				/** @var Product $product */
				$product = $product[0];
				$product->setMoreDataValue('stopImportPrice', (int) !$allow);
				$this->em->persist($product);
				$i++;

				if ($i % 20 === 0) {
					$this->em->flush();
					$this->em->clear();
					gc_collect_cycles();
				}
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
			Debugger::log($e);
		}

		return false;
	}
}
