<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Application\AppState;
use Core\Model\Entities\EntityRepository;
use Core\Model\Helpers\Arrays;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\ProductPriceLevelCountry;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Helpers\Helper;
use EshopCatalog\Model\Helpers\ProductsHelper;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\GroupCustomers;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\ProductsPriceLevel;
use Nette\Application\LinkGenerator;
use Core\Model\Event\DaoEvent;
use Core\Model\Sites;
use EshopCatalog\Model\Entities\ProductInSite;
use Core\Model\Event\EventDispatcher;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Customer;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Users\Model\Http\UserStorage;
use Users\Model\Users;

/**
 * TODO možnost přesunutí do fasády
 * @method EntityRepository getEr()
 */
class Products extends BaseFrontEntityService
{
	const CACHE_NAMESPACE       = 'eshopCatalogProducts';
	const CACHE_PRICE_NAMESPACE = 'eshopCatalogPrice';

	const MODE_ESHOP    = 'eshop';
	const MODE_CHECKOUT = 'checkout';

	protected static string $mode = self::MODE_ESHOP;

	public static ?string $loadKey = null;

	protected $entityClass = Product::class;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var Albums @inject */
	public $albumsService;

	/** @var Manufacturers @inject */
	public $manufacturersService;

	/** @var Sites @inject */
	public $sitesService;

	public FeatureProducts        $featureProductsService;
	protected CacheService        $cacheService;
	protected Users               $users;
	protected UserStorage         $userStorage;
	protected EventDispatcher     $eventDispatcher;
	protected Config              $config;
	protected AvailabilityService $availabilityService;
	protected ProductDocuments    $productDocuments;
	protected ProductVideos       $productVideos;
	protected Tags                $tagsService;
	protected Helper              $helper;
	protected AppState            $appState;
	protected Customers           $customers;
	protected GroupCustomers      $groupCustomers;
	protected ProductsPriceLevel  $productsPriceLevel;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var callable[] */
	public $customQB = [];

	protected array  $cProductsIdInCategory = [];
	protected array  $cProductsIdBySearch   = [];
	protected array  $cProductsIdAll        = [];
	protected array  $cGroupCustomer        = [];
	protected ?array $cExtraFields          = null;
	protected array  $cQuantity             = [];

	protected ?array $cExtraFields2 = null;

	protected array $cForNavigation = [];

	/** @var callable[] */
	public $onAfterProductsGet = [];

	/** @var array */
	public $cacheDep = [
		Cache::TAGS   => ['products'],
		Cache::EXPIRE => '15 minutes',
	];

	public function __construct(
		FeatureProducts     $featureProducts,
		CacheService        $cacheService,
		ProductDocuments    $productDocuments,
		Users               $users,
		UserStorage         $userStorage,
		Config              $config,
		EventDispatcher     $eventDispatcher,
		AvailabilityService $availabilityService,
		Helper              $helper,
		ProductVideos       $productVideos,
		Tags                $tags,
		AppState            $appState,
		Customers           $customers,
		GroupCustomers      $groupCustomers,
		ProductsPriceLevel  $productsPriceLevel
	)
	{
		$this->featureProductsService = $featureProducts;
		$this->cacheService           = $cacheService;
		$this->userStorage            = $userStorage;
		$this->config                 = $config;
		$this->eventDispatcher        = $eventDispatcher;
		$this->availabilityService    = $availabilityService;
		$this->productDocuments       = $productDocuments;
		$this->productVideos          = $productVideos;
		$this->tagsService            = $tags;
		$this->users                  = $users;
		$this->helper                 = $helper;
		$this->appState               = $appState;
		$this->customers              = $customers;
		$this->groupCustomers         = $groupCustomers;
		$this->productsPriceLevel     = $productsPriceLevel;

		$this->onAfterProductsGet[] = function($products) {
			/** @var Dao\Product[] $products */
			$this->loadPriceLevels($products,
				$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);
		};
	}

	public function findIdByAlias(string $alias): ?array
	{
		return $this->em->createQueryBuilder()
			->select('IDENTITY(pt.id) as id, IDENTITY(site.category) as category')
			->from(ProductTexts::class, 'pt')
			->innerJoin('pt.id', 'p')
			->innerJoin('p.sites', 'site', Join::WITH, 'site.site = :site')
			->where('pt.alias = :alias')
			->setParameters([
				'alias' => $alias,
				'site'  => $this->sitesService->getCurrentSite()->getIdent(),
			])
			->getQuery()->getArrayResult() ?: [];
	}

	public function getForNavigation(int $id, bool $withCategoryId = false, ?string $lang = null)
	{
		if (!$lang) {
			$lang = $this->translator->getLocale();
		}

		if (!array_key_exists($lang, $this->cForNavigation)) {
			$this->cForNavigation[$lang] = [];
		}

		if (($this->cForNavigation[$lang][$id] ?? null) !== null) {
			return $this->cForNavigation[$lang][$id];
		}

		$forLoad = [];
		foreach (array_merge(ProductsFacade::$allLoadedIds, [$id]) as $lId) {
			if (($this->cForNavigation[$lang][$lId] ?? null) === null) {
				$forLoad[]                         = $lId;
				$this->cForNavigation[$lang][$lId] = false;
			}
		}

		foreach (array_chunk($forLoad, 900) as $chunk) {
			$qb = $this->em->createQueryBuilder()->select('IDENTITY(pt.id) as id, pt.name, pt.alias, pt.lang')
				->from(ProductTexts::class, 'pt')
				->where('pt.id IN (' . implode(',', $chunk) . ')');

			if ($withCategoryId) {
				$qb->innerJoin('pt.id', 'p')
					->innerJoin('p.sites', 'site', Join::WITH, 'site.site = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent())
					->addSelect('IDENTITY(site.category) as category');
			}

			$resultCache = Config::load('productsList.enableResultCacheOther');

			$query = $qb->getQuery();
			if ($resultCache) {
				$query->enableResultCache($resultCache);
			}

			foreach ($query->getArrayResult() as $row) {
				$this->cForNavigation[$row['lang']][$row['id']] = $row;
			}
		}

		return $this->cForNavigation[$lang][$id] ?: null;
	}

	/**
	 * @param int|int[] $categoryId
	 *
	 * @return int[]
	 */
	public function getProductsIdInCategory($categoryId, $start = 0, $limit = null, $sort = null, $filter = [], bool $orderByPosition = true)
	{
		$lang    = $this->translator->getLocale();
		$site    = $this->sitesService->getCurrentSite();
		$keyBase = $site->getIdent() . '/' . md5(serialize([$categoryId, $sort, $filter,
				$orderByPosition])) . '/' . $lang;

		if (isset($this->cProductsIdInCategory[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

			return $this->cProductsIdInCategory[$keyBase];
		} else {
			$query = (new ProductQuery($lang))
				->selectIds()
				->disableAutoIdSort(true)
				->disableListing(0);

			if ($orderByPosition)
				$query->orderByPosition();

			if (self::$mode == self::MODE_ESHOP) {
				$query->inSite($site->getIdent());
			}

			$query2 = clone $query;

			$query->inSiteCategory($categoryId);
			$query2->inOtherCategories($categoryId);

			$ids = $this->processProductsResult($query, $filter, $sort, 'default', $orderByPosition);
			$ids = $ids + $this->processProductsResult($query2, $filter, $sort, 'default', $orderByPosition);

			if (is_null($sort) || $sort == '' || $sort === 'recommended') {
				uasort($ids, fn($a, $b) => $a['cPosition'] <=> $b['cPosition']);
				$ids                                   = array_unique(array_merge($this->getRecommendedIds($categoryId, 0, null, null, $filter), array_keys($ids)));
				$this->cProductsIdInCategory[$keyBase] = $ids;
			} else {
				uasort($ids, fn($a, $b) => $a['sort'] <=> $b['sort']);
				$ids = array_keys($ids);
				$tmp = preg_split('/\s|\+/', $sort);
				if (isset($tmp[1]) && strtolower($tmp[1]) == 'desc') {
					$ids = array_reverse($ids);
				}
				$this->cProductsIdInCategory[$keyBase] = $ids;
			}
		}

		if ($limit)
			return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

		return $this->cProductsIdInCategory[$keyBase];
	}

	/**
	 * @param int        $categoryId
	 * @param array|null $filter
	 *
	 * @return int
	 */
	public function getProductsInCategoryCount($categoryId, $filter = [])
	{
		return count($this->getProductsIdInCategory($categoryId, 0, null, null, $filter));
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdBySearch($q, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$q, $sort, $filter]));

		if (isset($this->cProductsIdBySearch[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

			return $this->cProductsIdBySearch[$keyBase];
		} else {
			$query            = (new ProductQuery($this->defaultLang->locale));
			$query->siteIdent = $this->sitesService->getCurrentSite()->getIdent();
			$query->search($q)
				->selectIds()
				->disableListing(0)
				->disableAutoIdSort()
				->inSite($query->siteIdent);

			$ids = $this->processProductsResult($query, $filter, $sort, 'search');

			if (is_null($sort) || $sort == '' || $sort === 'recommended') {
				$this->cProductsIdBySearch[$keyBase] = array_unique(array_merge(
					$this->getRecommendedIds($q, 0, null, null, $filter, 'search'),
					array_keys($ids),
				));
			} else {
				uasort($ids, static fn($a, $b) => $a['sort'] <=> $b['sort']);
				$ids = array_keys($ids);
				$tmp = explode(' ', (string) $sort);
				if (isset($tmp[1]) && strtolower($tmp[1]) === 'desc') {
					$ids = array_reverse($ids);
				}

				$this->cProductsIdBySearch[$keyBase] = $ids;
			}
		}

		if ($limit)
			return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

		return $this->cProductsIdBySearch[$keyBase];
	}

	protected function processProductsResult(ProductQuery $query, $filter, $sort, string $type = 'default', bool $orderByPosition = true): array
	{
		$ids               = [];
		$enableResultCache = Config::load('productsList.enableResultCache', 0);
		$availabilities    = $this->availabilityService->getAll();
		$showOnlyStock     = Config::load('onlyStockProductsInList', true);
		$avSort            = [1 => [], 0 => []];

		foreach ($availabilities as $av) {
			$availabilities[$av->getIdent()]       = $av;
			$avSort[$av->canAddToCart() ? 1 : 0][] = $av->getId();
		}

		if ($filter) {
			$query->useFilter($this->prepareFilter($filter));
		}

		if (is_string($sort)) {
			$sort = urldecode($sort);
		}

		$hideIfCountryPriceMissing = Config::load('product.hideIfCountryPriceMissing', []);
		if (Config::load('enableCountryPrices') && !empty($hideIfCountryPriceMissing)) {
			$country = $this->appState->getCountry();
			if (Arrays::contains($hideIfCountryPriceMissing, $country)) {
				$query->countryPriceFilled($country);
			}
		}

		$sort = str_replace('+', ' ', $sort);
		$this->prepareSort($query, $sort);

		$isAutoIdSortDisabled = $query->disableAutoIdSort;
		$qb                   = $query->getQueryBuilder($this->getEr());
		$qb->andWhere('p.isPublished = 1');
		foreach ($this->customQB as $cqb) {
			$cqb($qb);
		}

		$qb->addSelect('IDENTITY(p.availability) as av');

		if (!$sort || $sort === 'recommended' && !empty($avSort[1])) {
			$qb->addSelect('CASE WHEN p.availability IN (' . implode(',', $avSort[1]) . ') THEN 1 ELSE 0 END as avSort')
				->addOrderBy('avSort', 'DESC');
		}

		if ($sort && $sort !== 'recommended') {
			$tmp = explode(' ', $sort);
			$qb->addSelect($tmp[0] . ' as sort');
		}

		if ($isAutoIdSortDisabled) {
			$qb->addOrderBy('p.id', 'DESC');
		}

		$query = $qb->getQuery();
		if ($enableResultCache) {
			$query->enableResultCache($enableResultCache);
		}

		foreach ($query->getScalarResult() as $v) {
			if (!isset($v['av']) || empty($v['av'])) {
				$v['av'] = Availability::SOLD_OUT;
			}

			if ($type === 'search') {
				if ($showOnlyStock && (!isset($availabilities[$v['av']]) || !$availabilities[$v['av']]->canShowOnSearch())) {
					continue;
				}
			} else {
				if ($showOnlyStock && (!isset($availabilities[$v['av']]) || !$availabilities[$v['av']]->canShowOnList())) {
					continue;
				}
			}

			$ids[$v['id']] = $v;
		}

		return $ids;
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$site      = $this->sitesService->getCurrentSite();
		$keySuffix = $sort === null && empty($filter) ? 'all' : md5(serialize([$sort, $filter]));
		$keyBase   = $site->getIdent() . '_' . $keySuffix;

		if (isset($this->cProductsIdAll[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

			return $this->cProductsIdAll[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($sort, $filter, $site) {
				$dep = $this->cacheDep;

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->selectIds()
					->inSite($site->getIdent())
					->orderByPosition();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdAll[$keyBase] = array_unique(array_merge($this->getRecommendedIds(null, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdAll[$keyBase] = $ids;;
		}

		if ($limit)
			return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

		return $this->cProductsIdAll[$keyBase];
	}

	public function getRecommendedIds($q = null, $start = 0, $limit = null, $sort = null, $filter = [], $type = 'category')
	{
		$site = $this->sitesService->getCurrentSite();

		$allTags = $this->tagsService->getAll();
		$tag     = $allTags[Config::load('productsList.defaultSortTopTag')];
		if (!$tag)
			return [];

		$query            = (new ProductQuery($this->defaultLang->locale));
		$query->siteIdent = $this->sitesService->getCurrentSite()->getIdent();
		$query->withTag($tag->id)
			->inSite($query->siteIdent)
			->selectIds();

		if (Config::load('onlyStockProductsInList', true) == true)
			$query->onlyInStockOrSupplier();

		if ($filter)
			$query->useFilter($this->prepareFilter($filter));

		$ids = [];
		switch ($type) {
			case 'category':
				if ($q) {
					$query2 = clone $query;

					if (self::$mode == self::MODE_ESHOP)
						$query->inSite($site->getIdent());
					$query->inSiteCategory($q);
					$query2->inOtherCategories($q);
					$query->orderByPosition();
					$query2->orderByPosition();

					$ids = array_merge($this->endGetRecommended($query, $limit, $start), $this->endGetRecommended($query2, $limit, $start));
					$ids = array_unique($ids);
				}
				break;
			case 'search':
				$query->siteIdent = $this->sitesService->getCurrentSite()->getIdent();
				$query->search($q);

				$ids = $this->endGetRecommended($query, $limit, $start);

				break;
		}

		return $ids;
	}

	protected function endGetRecommended($query, $limit, $start): array
	{
		$qb = $query->getQueryBuilder($this->getEr());
		foreach ($this->customQB as $cqb)
			$cqb($qb);

		$ids = [];

		$resultCache = Config::load('productsList.enableResultCacheOther', 0);
		$query       = $qb->getQuery();
		if ($resultCache)
			$query->enableResultCache($resultCache);

		foreach ($query->setMaxResults($limit)->setFirstResult($start)->getScalarResult() as $v)
			$ids[] = $v['id'];

		return $ids;
	}

	/**
	 * @param Dao\Product[] $products
	 * @param null          $userId
	 * @param bool          $loadRetailPrice
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadFresh(
		&$products,
		$userId = null,
		bool $loadRetailPrice = true
	): void
	{
		if (empty($products))
			return;

		$ids = array_values(array_map(fn($p) => $p->getId(), $products));

		foreach (array_chunk($ids, 900) as $chunk) {
			$qb = $this->getEr()->createQueryBuilder('p')
				->select('p.id, p.isPublished, p.price, IDENTITY(p.availability) as availability');

			if (Config::load('product.allowRetailPrice')) {
				$qb->addSelect('p.retailPrice');
			}

			if (count($chunk) === 1) {
				$qb->andWhere('p.id = ' . $chunk[0]);
			} else {
				$qb->andWhere('p.id IN (' . implode(',', $chunk) . ')');
			}

			$qb->andWhere('p.isDeleted = 0');

			if (Config::load('enableCountryPrices') && $this->appState->getCountry()) {
				$qb->leftJoin('p.prices', 'cp', Join::WITH, 'cp.country = :country')
					->setParameter('country', $this->appState->getCountry())
					->leftJoin('cp.vatRate', 'cpv')
					->addSelect('cp.price as cPrice, cp.currency as cCurrency, cpv.rate as cVatRate');

				if (!Config::load('enablePriceHistory')) {
					$qb->addSelect('cp.retailPrice as cRetailPrice');
				}
			}

			$query   = $qb->getQuery();
			$country = $this->appState->getCountry();

			if (Config::load('productsList.enableFreshResultCache')) {
				$query->enableResultCache(Config::load('productsList.enableFreshResultCache'));
			}

			foreach ($query->getArrayResult() as $row) {
				if (!isset($products[$row['id']])) {
					continue;
				}

				if (!$row['isPublished']) {
					unset($products[$row['id']]);
					continue;
				}

				// Skryti produktu pokud neobsahuje cenu pro zemi
				if (Arrays::contains(Config::load('product.hideIfCountryPriceMissing', []), $country) && $row['cPrice'] === null) {
					unset($products[$row['id']]);
					continue;
				}

				$price             = (float) $row['price'];
				$retailPrice       = null;
				$canUseRetailPrice = Config::load('product.allowRetailPrice') && !Config::load('enablePriceHistory');

				if ($canUseRetailPrice) {
					$retailPrice = (float) $row['retailPrice'];
				}

				if ($row['cPrice']) {
					$price = (float) $row['cPrice'];

					if ($canUseRetailPrice) {
						$retailPrice = (float) $row['cRetailPrice'];
					}

					if ($row['cCurrency']) {
						$products[$row['id']]->currency = $row['cCurrency'];
					}
				}

				if ($row['cVatRate'] !== null) {
					$products[$row['id']]->setVatRate($row['cVatRate']);
				}

				if (Config::load('product.priceIsWithoutVat', false)) {
					$price = ProductsHelper::processPrice((float) $price, $products[$row['id']]->getVatRate());

					if ($retailPrice !== null && $canUseRetailPrice) {
						$retailPrice = ProductsHelper::processPrice((float) $retailPrice, $products[$row['id']]->getVatRate());
					}
				}

				$products[$row['id']]->setPrice($price);
				$products[$row['id']]->setBasePrice($price);

				if ($retailPrice !== null && $canUseRetailPrice) {
					$products[$row['id']]->setRetailPrice($retailPrice);
					$products[$row['id']]->retailPriceInBaseCurrency = $retailPrice;
				}

				$products[$row['id']]->priceInBaseCurrency     = $price;
				$products[$row['id']]->basePriceInBaseCurrency = $price;
				$preorderAv                                    = $this->availabilityService->getByIdent(Availability::PREORDER);
				if ($products[$row['id']]->unlimitedQuantity && $preorderAv && !in_array($row['availability'], [
						$preorderAv->getId(),
					])) {
					$products[$row['id']]->setAvailability($this->availabilityService->getByIdent(Availability::IN_STOCK));
				} else if ($row['availability']) {
					$products[$row['id']]->setAvailability($this->availabilityService->get((int) $row['availability']));
				} else {
					$products[$row['id']]->setAvailability($this->availabilityService->getByIdent(Availability::SOLD_OUT));
				}
			}
		}

		if ($loadRetailPrice) {
			$this->loadRetailPrice($products);
		}

		$this->loadPriceLevels($products, $userId);
		$this->loadQuantity($products);
		$this->loadSite($products);

		$this->eventDispatcher->dispatch(new ProductsEvent($products), Products::class . '::afterLoadFresh');
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @return void
	 */
	public function loadRetailPrice(array &$products): void
	{
		if (!Config::load('enablePriceHistory')) {
			return;
		}

		$conn            = $this->em->getConnection();
		$keys            = [];
		$keysCountry     = [];
		$whereIds        = [];
		$whereCountryIds = [];
		$country         = Config::load('enableCountryPrices') && $this->appState->getCountry()
			? $this->appState->getCountry()
			: null;

		foreach ($products as $product) {
			$keys[$product->getId()] = 'retailPrice/' . $product->getId();

			if ($country) {
				$keysCountry[$product->getId()] = 'retailPriceCountry/' . $country . '/' . $product->getId();
			}
		}

		if (!empty($keysCountry)) {
			foreach ($this->cacheService->priceCache->bulkLoad($keysCountry) as $key => $retailPrice) {
				$tmp = explode('/', $key);
				$id  = $tmp[2];

				if (is_float($retailPrice)) {
					$products[$id]->retailPrice               = $retailPrice;
					$products[$id]->retailPriceInBaseCurrency = $retailPrice;
					unset($keys[$id]);
				} else if ($retailPrice === false) {

				} else {
					$whereCountryIds[$id] = [
						'price' => $products[$id]->price,
					];
				}
			}
		}

		if (!empty($whereCountryIds)) {
			foreach ($whereCountryIds as $id => $v) {
				$currency = $products[$id]->currency;
				// Vytvoření proměnné kdy se vyskytuje navýšení ceny
				$conn->executeStatement("set @compareId = (SELECT id FROM `eshop_catalog__product_price_history_country`
					where direction = 1 and product = {$id} and country = '{$country}' and currency_code " . ($currency ? "= '{$currency}'" : " IS NULL") . "
					order by created desc
					limit 1)");

				// Porovnání nejnižší ceny za 30 dní vs stálá sleva podle navýšení
				$stmt        = $conn->executeQuery("SELECT case 
                        when s2.created < s1.created then s2.price 
                        else s1.price 
                    end as price
					FROM (SELECT * FROM `eshop_catalog__product_price_history_country`
					where id > case
					    when @compareId is not null then @compareId
					    else 0
				    end
					and product = {$id} and country = '{$country}' and currency_code " . ($currency ? "= '{$currency}'" : " IS NULL") . "
					order by created asc
					limit 1) as s1
					inner join (SELECT id, created, price FROM `eshop_catalog__product_price_history_country`
					where created >= date(now() - interval 30 day)
					and product = {$id} and country = '{$country}' and currency_code " . ($currency ? "= '{$currency}'" : " IS NULL") . "
					order by price ASC
					limit 1) as s2
					LIMIT 1");
				$retailPrice = $stmt->fetchOne();

				$retailPrice = $retailPrice === false
					? null
					: (float) $retailPrice;

				$products[$id]->retailPrice               = $retailPrice;
				$products[$id]->retailPriceInBaseCurrency = $products[$id]->retailPrice;
				unset($keys[$id]);

				$this->cacheService->priceCache->save('retailPriceCountry/' . $country . '/' . $id, $retailPrice ?: false, [
					Cache::EXPIRATION => '3 hours',
				]);
			}
		}

		if (!empty($keys)) {
			foreach ($this->cacheService->priceCache->bulkLoad($keys) as $key => $retailPrice) {
				$tmp = explode('/', $key);
				$id  = $tmp[1];

				if (is_float($retailPrice)) {
					$products[$id]->retailPrice               = $retailPrice;
					$products[$id]->retailPriceInBaseCurrency = $retailPrice;
				} else if ($retailPrice === false) {
					$products[$id]->retailPrice               = null;
					$products[$id]->retailPriceInBaseCurrency = null;
				} else {
					$whereIds[$id] = [
						'price' => $products[$id]->price,
					];
				}
			}
		}

		if (!empty($whereIds)) {
			foreach ($whereIds as $id => $v) {
				// Vytvoření proměnné kdy se vyskytuje navýšení ceny
				$conn->executeStatement("set @compareId = (SELECT id FROM `eshop_catalog__product_price_history`
					where direction = 1 and product = {$id}
					order by created desc
					limit 1)");

				// Porovnání nejnižší ceny za 30 dní vs stálá sleva podle navýšení
				$stmt        = $conn->executeQuery("SELECT case 
                        when s2.created < s1.created then s2.price 
                        else s1.price 
                    end as price
					FROM (SELECT * FROM `eshop_catalog__product_price_history`
					where id > case
					    when @compareId is not null then @compareId
					    else 0
				    end
					and product = {$id}
					order by created asc
					limit 1) as s1
					inner join (SELECT id, created, price FROM `eshop_catalog__product_price_history`
					where created >= date(now() - interval 30 day)
					and product = {$id}
					order by price ASC
					limit 1) as s2
					LIMIT 1");
				$retailPrice = $stmt->fetchOne();

				$retailPrice = $retailPrice === false
					? null
					: (float) $retailPrice;

				$products[$id]->retailPrice               = $retailPrice;
				$products[$id]->retailPriceInBaseCurrency = $products[$id]->retailPrice;

				$this->cacheService->priceCache->save('retailPrice/' . $id, $retailPrice ?: false, [
					Cache::EXPIRATION => '3 hours',
				]);
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 */
	public function loadSite(&$products)
	{
		$ids      = array_map(function($product) { return $product->getId(); }, $products);
		$variants = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			$qb = $this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
				->select('IDENTITY(ps.product) as prod, IDENTITY(ps.site) as site, IDENTITY(ps.category) as cat, ps.isActive')
				->where('ps.product IN (:ids)')
				->andWhere('ps.isActive = 1')
				->setParameters([
					'ids' => $chunk,
				]);

			if (self::$mode == self::MODE_ESHOP)
				$qb->andWhere('ps.site = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());

			$resultCache = Config::load('productsList.enableResultCacheOther', 0);
			$query       = $qb->getQuery();
			if ($resultCache && self::$loadKey == 'related')
				$query->enableResultCache($resultCache);

			foreach ($query->getScalarResult() as $row) {
				$products[$row['prod']]->defaultCategoryId = (int) $row['cat'];
				$products[$row['prod']]->isActive          = (bool) $row['isActive'];
			}

			foreach ($products as $product) {
				if ($product->variantOf)
					$variants[$product->variantOf][] = $product->getId();
			}

			// Projduti variant
			foreach ($variants as $k => $vals) {
				if (!isset($products[$k]))
					continue;

				foreach ($vals as $v) {
					$products[$v]->defaultCategoryId = $products[$k]->defaultCategoryId;
					$products[$v]->isActive          = $products[$k]->isActive;
				}
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 * @param int|null      $userId
	 */
	public function loadPriceLevels(&$products, $userId = null): void
	{
		if (!$userId || !Config::load('enablePriceLevels', false)) {
			return;
		}

		$customer = $this->customers->getByUser((int) $userId);
		if (!$customer) {
			return;
		}

		$ids    = [];
		$prices = [];
		foreach ($products as $product) {
			$ids[$product->getId()]    = $product->getId();
			$prices[$product->getId()] = $product->getPrice();
		}

		if ($customer->getGroupCustomers()) {
			// Pokud je uživatel ve skupině a je u produktu nastavena cenová hladina, aplikujeme ji
			if (!isset($this->cGroupCustomer[$userId])) {
				$this->cGroupCustomer[$userId] = $this->groupCustomers->getGroupArray($customer->getGroupCustomers()->getId());
			}

			$group = $this->cGroupCustomer[$userId] ?? null;
			if ($group['id']) {
				$oIds              = $ids;
				$priceLevelGroupId = $group['id'];

				if ($group['useProductsSaleOnGroup'] && EshopOrdersConfig::load('customerGroup.allowUseSaleOnOtherGroupPrices')) {
					foreach ($this->productsPriceLevel->getPricesForGroup($oIds, $group['useProductsSaleOnGroup']) as $k => $v) {
						$prices[$k]             = $v;
						$products[$k]->currency = null;
					}
				}

				if ($group['inheritPricesGroup'] && EshopOrdersConfig::load('customerGroup.allowInheritGroupPrices')) {
					$priceLevelGroupId = $group['inheritPricesGroup'];
				}

				foreach ($this->productsPriceLevel->getPricesForGroup($oIds, $priceLevelGroupId) as $k => $v) {
					$prices[$k]             = (float) $v;
					$products[$k]->currency = null;
					unset($ids[$k]);
				}

				if (Config::load('enableCountryPrices') && $this->appState->getCountry()) {
					$query = $this->em->getRepository(ProductPriceLevelCountry::class)->createQueryBuilder('pplc')
						->select('IDENTITY(pplc.product) as product, pplc.price, pplc.currency')
						->where('pplc.product IN (' . implode(',', array_values($oIds)) . ')')
						->andWhere('pplc.group = :groupId')
						->andWhere('pplc.country = :country')
						->setParameters([
							'groupId' => $priceLevelGroupId,
							'country' => $this->appState->getCountry(),
						])
						->getQuery();

					if (Config::load('productsList.enableFreshResultCache')) {
						$query->enableResultCache(Config::load('productsList.enableFreshResultCache'));
					}

					foreach ($query->getArrayResult() as $row) {
						if ($row['price']) {
							$prices[$row['product']]             = (float) $row['price'];
							$products[$row['product']]->currency = $row['currency'];
							unset($ids[$row['product']]);
						}
					}
				}
			}

			// Pokud je ve skupině nastavená sleva na všechny produkty, aplikuejeme ji na produkty. Buď na již zlevněné nebo ne
			if ($group['productsSale'] > 0) {
				$tmp = 1 - ((float) $group['productsSale'] / 100);

				$tmpLoop = $ids;
				if (EshopOrdersConfig::load('customerGroup.allowChangeProductSaleBasePrice') && $group['productsSaleUseGroupPrice']) {
					$tmpLoop = array_keys($prices);
				}

				foreach ($tmpLoop as $id) {
					$prices[$id] *= $tmp;
					unset($ids[$id]);
				}

				unset($tmpLoop);
			}
		}

		// Na neupravené produkty aplikujeme slevu hromadnou
		$user             = $this->users->get($userId);
		$registerUserSale = $this->helper->getRegisterUserSale($user);
		if ($registerUserSale > 0) {
			$tmp = 1 - ((float) $registerUserSale / 100);

			foreach ($ids as $id) {
				$prices[$id] *= $tmp;
				unset($ids[$id]);
			}
		}

		// Aplikujeme slevu na produkty
		foreach ($prices as $product => $price) {
			if (Config::load('product.priceIsWithoutVat', false))
				$price = ProductsHelper::processPrice((float) $price, $products[$product]->getVatRate());

			$products[$product]->price               = $price;
			$products[$product]->priceInBaseCurrency = $price;
		}
	}

	/**
	 * @param Dao\Product[]|Dao\Product $products
	 */
	public function loadFeatures(&$products): void
	{
		if ($products instanceof Dao\Product)
			$products->setFeatures($this->featureProductsService->getFeaturesForProduct([$products])[$products->getId()] ?? []);
		else {
			foreach ($this->featureProductsService->getFeaturesForProduct($products) as $k => $v)
				$products[$k]->setFeatures($v);
		}
	}

	/**
	 * @param Dao\Product[] $products
	 */
	public function loadDocuments(array &$products): void
	{
		$ids = array_map(fn(Dao\Product $product) => $product->getId(), $products);

		foreach ($this->productDocuments->getDocuments($ids) as $productId => $documents) {
			foreach ($documents as $document) {
				$products[$productId]->addDocument($document);
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 */
	public function loadVideos(array &$products): void
	{
		$ids = array_map(static fn(Dao\Product $product) => $product->getId(), $products);

		foreach ($this->productVideos->getVideos($ids) as $productId => $videos) {
			foreach ($videos as $video) {
				$products[$productId]->addVideo($video);
			}
		}
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 * @param bool                      $forceLoadData
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadQuantity(&$product, bool $forceLoadData = false)
	{
		if ($forceLoadData)
			$this->cQuantity = [];

		$id = [];
		if (is_array($product)) {
			foreach ($product as $p) {
				if (isset($this->cQuantity[$p->getId()])) {
					$product[$p->getId()]
						->setQuantity($this->cQuantity[$p->getId()]['local'])
						->setQuantityExternal($this->cQuantity[$p->getId()]['external']);
				} else
					$id[] = $p->getId();
			}
		} else {
			if (isset($this->cQuantity[$product->getId()]))
				$product
					->setQuantity($this->cQuantity[$product->getId()]['local'])
					->setQuantityExternal($this->cQuantity[$product->getId()]['external']);
			else
				$id = $product->getId();
		}

		if (!isset($id))
			return;

		foreach (array_chunk($id, 900) as $chunk) {
			$query = (new ProductQuery($this->defaultLang->locale))
				->byId($chunk)
				->selectQuantity();

			$data           = $query->getQueryBuilder($this->getEr())->groupBy('p.id')->getQuery()->getArrayResult();
			$availabilities = $this->availabilityService->getAll();

			foreach ($data as $id => $row) {
				$suppliersQ = [];
				$suppliersA = null;

				foreach (explode(',', (string) $row['suppliersQ']) as $v)
					$suppliersQ[] = (int) $v;

				foreach (explode(',', (string) $row['suppliersA']) as $k => $v) {
					if (!$suppliersA || ($suppliersQ[$k] > 0 && $suppliersA->getDelay() > $availabilities[$v]->getDelay()))
						$suppliersA = $availabilities[$v];
				}

				$suppliersQ = array_sum($suppliersQ);

				if (Config::load('sumLocalAndExternalQuantities'))
					$row['productQ'] = intval($row['productQ']) + intval($suppliersQ);

				$this->cQuantity[$id] = [
					'local'     => intval($row['productQ']),
					'external'  => intval($suppliersQ),
					'externalA' => $suppliersA,
				];
			}

			$allowExternal = (int) $this->settings->get('eshopCatalogDisableExternalStorage', 0) === 0;
			if (is_array($product)) {
				foreach ($data as $id => $row) {
					$product[$id]->setQuantity($this->cQuantity[$id]['local'])
						->setQuantityExternal($allowExternal ? $this->cQuantity[$id]['external'] : 0);
					$product[$id]->externalAvailability = $this->cQuantity[$id]['externalA'] ?? null;
				}
			} else if (isset($this->cQuantity[$product->getId()])) {
				$id = $product->getId();
				$product->setQuantity($this->cQuantity[$id]['local'])
					->setQuantityExternal($allowExternal ? $this->cQuantity[$id]['external'] : 0);
				$product->externalAvailability = $this->cQuantity[$id]['externalA'] ?? null;
			}
		}
	}

	/**
	 * @param array $product
	 *
	 * @return Dao\Product
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function fillDao($product): Dao\Product
	{
		$daoClass = Config::load('product.daoClass');
		/** @var Dao\Product $p */
		$p = new $daoClass($product['id']);
		$p->setQuantity((int) $product['quantity']);
		$p->setVatRate($product['vatRate']);

		if (Config::load('product.priceIsWithoutVat', false)) {
			foreach (['price', 'retailPrice'] as $c) {
				$product[$c] = ProductsHelper::processPrice((float) $product[$c], $p->getVatRate());
			}
		}

		$p->setPrice((float) $product['price']);
		$p->setBasePrice((float) $product['price']);
		$p->setRecyclingFee((float) $product['recyclingFee']);

		if (Config::load('product.allowRetailPrice') && (float) $product['retailPrice'] > $p->getPrice()) {
			$p->setRetailPrice((float) $product['retailPrice']);
			$p->retailPriceInBaseCurrency = $p->retailPrice;
		} else {
			$p->retailPrice               = null;
			$p->retailPriceInBaseCurrency = null;
		}

		$p->priceInBaseCurrency     = $p->getPrice();
		$p->basePriceInBaseCurrency = $p->basePrice;

		if ($product['hsCustomsCode'] && Config::load('product.allowHSCustomsCode')) {
			$p->hsCustomsCode = (int) $product['hsCustomsCode'];
		}

		if ($product['countryOfOrigin'] && Config::load('product.allowCountryOfOrigin')) {
			$p->countryOfOrigin = $product['countryOfOrigin'];
		}

		$p->setName($product['productTexts']['name']);
		$p->setName2($product['productTexts']['name2']);
		$p->setDescription($product['productTexts']['description']);
		$p->setSeo($product['productTexts']['seo']);
		$p->setEan($product['ean']);
		$p->setCode1($product['code1']);
		$p->setCode2($product['code2']);
		$p->setExtraFields($product['extraFields']);
		$p->setMaximumAmount($product['maximumAmount']);
		$p->minimumAmount     = (int) $product['minimumAmount'] ?: 1;
		$p->unlimitedQuantity = $product['unlimitedQuantity'] ? 1 : 0;
		$p->shortDescription  = $product['productTexts']['shortDescription'];
		$p->created           = $product['created'];
		$p->modified          = $product['modified'];
		$p->categories        = $product['categories'];
		$p->variantId         = $product['variantId'];
		$p->variantName       = $product['variantName'];
		$p->helperData        = [
			'variantImage' => $product['variantImage'],
		];

		$p->defaultCategoryId      = (int) $product['defaultCategory'];
		$p->canAddToCart           = $product['canProductsAddToCart'] ? true : false;
		$p->discountDisabled       = $product['discountDisabled'] ? true : false;
		$p->categoryGiftsAllowed   = $product['categoryGiftsAllowed'] ? true : false;
		$p->isAssort               = $product['isAssort'] ? true : false;
		$p->condition              = $product['condition'] ?: 'new';
		$p->disablePickUpSpedition = $product['disablePickUpSpedition'] ? true : false;
		$p->isDiscount             = $product['isDiscount'] ? true : false;
		$p->isOversize             = $product['isOversize'] ? true : false;
		$p->disabledSpeditions     = $product['disabledSpeditions'];

		foreach ($product['dynamicFeatures'] as $v) {
			foreach ($v as $k => $v2) {
				$p->features['dynamic_' . $k] = $v2;
			}
		}

		if (Config::load('product.allowPreorder'))
			$p->preorderText = $product['productTexts']['preorderText'];

		$p->width  = $product['width'];
		$p->height = $product['height'];
		$p->depth  = $product['depth'];
		$p->weight = $product['weight'];

		if ($product['gallery'])
			$p->galleryId = $product['gallery'];

		if ($product['manufacturer'])
			$p->manufacturerId = $product['manufacturer'];

		if (isset($product['inSites']) && !empty($product['inSites'])) {
			$p->inSites = $product['inSites'];
		}

		$this->eventDispatcher->dispatch(new DaoEvent($p, $product), Products::class . '::fillDao');

		return $p;
	}

	/**
	 * @param array $product
	 *
	 * @return array
	 */
	public function normalizeHydrateArray($product)
	{
		$result                 = $product[0];
		$result['variantId']    = (int) $product['variantId'];
		$result['variantName']  = $product['variantName'];
		$result['variantImage'] = $product['variantImage'] ? (int) $product['variantImage'] : null;
		$result['productTexts'] = array_values($product[0]['productTexts'])[0];
		foreach (['gallery', 'manufacturer', 'vatRate', 'defaultCategory', 'availability'] as $v)
			$result[$v] = $product[$v];

		if (isset($product['inSites'])) {
			$sites             = explode(',', $product['inSites']);
			$result['inSites'] = array_unique($sites);
		}

		return $result;
	}

	protected function prepareFilter($filter)
	{
		if (isset($filter['features'])) {
			$count = count($filter['features']);
			$ids   = [];
			$qb    = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')->select('IDENTITY(fp.product) as product');
			$tmp   = [];

			foreach ($filter['features'] as $group => $vals) {
				$or  = [];
				$qb2 = clone $qb;
				foreach ($vals as $k => $v) {
					$or[] = "IDENTITY(fp.featureValue) = :fpfv_{$group}_{$k}";
					$qb2->setParameter("fpfv_{$group}_{$k}", $v);
				}
				$qb2->andWhere(implode(' OR ', $or));
				foreach ($qb2->getQuery()->getArrayResult() as $row)
					$tmp[$row['product']] = isset($tmp[$row['product']]) ? $tmp[$row['product']] + 1 : 1;
			}

			foreach ($tmp as $k => $v) {
				if ($v == $count)
					$ids[] = $k;
			}
			$tmp = null;

			$filter['features'] = empty($ids) ? 'empty' : $ids;
		}

		return $filter;
	}

	public function prepareSort(ProductQuery &$query, $sort)
	{
		if (is_null($sort)) {
			$sort = 'p.price ASC';
		}

		if ($sort === 'recommended') {
			$sort = '';
		}

		if (is_string($sort)) {
			$sort = explode(' ', $sort, 2);

			if (!in_array(strtolower((string) $sort[1]), ['asc', 'desc'])) {
				$sort[1] = 'asc';
			}

			$query->addSortBy($sort[0], $sort[1] ?? null);
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v) {
				$query->addSortBy($k, $v);
			}
		}
	}

	public function getExtraFieldsByKey(string $key): array
	{
		if ($this->cExtraFields2[$key] === null) {
			$this->cExtraFields2[$key] = [];

			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				         ->select('ef.sectionKey, ef.value')
				         ->where('ef.sectionName = :secName')
				         ->andWhere('ef.key = :key')
				         ->andWhere('ef.lang IS NULL OR ef.lang = :lang')
				         ->setParameters([
					         'lang'    => $this->translator->getLocale(),
					         'secName' => Product::EXTRA_FIELD_SECTION,
					         'key'     => $key,
				         ])->getQuery()->getScalarResult() as $row)
				$this->cExtraFields2[$key][$row['sectionKey']] = $row['value'];
		}

		return $this->cExtraFields2[$key];
	}

	public function getAllExtraFields(): array
	{
		if ($this->cExtraFields === null) {
			$lang = $this->translator->getLocale();

			$this->cExtraFields = $this->cacheService->defaultCache->load('productsExtraFields_' . $lang, function(&$dep) use ($lang) {
				$dep = [
					Cache::EXPIRATION => '10 minutes',
				];

				$arr   = [];
				$query = $this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					->select('ef.sectionKey, ef.key, ef.value')
					->where('ef.sectionName = :secName')
					->andWhere('ef.lang IS NULL OR ef.lang = :lang')
					->setParameters([
						'lang'    => $lang,
						'secName' => Product::EXTRA_FIELD_SECTION,
					])->getQuery();

				foreach ($query->getScalarResult() as $row)
					$arr[$row['sectionKey']][$row['key']] = $row['value'];

				return $arr;
			});
		}

		return $this->cExtraFields;
	}

	public static function setMode(string $mode): void { self::$mode = $mode; }
}
