<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use Spatie\SchemaOrg\BankAccount;

/**
 * @ORM\Entity
 * @ORM\Table(name="eshop_catalog__seller")
 */
class Seller
{
	use TId;

	public const EMAIL_TYPE_SELLER = 'seller';
	public const EMAIL_TYPE_SITE   = 'site';

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", nullable=false)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string", nullable=false)
	 */
	public $name;

	/**
	 * @var string
	 * @ORM\Column(name="address", type="string", nullable=true)
	 */
	public $address;

	/**
	 * @var string
	 * @ORM\Column(name="city", type="string", nullable=true)
	 */
	public $city;

	/**
	 * @var string
	 * @ORM\Column(name="postal", type="string", nullable=true)
	 */
	public $postal;

	/**
	 * @var string
	 * @ORM\Column(name="country", type="string", nullable=true)
	 */
	public $country;

	/**
	 * @var string
	 * @ORM\Column(name="address2", type="string", nullable=true)
	 */
	public $address2;

	/**
	 * @var string
	 * @ORM\Column(name="city2", type="string", nullable=true)
	 */
	public $city2;

	/**
	 * @var string
	 * @ORM\Column(name="postal2", type="string", nullable=true)
	 */
	public $postal2;

	/**
	 * @var string
	 * @ORM\Column(name="country2", type="string", nullable=true)
	 */
	public $country2;

	/**
	 * @var string
	 * @ORM\Column(name="ic", type="string", length=12, nullable=true)
	 */
	public $ic;

	/**
	 * @var string
	 * @ORM\Column(name="dic", type="string", length=14, nullable=true)
	 */
	public $dic;

	/**
	 * @var string
	 * @ORM\Column(name="email", type="string", nullable=true)
	 */
	public $email;

	/**
	 * @var string
	 * @ORM\Column(name="phone", type="string", nullable=true)
	 */
	public $phone;

	/**
	 * @var Collection<string, SellerInSite>
	 * @ORM\OneToMany(targetEntity="SellerInSite", mappedBy="seller", indexBy="site")
	 */
	protected Collection $sites;

	/**
	 * @var SellerBankAccount[]
	 * @ORM\OneToMany(targetEntity="SellerBankAccount", mappedBy="seller", indexBy="currencyCode")
	 */
	protected $bankAccounts;

	/**
	 * @ORM\Column(name="send_order", type="smallint", length=1, options={"default": 1})
	 */
	public int $sendOrder = 1;

	/**
	 * @ORM\Column(name="send_status", type="smallint", length=1, options={"default": 1})
	 */
	public int $sendStatus = 1;

	/**
	 * @ORM\Column(name="send_email_type", type="string", nullable=false, options={"default": "seller"})
	 */
	public string $sendEmailType = 'seller';

	public function __construct(string $title, string $name)
	{
		$this->title        = $title;
		$this->name         = $name;
		$this->sites        = new ArrayCollection;
		$this->bankAccounts = new ArrayCollection;
	}

	public function getIc(): ?string { return $this->ic; }

	public function setIc(?string $ic): self
	{
		$this->ic = $ic;

		return $this;
	}

	public function getDic(): ?string { return $this->dic; }

	public function setDic(?string $dic): self
	{
		$this->dic = $dic;

		return $this;
	}

	public function isVatPayer(): bool { return $this->dic ? true : false; }

	/** @return Collection<string, SellerInSite> */
	public function getSites() { return $this->sites; }

	public function setSites(array $sites): self
	{
		$this->sites = new ArrayCollection($sites);

		return $this;
	}

	public function getSitesIdent(): array
	{
		$sites = [];

		foreach ($this->getSites()->toArray() as $k => $v)
			$sites[$k] = $v->getSite()->getIdent();

		return $sites;
	}

	/** @return SellerBankAccount[]|ArrayCollection */
	public function getBankAccounts() { return $this->bankAccounts; }

	/**
	 * @param string|null $currencyCode
	 *
	 * @return SellerBankAccount|null
	 */
	public function getBankAccount(?string $currencyCode = null)
	{
		if ($currencyCode) {
			$currencyCode = strtolower($currencyCode);
			$arr          = $this->getBankAccounts()->toArray();

			if (isset($arr[$currencyCode]))
				return $arr[$currencyCode];
		}

		return $this->bankAccounts->first();
	}
}
