<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\Site;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Nette\ComponentModel\IComponent;
use Nette\Localization\ITranslator;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Entities\Category;

/**
 * Class CategoryContainer
 * @package EshopCatalog\AdminModule\Components\Products
 */
class CategoryContainer
{
	protected EntityManagerDecorator $em;

	protected ITranslator $translator;

	protected Categories $categoryServices;

	protected Products $productsService;

	protected CategoryProducts $categoryProductService;

	protected Sites $sitesService;

	protected VariantsHelper $variantsHelper;

	public function __construct(EntityManagerDecorator $em, Categories $categories, CategoryProducts $categoryProducts, Products $products, ITranslator $translator,
	                            Sites $sites, VariantsHelper $variantsHelper)
	{
		$this->em                     = $em;
		$this->categoryServices       = $categories;
		$this->translator             = $translator;
		$this->productsService        = $products;
		$this->categoryProductService = $categoryProducts;
		$this->sitesService           = $sites;
		$this->variantsHelper         = $variantsHelper;
	}

	public function getContainers()
	{
		$categoriesContainer = new BaseContainer();
		$sites               = $this->sitesService->getAll();
		foreach ($sites as $site) {
			$categoryContainer = new BaseContainer();

			if (count($sites) > 1)
				$categoryContainer->addBool('isActive', 'eshopCatalog.productForm.activeInEshop')->setDefaultValue(0);
			else
				$categoryContainer->addHidden('isActive', '1');

			$trees = $this->categoryServices->getTreeForSelect($site->getIdent());

			$categoryContainer->addCustomData('trees', $trees);
			$categoryContainer->addCustomData('template', __DIR__ . '/CategoryContainer.latte');

			$categoryContainer->addRadioList('defaultCategory', 'eshopCatalog.productForm.defaultCategory',
				$this->categoryServices->getOptionsForSelect($site->getIdent()))
				->setHtmlAttribute('data-disable-toggle', 'true');;
			$categoryContainer->addCheckboxList('category', 'eshopCatalog.productForm.categories',
				$this->categoryServices->getOptionsForSelect($site->getIdent()))
				->setHtmlAttribute('data-disable-toggle', 'true');

			$categoriesContainer->addComponent($categoryContainer, $site->getIdent());
		}

		return $categoriesContainer;
	}

	public function getContainer(string $siteIdent)
	{
		$container = new BaseContainer();

		$trees = $this->categoryServices->getTreeForSelect($siteIdent);

		$container->addCustomData('trees', $trees);
		$container->addCustomData('template', __DIR__ . '/CategoryContainer.latte');

		$container->addRadioList('defaultCategory', 'eshopCatalog.productForm.defaultCategory', $this->categoryServices->getOptionsForSelect($siteIdent))
			->setHtmlAttribute('data-disable-toggle', 'true');
		$container->addCheckboxList('category', 'eshopCatalog.productForm.categories', $this->categoryServices->getOptionsForSelect($siteIdent))
			->setHtmlAttribute('data-disable-toggle', 'true');

		return $container;
	}

	/**
	 * @param array|ArrayHash $values
	 *
	 * @return array
	 */
	public function getFormData($values): array
	{
		return (array) $values;
	}

	/**
	 * @param BaseContainer|IComponent $container
	 * @param ProductInSite[]          $siteValues
	 * @param array                    $allCategories
	 */
	public function setDefaultsMultipleSite(&$container, array $siteValues, array $allCategories): void
	{
		foreach ($siteValues as $siteIdent => $v) {
			/** @var ProductInSite $v */
			$formCat = &$container->getComponent($siteIdent, false) ?? null;

			if (!$formCat)
				continue;

			$formCat->setDefaults([
				'isActive'        => $v->isActive(),
				'defaultCategory' => $v->category ? $v->category->getId() : null,
				'category'        => [],
			]);
		}

		foreach ($container->getComponents() as $component) {
			$component->setDefaults([
				'category' => $allCategories,
			]);
		}
	}

	public function saveDataMultipleSite(array $productsId, array $data): void
	{
		$allCategories = [];
		/** @var Product[] $products */
		$products = array_map(function($id) { return $this->productsService->getReference($id); }, $productsId);

		foreach ($data as $siteIdent => $v) {
			$site            = $this->em->getReference(Site::class, $siteIdent);
			$defaultCategory = $v->defaultCategory ? $this->categoryServices->getReference($v->defaultCategory) : null;

			foreach ($products as $product) {
				$productSite = $product->sites->get($siteIdent) ?? null;

				if (!$productSite)
					$productSite = new ProductInSite($product, $site);

				$productSite->setActive((int) (count($data) <= 1 ? 1 : $v->isActive));

				if ($defaultCategory) {
					$productSite->category = $defaultCategory;

					foreach ($v->category as $k1 => $v1) {
						if ($v->defaultCategory != $v1)
							$allCategories[] = $v1;
					}
				} else {
					$productSite->category = null;
				}

				$this->em->persist($productSite);
			}
		}

		$allCategories = array_unique($allCategories);

		$currentProductCategories = [];
		foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
			         ->select('IDENTITY(cp.product) as product, IDENTITY(cp.category) as category')
			         ->where('cp.product IN (:ids)')
			         ->setParameter('ids', $productsId)
			         ->getQuery()->getScalarResult() as $row)
			$currentProductCategories[$row['product']][] = $row['category'];

		foreach ($products as $product) {
			$productCategories = $currentProductCategories[$product->getId()] ?: [];

			// Smazat
			$remove = array_diff($productCategories, $allCategories);
			if (!empty($remove)) {
				$removeProds = [$product->getId()] + $this->variantsHelper->getProductVariants($product->getId());

				foreach ($removeProds as $rProd) {
					$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
						->where('cp.product = :prod')
						->andWhere('cp.category IN (:categories)')
						->setParameters([
							'prod'       => $rProd,
							'categories' => $remove,
						])->getQuery()->execute();
				}
			}

			// Přidat
			foreach (array_diff($allCategories, $productCategories) as $id) {
				$tmp = new CategoryProduct($product, $this->em->getReference(Category::class, $id));
				$this->em->persist($tmp);
			}

			$this->em->persist($product);
		}
	}
}
