<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\ExtraField;
use EshopCatalog\AdminModule\Model\Helpers\ProductFormHelper;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopCatalog\Model\Entities\ProductVariant;
use Gallery\Model\Entities\Image;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Components\Categories\CategoryForm;
use EshopCatalog\AdminModule\Components\Categories\ICategoryFormFactory;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Tags;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\VatRate;
use Core\Model\Entities\QueryBuilder;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\FilterService as FrontFilterService;
use Tracy\Debugger;

/**
 * Class ProductsGrid
 * @package EshopCatalog\AdminModule\Components\Products
 */
class ProductsGrid extends BaseControl
{
	protected Products $productServices;

	protected array $quantities = [];

	protected array $galleries = [];

	protected array $extraFields = [];

	protected array $productCategories = [];

	protected array $speditions = [];

	protected Categories $categories;

	protected ICategoryFormFactory $categoryFormFactory;

	protected Sites $sitesService;

	protected Manufacturers $manufacturers;

	protected Features $features;

	protected FeatureValues $featureValues;

	protected Tags $tags;

	protected FrontFilterService $frontFilterService;

	protected ProductFormHelper $productFormHelper;

	public function __construct(Products $products, Categories $categories, ICategoryFormFactory $categoryFormFactory, Sites $sites, Tags $tags,
	                            Manufacturers $manufacturers, Features $features, FeatureValues $featureValues, FrontFilterService $filterService,
	                            ProductFormHelper $productFormHelper)
	{
		$this->productServices     = $products;
		$this->categories          = $categories;
		$this->categoryFormFactory = $categoryFormFactory;
		$this->sitesService        = $sites;
		$this->tags                = $tags;
		$this->manufacturers       = $manufacturers;
		$this->features            = $features;
		$this->featureValues       = $featureValues;
		$this->frontFilterService  = $filterService;
		$this->productFormHelper   = $productFormHelper;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleEditCategory(int $categoryId): void
	{
		$category = $this->categories->get($categoryId);

		if ($category) {
			$this['categoryForm']->categoryId = $categoryId;
			$this->template->modalTitle       = $this->t('eshopCatalog.title.editCategory') . ' ' . $category->getCategoryText()->getName();
			$this->template->modal            = 'categoryForm';
			$this->redrawControl('modal');
		} else {
			$this->getPresenter()->flashMessageDanger('eshopCatalog.category.notFound');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	public function handleDeleteDuplicity(): void
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->deleteDuplicity())
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.duplicityProductsDeleted');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.duplicityProductsDeleteFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGetFeatures(): void
	{
		$features = [];
		foreach ($this->features->getOptionsForSelect() as $k => $v)
			$features[] = [
				'id'    => $k,
				'value' => $v,
			];

		$values = [];
		foreach ($this->featureValues->getOptionsForSelect() as $feature => $vals) {
			foreach ($vals as $k => $v) {
				$values[$feature][] = [
					'id'    => $k,
					'value' => $v,
				];
			}
		}

		$this->getPresenter()->sendJson([
			'features' => $features,
			'values'   => $values,
		]);
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false)
			->setRememberState()
			->setItemsPerPageList([20, 50, 100, 200, 500], false)
			->setDefaultPerPage(50);

		$grid->setTemplateFile(__DIR__ . '/ProductsGrid_grid.latte');
		$sites = $this->sitesService->getAll();

		$availabilities = [];
		foreach ($this->em->getRepository(Availability::class)->createQueryBuilder('av')
			         ->addSelect('avt')
			         ->innerJoin('av.texts', 'avt', 'WITH', 'avt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $row) {
			$row['texts']                  = $row['texts'][$this->translator->getLocale()];
			$availabilities[$row['id']]    = $row;
			$availabilities[$row['ident']] = $row;
		}

		$qb = $this->productServices->getEr()->createQueryBuilder('p')
			->addSelect('pt')
			->innerJoin('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale());

		if (Config::load('productsGrid.visualVariants')) {
			$qb->addSelect('isVar')
				->addSelect('CASE WHEN isVar.variantId IS NOT NULL THEN CONCAT(isVar.createdDefault, isVar.variantId) ELSE CONCAT(p.created, p.id) END as hidden createdSort')
				->leftJoin('p.isVariant', 'isVar');
		}

		if (Config::load('productsGrid.groupVariants')) {
			$qb->addSelect('var')
				->addSelect('CASE WHEN var.variantId IS NOT NULL THEN CONCAT(var.variantId, \'_\', p.id) ELSE CONCAT(p.id, \'_\') END as hidden ord')
				->leftJoin('p.isVariant', 'var');
		}

		if (Config::load('productsGrid.showCategories')) {
			$qb->addSelect('sites, cp, cpc, cpct')
				->leftJoin('p.sites', 'sites')
				->leftJoin('p.categoryProducts', 'cp')
				->leftJoin('cp.category', 'cpc')
				->leftJoin('cpc.categoryTexts', 'cpct', 'WITH', 'cpct.lang = :lang');
		}

		if (Config::load('productsGrid.showTags')) {
			$qb->addSelect('ptag, tags')
				->leftJoin('p.productTags', 'ptag')
				->leftJoin('ptag.tag', 'tags');
		}

		if (Config::load('productsGrid.showManufacturerColumn') || Config::load('productsGrid.showManufacturerInName')) {
			$qb->addSelect('manu')
				->leftJoin('p.manufacturer', 'manu');
		}

		$configSort          = Config::load('productsGrid.sort');
		$configSortDirection = Config::load('productsGrid.sortDirection');
		$grid->setDefaultSort([$configSort => $configSortDirection], true);

		$dataSource           = new DoctrineDataSource($qb, 'p.id');
		$dataSource->lastSort = ['p.id' => 'DESC'];

		$grid->setDataSource($dataSource);

		$grid->getDataSource()->onDataLoaded[] = function($items) use ($sites) {
			$ids = array_map(function($row) { return $row->getId(); }, $items);

			if (is_string($this->productCategories))
				$this->productCategories = [];

			if ($ids) {
				foreach ($this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')
					         ->select('IDENTITY(ps.product) as id, ps.quantity')
					         ->where('ps.product IN (:ids)')
					         ->setParameter('ids', $ids)
					         ->getQuery()->getArrayResult() as $v) {
					if (!isset($this->quantities[$v['id']]))
						$this->quantities[$v['id']] = 0;
					$this->quantities[$v['id']] += intval($v['quantity']);
				}

				foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
					         ->select('ct.name, IDENTITY(cp.product) as prod')
					         ->where('cp.product IN (:prod)')
					         ->join('cp.category', 'c')
					         ->join('c.categoryTexts', 'ct', 'with', 'ct.lang = :lang')
					         ->setParameters([
						         'prod' => $ids,
						         'lang' => $this->translator->getLocale(),
					         ])->getQuery()->getArrayResult() as $row) {
					if (!isset($this->productCategories[$row['prod']]) || is_string($this->productCategories[$row['prod']]))
						$this->productCategories[$row['prod']] = [];
					$this->productCategories[$row['prod']][] = $row['name'];
				}

				if (count($sites) > 1) {
					foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
						         ->select('ct.name, IDENTITY(ps.product) as prod')
						         ->where('ps.product IN (:prod)')
						         ->join('ps.category', 'c')
						         ->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
						         ->setParameters([
							         'prod' => $ids,
							         'lang' => $this->translator->getLocale(),
						         ])->getQuery()->getArrayResult() as $row) {
						$this->productCategories[$row['prod']][] = $row['name'];
					}
				}

				$variants = [];
				foreach ($this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					         ->select('IDENTITY(pv.product) as productId, pv.defaultImage as image')
					         ->where('pv.product IN (:products)')
					         ->setParameter('products', $ids)
					         ->getQuery()->getArrayResult() as $row) {
					$variants[$row['productId']] = $row['image'];
				}

				$galleries = [];
				foreach ($this->em->getRepository(Image::class)->createQueryBuilder('i')
					         ->select('i.id as imgId, i.filename, i.path, i.position, i.isCover, p.id as productId')
					         ->innerJoin('i.album', 'a')
					         ->innerJoin(Product::class, 'p', Join::WITH, 'p.gallery = a.id AND p.id IN (:products)')
					         ->andWhere('i.isPublished = 1')
					         ->setParameter('products', $ids)
					         ->getQuery()->getArrayResult() as $row) {
					$file = $row['path'] . DS . $row['filename'];

					$galleries[$row['productId']]['pos'][$row['position']] = $file;
					$galleries[$row['productId']][$row['imgId']]           = $file;
					if ($row['isCover'])
						$galleries[$row['productId']]['cover'] = $file;
				}

				foreach ($galleries as $productId => $images) {
					ksort($images['pos']);
					$this->galleries[$productId] = $images[$variants[$productId]] ?: ($images['cover'] ?: array_shift($images['pos']));
				}

				foreach ($this->productCategories as $k => $v)
					$this->productCategories[$k] = implode(', ', $v);

				foreach ($this->em->createQueryBuilder()
					         ->select('IDENTITY(ps.product) as product, s.name, ps.speditionDisabled')
					         ->from(ProductSpedition::class, 'ps')
					         ->innerJoin('ps.spedition', 's')
					         ->where('ps.product IN (:ids)')->setParameter('ids', $ids)
					         ->getQuery()->getArrayResult() as $row) {
					$this->speditions[$row['product']][] = $row;
				}
			}

			if (Config::load('productsGrid.loadExtraFields')) {
				foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					         ->where('ef.sectionName = :secName')->setParameter('secName', Product::EXTRA_FIELD_SECTION)
					         ->getQuery()->getArrayResult() as $row)
					$this->extraFields[$row['sectionKey']][$row['key']] = $row['value'];
			}
		};

		/**
		 * === Columns
		 */
		$grid->addColumnText('image', 'eshopCatalog.defaultGrid.image', '')->setRenderer(function(Product $row) {
			$image = $this->galleries[$row->getId()] ?? null;
			if (!$image)
				return '';

			// TODO dalo by se předělat do nové ColumnImagePreview
			return Html::el('')
				->addHtml(Html::el('img', [
					'onMouseOver' => "showPicture('spt-{$row->getId()}', 1)",
					'onMouseOut'  => "showPicture('spt-{$row->getId()}', 0)",
					'src'         => $this->imagePipe->request($image, '50x40'),
				]))
				->addHtml(Html::el('img', [
					'id'    => 'spt-' . $row->getId(),
					'class' => 'show-picture-target',
					'src'   => $this->imagePipe->request($image, '400x400'),
				]));
		})->setAlign('center');

		// Name
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(function(Product $row) {
			$wrap  = Html::el();
			$icons = [];

			if (Config::load('productsGrid.visualVariants') && $row->isVariant()) {
				$marker = Html::el('div')
					->addClass('marker-square mr-2');

				if ($row->isVariant()->isDefault)
					$marker->addClass('marker--blue-full');
				else
					$marker->addClass('marker--blue-border');

				$wrap->addHtml($marker);
			}

			$link = Html::el('a', [
				'href' => $this->getPresenter()->link('Products:edit', [$row->getId()]),
			])->setText($row->getText()->name);
			$wrap->addHtml($link);

			if (Config::load('productsGrid.showManufacturerInName')) {
				$manufacturer = Html::el('div', [
					'class' => 'text-secondary',
				])->setText($this->t('eshopCatalog.defaultGrid.manufacturer') . ': ' . $row->getManufacturer()->name);
				$wrap->addHtml($manufacturer);
			}

			if (Config::load('product.allowModifySpedition') && isset($this->speditions[$row->getId()])) {
				$speditions = [];
				foreach ($this->speditions[$row->getId()] as $sped) {
					if ($sped['speditionDisabled'])
						$speditions[] = $sped['name'];
				}

				if ($speditions)
					$icons[] = Html::el('span', ['title' => implode(', ', $speditions),
					                             'class' => 'icon--disabled-spedition'])
						->addHtml('<i class="fas fa-truck-loading color-red"></i>');
			}

			if ($icons) {
				$i = Html::el('div', ['class' => 'grid-cell__icons']);

				foreach ($icons as $ico)
					$i->addHtml($ico);

				$wrap->addHtml($i);
			}

			return $wrap;
		})
			->setSortable('pt.name')
			->setFilterText()->setCondition([$this, 'conditionGridName']);

		// Manufacturer
		if (Config::load('productsGrid.showManufacturerColumn'))
			$grid->addColumnText('manufacturer', 'eshopCatalog.defaultGrid.manufacturer', 'manufacturer.name')
				->setSortable('manu.name')
				->getElementPrototype('td')->class[] = 'w1nw';
		$grid->addFilterSelect('manufacturer', 'eshopCatalog.defaultGrid.manufacturer', [
				''  => '',
				'-' => $this->translator->translate('eshopCatalog.defaultGrid.withoutManufacturer'),
			] + $this->manufacturers->getOptionsForSelect(),
			'manufacturer.name')
			->setCondition([$this, 'conditionGridManufacturer']);

		// Variant
		$grid->addFilterSelect('variantType', 'eshopCatalog.defaultGrid.variantType', [
			''               => '',
			'withoutVariant' => $this->t('eshopCatalog.defaultGrid.withoutVariant'),
			'main'           => $this->t('eshopCatalog.defaultGrid.mainProduct'),
			'variant'        => $this->t('eshopCatalog.defaultGrid.variant'),
		])->setCondition([$this, 'conditionGridVariantType']);

		// EAN
		if (Config::load('productsGrid.showEAN')) {
			$grid->addColumnText('ean', 'eshopCatalog.defaultGrid.ean')
				->setSortable()
				->setFilterText();
		}

		// Codes
		if (Config::load('productsGrid.showCodes')) {
			$grid->addColumnText('codes', 'eshopCatalog.defaultGrid.codes')->setRenderer(function(Product $row) {
				$html = Html::el();

				if ($row->code1)
					$html->addHtml(Html::el('div')->setText($row->code1));
				if ($row->code2)
					$html->addHtml(Html::el('div')->setText($this->t('eshopCatalog.defaultGrid.code2') . ': ' . $row->code2));

				return $html;
			})
				->setSortable()->setSortableCallback(function(QueryBuilder $qb, $vals) {
					$qb->addOrderBy('p.code1', $vals['codes'])->addOrderBy('p.code2', $vals['codes'])->addOrderBy('p.id', $vals['codes']);
				})
				->setFilterText(['code1', 'code2']);
		} else {
			$grid->addColumnText('codes', 'eshopCatalog.defaultGrid.code1', 'code1')
				->setSortable()
				->setFilterText();
		}

		// Categories
		if (Config::load('productsGrid.showCategories')) {
			if (count($sites) == 1) {
				$grid->addColumnText('defaultCategory', 'eshopCatalog.productGrid.defaultCategory')->setRenderer(function(Product $row) {
					/** @var ProductInSite $site */
					$site = $row->sites->first();

					return $site && $site->category
						? Html::el('a', [
							'class' => 'ajax',
							'href'  => $this->link('editCategory!', $site->category->getId()),
						])->setText($site->category->getCategoryText()->name)
						: '';
				});

				$grid->addColumnText('categories', 'eshopCatalog.productGrid.categories')->setRenderer(function($row) {
					return $this->productCategories[$row->getId()];
				});
			} else {
				$grid->addColumnText('categories', 'eshopCatalog.productGrid.allCategories')->setRenderer(function($row) {
					return $this->productCategories[$row->getId()];
				});
			}

			$grid->getColumn('categories')->setFilterText()->setCondition([$this, 'conditionGridCategories']);
		}

		// Tags
		if (Config::load('productsGrid.showTags')) {
			$grid->addColumnText('tags', 'eshopCatalog.productGrid.tags')->setRenderer(function(Product $row) {
				$arr = [];
				foreach ($row->getProductTags()->toArray() as $pt)
					$arr[] = $pt->getTag()->type;

				return implode(', ', $arr);
			})
				->setFilterText()->setCondition([$this, 'conditionGridTags']);
		}

		// Price
		$grid->addColumnPrice('price', 'eshopCatalog.productGrid.price')
			->setSortable();

		// Purchase price
		if (Config::load('enablePurchasePrice', false)) {
			$grid->addColumnText('purchasePrice', 'eshopCatalog.productGrid.purchaseMargin')->setRenderer(function(Product $row) {
				if (!$row->purchasePrice)
					return '';

				$purchase = ($row->purchasePrice * ((float) ((100 + $row->vatRate->rate) / 100)));

				return round($row->price / $purchase * 100, 2) - 100 . '% (' . round($purchase, 2) . ')';
			})->setAlign('right');
		}

		// Availability
		$grid->addColumnText('availability', 'eshopCatalog.productGrid.availability')->setRenderer(function(Product $row) use ($availabilities) {
			$av = $row->getAvailability() ? $availabilities[$row->getAvailability()->getId()] : null;

			if ($row->unlimitedQuantity)
				$av = $availabilities['inStock'];

			return !$av
				? ''
				: Html::el('span', [
					'class' => 'btn',
					'style' => sprintf('cursor: default; color: %s; background-color: %s;', $av['color'], $av['bgColor']),
				])->setText($av['texts']['name']);
		})->setAlign('center');

		// Quantity
		$grid->addColumnNumber('quantity', 'eshopCatalog.productGrid.quantity')->setRenderer(function(Product $row) {
			if (Config::load('pseudoWarehouse') && $row->unlimitedQuantity)
				return '∞';

			$html = Html::el('');
			$html->addHtml(Html::el('span', [
				'class' => 'btn btn-xs ' . ($row->quantity > 0 ? 'btn-success' : 'btn-danger'),
			])->setText($row->quantity));

			if (isset($this->quantities[$row->getId()]) || Config::load('productsGrid.suppliersCountAlways', false))
				$html
					->addHtml(Html::el('span class=quantity-separator')->setText('/ '))
					->addHtml(Html::el('span', [
						'class' => 'btn btn-xs ' . ($this->quantities[$row->getId()] > 0 ? 'btn-success' : 'btn-danger'),
					])->setText($this->quantities[$row->getId()] ?? 0));

			return $html;
		})
			->setSortable();

		// Is published
		$grid->addColumnStatus('isPublished', 'eshopCatalog.productGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		$grid->addColumnDateTime('created', '')->setDefaultHide();

		/**
		 * === Filters
		 */
		$grid->addFilterText('multiSearch', 'eshopCatalog.productGrid.search')->setCondition(function(QueryBuilder $qb, $value) {
			$qb->andWhere('pt.name LIKE :ms OR pt.name2 LIKE :ms OR p.ean LIKE :ms OR p.code1 LIKE :ms OR p.code2 LIKE :ms')
				->setParameter('ms', '%' . $value . '%');
		});

		$grid->onRender[] = function() use ($grid) {
			$productsId = [];
			$ds         = new DoctrineDataSource(clone $grid->getDataSource()->getQueryBuilder(), 'p.id');
			$assembled  = $grid->assembleFilters();

			if ($assembled['categoriesId']->getValue()) {
				$assembled['features']->setValue([]);
				$ds->filter($assembled);
				$ds->updateQueryBuilder(function(QueryBuilder $qb) {
					$qb->select('p.id')
						->groupBy('p.id');
				});

				foreach ($ds->getQuery()->getArrayResult() as $row) {
					$productsId[] = $row['id'];
				}
			}

			$flatItems = [];
			foreach ($this->frontFilterService->getFeatures($productsId) as $group) {
				$flatItems[] = [
					'id'     => 'g_' . $group->id,
					'parent' => 0,
					'name'   => $group->name,
				];
				foreach ($group->items as $item) {
					$flatItems[] = [
						'id'     => $group->id . '_' . $item->id,
						'parent' => 'g_' . $group->id,
						'name'   => $item->value,
					];
				}
			}
			$grid->getFilter('features')->setFlatItems($flatItems);

			if ($grid->getPresenter()->isAjax()) {
				$grid['filter']['filter']['features']->setFlat($flatItems);
				$grid->redrawControl('filterFeature');
			}
		};

		$grid->addFilterCheckboxNestedList('features', 'eshopCatalog.product.features', [])
			->enableSelectOne()
			->setCondition(function(QueryBuilder $qb, ArrayHash $values) {
				if (!in_array('fp', $qb->getAllAliases()))
					$qb->leftJoin('p.featureProducts', 'fp');
				$groupedValues = [];

				foreach ((array) $values as $k => $v) {
					$tmp = explode('_', $v);
					$key = 'fv' . $k;

					if ($tmp[0] === 'g') {
						// Vse podle hlavniho parametru
						$qb->innerJoin('fp.featureValue', $key, Join::WITH, $key . '.feature = :f' . $key)
							->setParameter('f' . $key, $tmp[1]);
					} else if (is_numeric($tmp[0])) {
						$groupedValues[$tmp[0]][] = $tmp[1];
					}
				}

				foreach ($groupedValues as $groupId => $values) {
					$key = 'fp' . $groupId;
					// Hodnoty
					$qb->innerJoin('p.featureProducts', $key, Join::WITH, $key . '.featureValue IN (:' . $key . ')')
						->setParameter($key, $values);
				}
			});
		$grid->getFilter('features')->setAttribute('data-disable-toggle', 'true');

		$tags = [];
		foreach ($this->tags->getOptionsForSelect() as $k => $v)
			$tags[] = [
				'id'     => $k,
				'parent' => 0,
				'name'   => $v,
			];
		$grid->addFilterCheckboxNestedList('tagsFilter', 'eshopCatalog.product.tags', $tags)
			->setCondition(function(QueryBuilder $qb, ArrayHash $values) {
				$qb->innerJoin('p.productTags', 'pTags', Join::WITH, 'pTags.tag IN (:tags)')
					->setParameter('tags', array_values((array) $values));
			});
		$grid->getFilter('tagsFilter')->setAttribute('data-disable-toggle', 'true');

		if (count($sites) > 1) {
			$flat = [];
			foreach ($sites as $site) {
				$tmp = $this->categories->getFlatTree($site->getIdent());

				if (empty($tmp))
					continue;

				$flat[] = [
					'id'     => $tmp[0]['parent'],
					'parent' => 0,
					'name'   => $site->getIdent(),
				];

				$flat = array_merge($flat, $tmp);
			};
		} else {
			$flat = $this->categories->getFlatTree();
		}

		// Bez kategorii
		$grid->addFilterCheckboxNestedList('withoutCategory', 'eshopCatalog.productGrid.withoutCategory', [
			[
				'id'     => 'site',
				'parent' => 0,
				'name'   => $this->t('eshopCatalog.productGrid.withoutBaseCategory'),
			], [
				'id'     => 'other',
				'parent' => 0,
				'name'   => $this->t('eshopCatalog.productGrid.withoutOtherCategories'),
			],
		])->setCondition(function(QueryBuilder $qb, ArrayHash $values) {
			$values = (array) $values;
			if (in_array('site', $values)) {
				if (!in_array('sites', $qb->getAllAliases()))
					$qb->leftJoin('p.sites', 'sites');
				$qb->andWhere('sites.site IS NULL');
			}
			if (in_array('other', $values)) {
				if (!in_array('cp', $qb->getAllAliases()))
					$qb->leftJoin('p.categoryProducts', 'cp');
				$qb->andWhere('cp.category IS NULL');
			}

			if (!in_array('isVar', $qb->getAllAliases()))
				$qb->leftJoin('p.isVariant', 'isVar');
			$qb->andWhere('isVar IS NULL OR isVar.isDefault = 1');
		});

		$grid->addFilterCheckboxNestedList('categoriesId', 'eshopCatalog.productGrid.defaultCategory', $flat)
			->enableSelectOne()
			->setCondition(function(QueryBuilder $qb, $value) {
				$variantIds = [];
				$qb2        = $this->productServices->getEr()->createQueryBuilder('p')
					->select('p.id, isVar.variantId')
					->innerJoin('p.isVariant', 'isVar', Join::WITH, 'isVar.isDefault = 1')
					->innerJoin('p.sites', 'sites')
					->innerJoin('p.categoryProducts', 'cp')
					->where('cp.category IN (:categoriesId) OR sites.category IN (:categoriesId)')
					->groupBy('p.id')
					->setParameter('categoriesId', (array) $value);
				foreach ($qb2->getQuery()->getArrayResult() as $row) {
					$variantIds[] = $row['variantId'];
				}

				if (!in_array('sites', $qb->getAllAliases()))
					$qb->leftJoin('p.sites', 'sites');
				if (!in_array('cp', $qb->getAllAliases()))
					$qb->leftJoin('p.categoryProducts', 'cp');
				if (!in_array('isVar', $qb->getAllAliases()))
					$qb->leftJoin('p.isVariant', 'isVar');
				$qb->andWhere('(cp.category IN (:categoriesId)  
						OR sites.category IN (:categoriesId)
						OR isVar.variantId IN (:variantIds))')
					->setParameter('categoriesId', (array) $value)
					->setParameter('variantIds', $variantIds);
			});
		$grid->getFilter('categoriesId')->setAttribute('data-disable-toggle', 'true');
		if (isset($grid->getColumns()['defaultCategory']))
			$grid->getColumn('defaultCategory')->setFilterText()->setCondition([$this, 'conditionGridCategory']);
		$grid->addFilterSelect('inStock', 'eshopCatalog.defaultGrid.isInStock', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		])->setCondition([$this, 'conditionGridQuantity']);

		$availabilitiesFilter = [null => ''];
		foreach ($availabilities as $row)
			$availabilitiesFilter[$row['id']] = $row['texts']['name'];
		$grid->getColumn('availability')->setFilterSelect($availabilitiesFilter);

		$grid->getColumn('quantity')->setFilterRange()->setCondition([$this, 'conditionGridQuantity']);
		$grid->getColumn('price')->setFilterRange()->setCondition(function(QueryBuilder $qb, $values) {
			if (is_numeric($values->from))
				$qb->andWhere('p.price >= :priceFrom')->setParameter('priceFrom', $values->from);
			if (is_numeric($values->to))
				$qb->andWhere('p.price <= :priceTo')->setParameter('priceTo', $values->to);
		});
		$grid->addFilterSelect('canApplyDiscount', 'eshopCatalog.productForm.discountDisabled', [
			'' => '',
			0  => $this->t('default.yes'),
			1  => $this->t('default.no'),
		], 'p.discountDisabled');
		if (Config::load('pseudoWarehouse')) {
			$grid->addFilterSelect('unlimitedQuantity', 'eshopCatalog.productForm.unlimitedQuantity', [
				'' => '',
				1  => $this->t('default.yes'),
				0  => $this->t('default.no'),
			], 'p.unlimitedQuantity');
		}
		$grid->getColumn('isPublished')->setFilterSelect([
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		], 'p.isPublished');

		$manufacturers = [];
		foreach ($this->em->getRepository(Manufacturer::class)->createQueryBuilder('m')
			         ->select('m.id, m.name')->orderBy('m.position')->getQuery()->getArrayResult() as $m)
			$manufacturers[$m['id']] = $m['name'];

		$vatRates = [];
		foreach ($this->em->getRepository(VatRate::class)->createQueryBuilder('vr')
			         ->select('vr.id, vr.name')->getQuery()->getArrayResult() as $vr)
			$vatRates[$vr['id']] = $vr['name'];

		/**
		 * === Action
		 */
		$grid->addAction('edit', '', 'Products:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		$grid->addGroupAction('eshopCatalog.productGrid.editCategories')
			->onSelect[] = [$this, 'gridEditCategories'];
		$grid->addGroupAction('eshopCatalog.productGrid.editFeatures')
			->onSelect[] = [$this, 'gridEditFeatures'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.editManufacturer', $manufacturers)
			->onSelect[] = [$this, 'gridEditManufacturer'];
		$grid->addGroupAction('eshopCatalog.productGrid.changePublished')
			->onSelect[] = [$this, 'gridChangePublished'];
		$grid->addGroupAction('eshopCatalog.productGrid.changeUnpublished')
			->onSelect[] = [$this, 'gridChangeUnpublished'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.changeVatRate', $vatRates)
			->onSelect[] = [$this, 'gridChangeVatRate'];
		if (Config::load('allowRelatedProducts')) {
			$grid->addGroupSelectAction('eshopCatalog.productGrid.assignRelatedProducts')->onSelect[] = [
				$this,
				'gridAssignRelatedProducts',
			];
		}
		$grid->addGroupSelectAction('eshopCatalog.productGrid.discountDisabled', [
			1 => 'default.yes',
			0 => 'default.no',
		])->onSelect[] = [$this, 'gridDiscountDisabled'];

		$grid->addGroupSelectAction('eshopCatalog.productGrid.tagFastAdd', $this->tags->getOptionsForSelect())
			->onSelect[] = [$this, 'gridTagFastAdd'];

		$grid->addGroupAction('eshopCatalog.productGrid.tagSet')
			->onSelect[] = [$this, 'gridSetTags'];

		$grid->addGroupAction('eshopCatalog.productGrid.setAsVariantsFor')
			->onSelect[] = [$this, 'gridSetAsVariantsFor'];

		$grid->addGroupSelectAction('eshopCatalog.productGrid.setAvailabilityAfterSoldOut', [null => ''] + $this->productFormHelper->availabilityService->getOptionsForSelect())
			->onSelect[] = [$this, 'gridSetAvailabilityAfterSoldOut'];

		if (Config::load('product.allowModifySpedition', false))
			$grid->addGroupAction('eshopCatalog.productGrid.editSpedition')
				->onSelect[] = [$this, 'gridEditSpedition'];

		$grid->addGroupAction('eshopCatalog.productGrid.delete')->onSelect[] = [$this, 'gridDeleteProducts'];

		// Columns prototype
		$imageTdElement          = $grid->getColumn('image')->getElementPrototype('td');
		$imageTdElement->class[] = 'w1';
		$imageTdElement->style   = 'padding: 0;';

		return $grid;
	}

	protected function createComponentCategoryForm(): CategoryForm
	{
		$control = $this->categoryFormFactory->create();

		if ($this->getParameter('categoryId'))
			$control->setCategory((int) $this->getParameter('categoryId'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this['grid']->reload();
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this['grid']->reload();
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->productServices->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridCategories(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('cpct'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('cpct.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridTags(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('tags'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('tags.type', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridName(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		foreach (explode(',', $value) as $val)
			$criteria->orWhere(Criteria::expr()->contains('pt.name', trim($val)));

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridQuantity(QueryBuilder $qb, $value)
	{
		$sign = null;

		if (is_numeric($value))
			$sign = $value == 1 ? '>' : '<=';

		$arr = [];

		if (!in_array('ps', $qb->getAllAliases()))
			$qb->leftJoin('p.suppliers', 'ps');

		if ($sign) { // Ano/ne
			$arr[] = '(p.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR p.quantity IS NULL)');
			$arr[] = '(ps.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR ps.quantity IS NULL)');

			$where = implode($value == 1 ? ' OR ' : ' AND ', $arr);

			if (Config::load('pseudoWarehouse')) {
				$where .= ' OR p.unlimitedQuantity = 1';
			}

			$qb->andWhere($where);
		} else { // Od do
			if ($value->from)
				$qb->andWhere('(ifnull(p.quantity, 0) + ifnull(ps.quantity, 0) >= :quantityFrom)')->setParameter('quantityFrom', $value->from);

			if ($value->to)
				$qb->andWhere('(ifnull(p.quantity, 0) + ifnull(ps.quantity, 0) <= :quantityTo)')->setParameter('quantityTo', $value->to);
		}
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridManufacturer(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if (!in_array('manu', $qb->getAllAliases()))
			$qb->innerJoin('p.manufacturer', 'manu');

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('manu'));
		} else {
			$criteria->orWhere(Criteria::expr()->eq('manu.id', $value));
		}

		$qb->addCriteria($criteria);
	}

	public function conditionGridVariantType(QueryBuilder $qb, string $value)
	{
		if ($value === 'main')
			$qb->innerJoin('p.isVariant', 'variantType', Join::WITH, 'variantType.isDefault = 1');
		else if ($value === 'variant')
			$qb->innerJoin('p.isVariant', 'variantType', Join::WITH, 'variantType.isDefault = 0');
		else if ($value === 'withoutVariant')
			$qb->leftJoin('p.isVariant', 'variantType')
				->andWhere('variantType IS NULL');
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridCategory(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if (!in_array('sites', $qb->getAllAliases()))
			$qb->leftJoin('p.sites', 'sites');

		$qb->leftJoin('sites.category', 'sitesCategory')
			->leftJoin('sitesCategory.categoryTexts', 'sitesCategoryT');

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('sites'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('sitesCategoryT.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditCategories(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:editCategories', implode('-', $ids));
	}

	public function gridEditSpedition(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:editSpeditions', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditFeatures(array $ids)
	{
		$this->getPresenter()->redirect('Products:editFeatures', implode('-', $ids));
	}

	/**
	 * @param array  $ids
	 * @param string $manufacturer
	 */
	public function gridEditManufacturer(array $ids, $manufacturer)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setManufacturer($ids, $manufacturer)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.manufacturerChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.manufacturerChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array  $ids
	 * @param string $vatRate
	 */
	public function gridChangeVatRate(array $ids, $vatRate)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setVatRate($ids, $vatRate)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.vatRateChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.vatRateChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridChangePublished(array $ids)
	{
		$presenter = $this->getPresenter();

		foreach ($ids as $id) {
			if (!$this->productServices->setPublish($id, 1)) {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
				break;
			}
		}

		$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridChangeUnpublished(array $ids)
	{
		$presenter = $this->getPresenter();


		foreach ($ids as $id) {
			if (!$this->productServices->setPublish($id, 0)) {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
				break;
			}
		}

		$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array $ids
	 * @param int   $v
	 */
	public function gridDiscountDisabled(array $ids, $v)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setDiscountDisabled($ids, $v))
			$presenter->flashMessageSuccess('default.saved');
		else
			$presenter->flashMessageDanger('default.error');

		$presenter->redrawControl('flashes');
	}

	/**
	 * @param array $ids
	 * @param int   $v
	 */
	public function gridTagFastAdd(array $ids, $v)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setTag($ids, (int) $v))
			$presenter->flashMessageSuccess('default.saved');
		else
			$presenter->flashMessageDanger('default.error');

		$presenter->redrawControl('flashes');
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridSetTags(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:setTags', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridAssignRelatedProducts(array $ids)
	{
		$this->getPresenter()->redirect('Products:assignRelatedProducts', implode('-', $ids));
	}

	public function gridSetAsVariantsFor(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:setAsVariantsFor', implode('-', $ids));
	}

	public function gridSetAvailabilityAfterSoldOut(array $ids, $value)
	{
		$presenter = $this->getPresenter();

		if ($this->productServices->setAvailabilityAfterSoldOut($ids, $value)) {
			$presenter->flashMessageSuccess('default.saved');
		} else {
			$presenter->flashMessageDanger('default.error');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array $ids
	 */
	public function gridDeleteProducts(array $ids): void
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->remove($ids)) {
			$presenter->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else
			$presenter->flashMessageDanger('default.removeFailed');

		$presenter->redrawControl('flashes');
	}
}
