<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Caching\Cache;

/**
 * Class FeatureProducts
 * @package EshopCatalog\FrontModule\Model
 */
class FeatureProducts extends BaseFrontEntityService
{
	protected $entityClass = FeatureProduct::class;

	protected Features $features;

	protected array $cProductFeatures = [];

	protected array $cFeatureValuesIds = [];

	protected array $cacheDep = [
		Cache::TAGS   => ['featureProduct'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(Features $features)
	{
		$this->features = $features;
	}

	public function getFeatureValuesIdsForProducts(array $ids): array
	{
		$whereIds = [];
		$result   = [];

		foreach ($ids as $id) {
			if (isset($this->cFeatureValuesIds[$id])) {
				$result[$id] = $this->cFeatureValuesIds[$id];
			} else {
				$whereIds[]                   = $id;
				$this->cFeatureValuesIds[$id] = [];
				$result[$id]                  = [];
			}
		}

		if (!empty($whereIds)) {
			foreach ($this->getEr()->createQueryBuilder('fp')
				         ->select('IDENTITY(fp.product) as product, IDENTITY(fp.featureValue) as value')
				         ->andWhere('fp.product IN (:id)')->setParameter('id', $ids)
				         ->getQuery()->getScalarResult() as $row) {
				$this->cFeatureValuesIds[$row['product']][] = $row['value'];
				$result[$row['product']][]                  = $row['value'];
			}
		}

		return $result;
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @return Dao\FeatureProduct[][]
	 */
	public function getFeaturesForProduct(array $products): array
	{
		/** @var Dao\Product[] $whereIds */
		$whereIds = [];
		$result   = [];

		foreach ($products as $product) {
			$id = $product->getId();

			if (isset($this->cProductFeatures[$id])) {
				$result[$id] = $this->cProductFeatures[$id];
			} else {
				$whereIds[] = $product;
			}
		}

		if (!empty($whereIds)) {
			$data = [];
			foreach ($whereIds as $product) {
				foreach ($product->featureValuesIds as $valueId) {
					$value = $this->features->getFeatureValueById($valueId);
					if (!$value)
						continue;
					$feature = $this->features->getFeatureById($value['featureId']);
					if (!$feature)
						continue;

					$data[$product->getId()][$valueId] = $this->fillDao($product->getId(), $feature, $value);
				}
			}

			foreach ($data as $productId => $features) {
				uasort($features, fn($a, $b) => $a->position <=> $b->position);
				$result[$productId]                 = $features;
				$this->cProductFeatures[$productId] = $features;
			}
		}

		return $result;
	}

	protected function fillDao(int $productId, array $feature, array $value): Dao\FeatureProduct
	{
		$fp                 = new Dao\FeatureProduct();
		$fp->idProduct      = $productId;
		$fp->idFeature      = $feature['id'];
		$fp->position       = $feature['position'];
		$fp->name           = $feature['name'];
		$fp->idFeatureValue = $value['id'];
		$fp->valuePosition  = $value['position'];
		$fp->value          = $value['name'];

		return $fp;
	}
}
