<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Countries;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\DaoHelper;
use EshopCatalog\Model\Entities\Seller;
use EshopCatalog\FrontModule\Model\Dao;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\SellerBankAccount;
use Nette\Caching\Cache;

class Sellers extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'sellers';

	protected $entityClass = Seller::class;

	/** @var array */
	protected $cSellersForSite = [];

	/** @var Countries */
	protected Countries $countries;

	public function __construct(Countries $countries)
	{
		$this->countries = $countries;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function getSellerForSite(string $siteIdent): ?Dao\Seller
	{
		if (!isset($this->cSellersForSite[$siteIdent])) {
			$this->cSellersForSite[$siteIdent] = $this->getCache()->load('seller/' . $siteIdent, function(&$dep) use ($siteIdent) {
				$dep    = [
					Cache::EXPIRATION => '1 month',
					Cache::TAGS       => [self::CACHE_NAMESPACE],
				];
				$result = null;

				$seller = $this->getEr()->createQueryBuilder('s')
						->innerJoin('s.sites', 'sites', Join::WITH, 'sites.site = :site')
						->setParameter('site', $siteIdent)
						->getQuery()
						->setMaxResults(1)->getArrayResult()[0] ?? null;

				if ($seller) {
					$seller['bankAccounts'] = $this->em->createQueryBuilder()->select('ba')
						->from(SellerBankAccount::class, 'ba')
						->where('ba.seller = :seller')
						->setParameter('seller', $seller['id'])
						->getQuery()
						->getArrayResult();

					$result = $this->fillDao($seller);
				}

				return $result;
			});
		}

		return $this->cSellersForSite[$siteIdent] ?: null;
	}

	protected function fillDao(array $data): Dao\Seller
	{
		$dao       = new Dao\Seller((int) $data['id']);
		$countries = $this->countries->getDao();

		/** @var Dao\Seller $dao */
		$dao = DaoHelper::fillDaoFromArray($data, $dao, ['country', 'country2']);
		if ($data['address'])
			$dao->street = $data['address'];
		if ($data['address2'])
			$dao->street2 = $data['address2'];
		$dao->sendOrderToSeller   = (bool) $data['sendOrder'];
		$dao->sendStatusToSeller  = (bool) $data['sendStatus'];
		$dao->sendSellerEmailType = $data['sendEmailType'];

		$dao->country  = $countries[$data['country']] ?? null;
		$dao->country2 = $countries[$data['country2']];

		foreach ($data['bankAccounts'] ?? [] as $row) {
			$bankAccount = new Dao\BankAccount();
			$bankAccount = DaoHelper::fillDaoFromArray($row, $bankAccount);
			$dao->addBankAccount($bankAccount);
		}

		return $dao;
	}
}
